%INITIALIZATION class definition
%
%   Usage:
%      initialization=initialization();

classdef initialization
	properties (SetAccess=public) 
		vx            = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1,'enum',VxEnum);
		vy            = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1,'enum',VyEnum);
		vz            = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1,'enum',VzEnum);
		vel           = modelfield('default',NaN,'marshall',false);
		pressure      = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1,'enum',PressureEnum);
		temperature   = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1,'enum',TemperatureEnum);
		watercolumn   = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1,'enum',WatercolumnEnum);
		waterfraction = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1,'enum',WaterfractionEnum);
	end
	methods
		function obj = initialization(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				case 1
					in=varargin{1};
					if (isa(in,'numeric') & in==0),
						% requesting templates do nothing
					else
						error('constructor not supported');
					end 
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%first, use the defaults provided by the properties definition above. 
			fieldnames=fields(obj);
			for i=1:length(fieldnames),
				fieldname=fieldnames{i};
				obj.(fieldname)=obj.(fieldname).default;
			end

		end % }}}
		function checkconsistency(obj,md,solution,analyses) % {{{
			if ismember(DiagnosticHorizAnalysisEnum,analyses)
				if ~isnan(md.initialization.vx) & ~isnan(md.initialization.vy),
					checkfield(md,'initialization.vx','NaN',1,'size',[md.mesh.numberofvertices 1]);
					checkfield(md,'initialization.vy','NaN',1,'size',[md.mesh.numberofvertices 1]);
				end
			end
			if ismember(PrognosticAnalysisEnum,analyses),
				checkfield(md,'initialization.vx','NaN',1,'size',[md.mesh.numberofvertices 1]);
				checkfield(md,'initialization.vy','NaN',1,'size',[md.mesh.numberofvertices 1]);
			end
			if ismember(HydrologyAnalysisEnum,analyses),
				checkfield(md,'initialization.watercolumn','NaN',1,'size',[md.mesh.numberofvertices 1]);
			end
			if ismember(BalancethicknessAnalysisEnum,analyses),
				checkfield(md,'initialization.vx','NaN',1,'size',[md.mesh.numberofvertices 1]);
				checkfield(md,'initialization.vy','NaN',1,'size',[md.mesh.numberofvertices 1]);
				%Triangle with zero velocity
				if any(sum(abs(md.initialization.vx(md.mesh.elements)),2)==0 & sum(abs(md.initialization.vy(md.mesh.elements)),2)==0)
					checkmessage('model not consistent: at least one triangle has all its vertices with a zero velocity');
				end
			end
			if ismember(ThermalAnalysisEnum,analyses),
				checkfield(md,'initialization.vx','NaN',1,'size',[md.mesh.numberofvertices 1]);
				checkfield(md,'initialization.vy','NaN',1,'size',[md.mesh.numberofvertices 1]);
				checkfield(md,'initialization.vz','NaN',1,'size',[md.mesh.numberofvertices 1]);
				checkfield(md,'initialization.pressure','NaN',1,'size',[md.mesh.numberofvertices 1]);
			end
		end % }}}
		function disp(obj) % {{{
			disp(sprintf('   initial field values:'));

			fielddisplay(obj,'vx','x component of velocity');
			fielddisplay(obj,'vy','y component of velocity');
			fielddisplay(obj,'vz','z component of velocity');
			fielddisplay(obj,'vel','velocity norm');
			fielddisplay(obj,'pressure','pressure field');
			fielddisplay(obj,'temperature','temperature in Kelvins');
			fielddisplay(obj,'watercolumn','thickness of subglacial water');
			fielddisplay(obj,'waterfraction','fraction of water in the ice');

		end % }}}
	end
end
