%DIAGNOSTIC class definition
%
%   Usage:
%      diagnostic=diagnostic();

classdef diagnostic
	properties (SetAccess=public) 
		spcvx                    = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		spcvy                    = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		spcvz                    = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		restol                   = modelfield('default',0,'marshall',true,'format','Double');
		reltol                   = modelfield('default',0,'marshall',true,'format','Double');
		abstol                   = modelfield('default',0,'marshall',true,'format','Double');
		stokesreconditioning     = modelfield('default',0,'marshall',true,'format','Double');
		viscosity_overshoot      = modelfield('default',0,'marshall',true,'format','Double');
		icefront                 = modelfield('default',NaN,'marshall',true,'preprocess','marshallicefront','format','DoubleMat','mattype',3);
		maxiter                  = modelfield('default',0,'marshall',true,'format','Integer');
		shelf_dampening          = modelfield('default',0,'marshall',true,'format','Integer');
		vertex_pairing           = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',3);
		penalty_factor           = modelfield('default',0,'marshall',true,'format','Double');
		rift_penalty_lock        = modelfield('default',0,'marshall',true,'format','Integer');
		rift_penalty_threshold   = modelfield('default',0,'marshall',true,'format','Integer');
		referential              = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		requested_outputs        = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',3);
	end
	methods
		function obj = diagnostic(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				case 1
					in=varargin{1};
					if (isa(in,'numeric') & in==0),
						% requesting templates do nothing
					else
						error('constructor not supported');
					end 
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%first, use the defaults provided by the properties definition above. 
			fieldnames=fields(obj);
			for i=1:length(fieldnames),
				fieldname=fieldnames{i};
				obj.(fieldname)=obj.(fieldname).default;
			end

			 %maximum of non-linear iterations.
			 obj.maxiter=100;
			 
			 %Convergence criterion: absolute, relative and residual
			 obj.restol=10^-4; 
			 obj.reltol=0.01;
			 obj.abstol=10;
			 
			 obj.stokesreconditioning=10^13;
			 obj.shelf_dampening=0;

			 %Penalty factor applied kappa=max(stiffness matrix)*10^penalty_factor
			 obj.penalty_factor=3;

			 %coefficient to update the viscosity between each iteration of
			 %a diagnostic according to the following formula
			 %viscosity(n)=viscosity(n)+viscosity_overshoot(viscosity(n)-viscosity(n-1))
			 obj.viscosity_overshoot=0;

			 %Stop the iterations of rift if below a threshold
			 obj.rift_penalty_threshold=0;

			 %in some solutions, it might be needed to stop a run when only
			 %a few constraints remain unstable. For thermal computation, this
			 %parameter is often used.
			 obj.rift_penalty_lock=10;

		end % }}}
		function checkconsistency(obj,md,solution,analyses) % {{{

			fields={'restol','maxiter'};
			checknan(md,'diagnostic',fields);

			fields={'abstol','restol','reltol'};
			checkgreaterstrict(md,'diagnostic',fields,0);

			fields={'icefront'};
			if (md.mesh.dimension==2),
				checksize(md,'diagnostic',fields,[NaN 4]);
			elseif md.mesh.dimension==3,
				checksize(md,'diagnostic',fields,[NaN 6]);
			end
			fields={'icefront(:,end)'};
			checkvalues(md,'diagnostic',fields,[0 1 2]);

			fields={'referential'};
			checksize(md,'diagnostic',fields,[md.mesh.numberofvertices 6]);

			if ~isempty(md.diagnostic.requested_outputs),
				if(size(md.diagnostic.requested_outputs,2)~=1),
					checkmessage(['model ' md.miscellaneous.name ' requested outputs should be a column vector']);
				end
			end

		end % }}}
		function disp(obj) % {{{

			disp(sprintf('   Diagnostic solution parameters:'));

			disp(sprintf('\n      %s','Convergence criteria:'));
			fielddisplay(obj,'restol','mechanical equilibrium residue convergence criterion');
			fielddisplay(obj,'reltol','velocity relative convergence criterion, NaN -> not applied');
			fielddisplay(obj,'abstol','velocity absolute convergence criterion, NaN -> not applied');
			fielddisplay(obj,'maxiter','maximum number of nonlinear iterations');
			fielddisplay(obj,'viscosity_overshoot','over-shooting constant new=new+C*(new-old)');

			disp(sprintf('\n      %s','boundary conditions:'));
			fielddisplay(obj,'spcvx','x-axis velocity constraint (NaN means no constraint)');
			fielddisplay(obj,'spcvy','y-axis velocity constraint (NaN means no constraint)');
			fielddisplay(obj,'spcvz','z-axis velocity constraint (NaN means no constraint)');
			fielddisplay(obj,'icefront','segments on ice front list');

			disp(sprintf('\n      %s','Rift options:'));
			fielddisplay(obj,'rift_penalty_threshold','threshold for instability of mechanical constraints');
			fielddisplay(obj,'rift_penalty_lock','number of iterations before rift penalties are locked');

			disp(sprintf('\n      %s','Penalty options:'));
			fielddisplay(obj,'penalty_factor','offset used by penalties: penalty = Kmax*10^offset');
			fielddisplay(obj,'vertex_pairing','pairs of vertices that are penalized');

			disp(sprintf('\n      %s','Other:'));
			fielddisplay(obj,'shelf_dampening','use dampening for floating ice ? Only for Stokes model');
			fielddisplay(obj,'referential','local referential');
			fielddisplay(obj,'requested_outputs','additional outputs requested');

		end % }}}
	end
end
