%CONSTANTS class definition
%
%   Usage:
%      inversion=inversion();

classdef inversion
	properties (SetAccess=public) 
		iscontrol                   = modelfield('default',0,'marshall',true,'format','Boolean');
		control_parameters          = modelfield('default',NaN,'marshall',true,'preprocess','marshallcontroltype','format','DoubleMat','mattype',3);
		cost_functions_coefficients = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		nsteps                      = modelfield('default',0,'marshall',true,'format','Integer');
		maxiter_per_step            = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',3);
		cost_functions              = modelfield('default',NaN,'marshall',true,'preprocess','marshallcmresponses','format','DoubleMat','mattype',3);
		gradient_scaling            = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',3);
		cost_function_threshold     = modelfield('default',0,'marshall',true,'format','Double');
		min_parameters              = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',3);
		max_parameters              = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',3);
		step_threshold              = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',3);
		gradient_only               = modelfield('default',0,'marshall',true,'format','Boolean');
		num_control_parameters      = modelfield('default',0,'marshall',true,'format','Integer');
		num_cost_functions          = modelfield('default',0,'marshall',true,'format','Integer');
		vx_obs                      = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		vy_obs                      = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		vz_obs                      = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		vel_obs                     = modelfield('default',NaN,'marshall',false);
		thickness_obs               = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
	end
	methods
		function obj = inversion(varargin) % {{{
			switch nargin
				case 0
					obj=setdefaultparameters(obj);
				case 1
					in=varargin{1};
					if (isa(in,'numeric') & in==0),
						% requesting templates do nothing
					else
						error('constructor not supported');
					end 
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function obj = setdefaultparameters(obj) % {{{

			%first, use the defaults provided by the properties definition above. 
			fieldnames=fields(obj);
			for i=1:length(fieldnames),
				fieldname=fieldnames{i};
				obj.(fieldname)=obj.(fieldname).default;
			end

			 %parameter to be inferred by control methods (only
			 %drag and B are supported yet)
			 obj.control_parameters={'FrictionCoefficient'};

			 %number of steps in the control methods
			 obj.nsteps=20;

			 %maximum number of iteration in the optimization algorithm for
			 %each step
			 obj.maxiter_per_step=20*ones(obj.nsteps,1);

			 %the inversed parameter is updated as follows:
			 %new_par=old_par + gradient_scaling(n)*C*gradient with C in [0 1];
			 %usually the gradient_scaling must be of the order of magnitude of the 
			 %inversed parameter (10^8 for B, 50 for drag) and can be decreased
			 %after the first iterations
			 obj.gradient_scaling=50*ones(obj.nsteps,1);

			 %several responses can be used:
			 obj.cost_functions=101*ones(obj.nsteps,1);

			 %step_threshold is used to speed up control method. When
			 %misfit(1)/misfit(0) < obj.step_threshold, we go directly to
			 %the next step
			 obj.step_threshold=.7*ones(obj.nsteps,1); %30 per cent decrement.

			 %stop control solution at the gradient computation and return it? 
			 obj.gradient_only=0;

			 %cost_function_threshold is a criteria to stop the control methods.
			 %if J[n]-J[n-1]/J[n] < criteria, the control run stops
			 %NaN if not applied
			 obj.cost_function_threshold=NaN; %not activated

		end % }}}
		function checkconsistency(obj,md) % {{{

			%Early return
			if ~obj.iscontrol, return; end

			%Check NaN
			fields={'nsteps','maxiter_per_step'};
			checknan(md,'inversion',fields);

			%CONTROL TYPE
			num_controls=numel(md.inversion.control_parameters);
			num_costfunc=size(md.inversion.cost_functions,2);
			if ~iscell(md.inversion.control_parameters)
				checkmessage(['model not consistent: model ' md.miscellaneous.name ' inversion.control_parameters field should be a cell of strings']);
			end
			if ~ismember(md.inversion.control_parameters,{'BalancethicknessThickeningRate' 'FrictionCoefficient' 'MaterialsRheologyBbar' 'Vx' 'Vy'});
				checkmessage(['model not consistent: model ' md.miscellaneous.name ' inversion.control_parameters field should be ''BalancethicknessThickeningRate'' ''FrictionCoefficient'' ''MaterialsRheologyBbar'' ''Vx'' ''Vy''']);
			end

			%LENGTH CONTROL FIELDS
			fields={'maxiter_per_step','step_threshold'};
			checksize(md,'inversion',fields,[md.inversion.nsteps 1]);
			fields={'cost_functions'};
			checksize(md,'inversion',fields,[md.inversion.nsteps num_costfunc]);
			fields={'gradient_scaling'};
			checksize(md,'inversion',fields,[md.inversion.nsteps num_controls]);
			fields={'min_parameters','max_parameters'};
			checksize(md,'inversion',fields,[md.mesh.numberofvertices num_controls]);

			%RESPONSES
			checkvalues(md,'inversion',{'cost_functions'},[101:105 201 501:503]);

			%WEIGHTS
			fields={'cost_functions_coefficients'};
			checksize(md,'inversion',fields,[md.mesh.numberofvertices num_costfunc]);
			checkgreater(md,'inversion',fields,0);

			%OBSERVED VELOCITIES
			if md.private.solution==BalancethicknessSolutionEnum
				fields={'thickness_obs'};
				checksize(md,'inversion',fields,[md.mesh.numberofvertices 1]);
				checknan(md,'inversion',fields);
			else
				fields={'vx_obs','vy_obs'};
				checksize(md,'inversion',fields,[md.mesh.numberofvertices 1]);
				checknan(md,'inversion',fields);
			end

			%DIRICHLET IF THICKNESS <= 0
			if any(md.geometry.thickness<=0),
				pos=find(md.geometry.thickness<=0);
				if any(isnan(md.balancethickness.spcthickness(pos))),
					checkmessage(['model not consistent: model ' md.miscellaneous.name ' has some nodes with 0 thickness']);
				end
			end
		end % }}}
		function disp(obj) % {{{
			fielddisplay(obj,'iscontrol','is inversion activated?');
			fielddisplay(obj,'control_parameters','parameter where inverse control is carried out; ex: {''FrictionCoefficient''}, or {''RheologyBbar''}');
			fielddisplay(obj,'cost_functions_coefficients','cost_functions_coefficients applied to the misfit of each node');
			fielddisplay(obj,'nsteps','number of optimization searches');
			fielddisplay(obj,'cost_function_threshold','misfit convergence criterion. Default is 1%, NaN if not applied');
			fielddisplay(obj,'gradient_scaling','scaling factor on gradient direction during optimization, for each optimization step');
			fielddisplay(obj,'cost_functions','indicate the type of response for each optimization steps');
			fielddisplay(obj,'maxiter_per_step','maximum iterations during each optimization step');
			fielddisplay(obj,'step_threshold','decrease threshold for misfit, default is 30%');
			fielddisplay(obj,'min_parameters','absolute minimum acceptable value of the inversed parameter');
			fielddisplay(obj,'max_parameters','absolute maximum acceptable value of the inversed parameter');
			fielddisplay(obj,'gradient_only','stop control method solution at gradient');
			fielddisplay(obj,'vx_obs','observed velocity x component [m/a]');
			fielddisplay(obj,'vy_obs','observed velocity y component [m/a]');
			fielddisplay(obj,'vel_obs','observed velocity magnitude [m/a]');
			disp('Available responses:');
			disp('   101: SurfaceAbsVelMisfit');
			disp('   102: SurfaceRelVelMisfit');
			disp('   103: SurfaceLogVelMisfit');
			disp('   104: SurfaceLogVxVyMisfit');
			disp('   105: SurfaceAverageVelMisfit');
			disp('   201: ThicknessAbsMisfit');
			disp('   501: DragCoefficientAbsGradient');
			disp('   502: RheologyBbarAbsGradient');
			disp('   503: ThicknessAbsGradient');
		end % }}}
	end
end
