/*!\file Pengrid.c
 * \brief: implementation of the Pengrid object
 */

/*Headers*/
/*{{{1*/
#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <stdio.h>
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../include/include.h"
#include "../../shared/shared.h"
#include "../../Container/Container.h"
/*}}}*/
	
/*Element macros*/
#define NUMVERTICES   1

/*Pengrid constructors and destructor*/
/*FUNCTION Pengrid::Pengrid(){{{1*/
Pengrid::Pengrid(){
	this->inputs=NULL;
	this->parameters=NULL;
	this->hnode=NULL;
	this->node=NULL;
	this->helement=NULL;
	this->element=NULL;
	this->hmatpar=NULL;
	this->matpar=NULL;
	
	/*not active, not zigzagging: */
	active=0;
	zigzag_counter=0;

}
/*}}}1*/
/*FUNCTION Pengrid::Pengrid(int index, int id, IoModel* iomodel,int analysis_type){{{1*/
Pengrid::Pengrid(int id, int index, IoModel* iomodel, int in_analysis_type){ //i is the element index

	int i,j;
	int pengrid_node_id;
	int pengrid_matpar_id;
	int pengrid_element_id;

	int numberofvertices;
	int numberofelements;

	/*Fetch parameters: */
	iomodel->Constant(&numberofvertices,MeshNumberofverticesEnum);
	iomodel->Constant(&numberofelements,MeshNumberofelementsEnum);

	/*Some checks if debugging activated*/
	_assert_(iomodel->singlenodetoelementconnectivity);
	_assert_(index>=0 && index<numberofvertices);
	_assert_(id);

	/*id: */
	this->id=id;
	this->analysis_type=in_analysis_type;
	
	/*hooks: */
	pengrid_node_id=iomodel->nodecounter+index+1;
	pengrid_element_id=iomodel->singlenodetoelementconnectivity[index];
	_assert_(pengrid_element_id);
	pengrid_matpar_id=numberofelements+1; //refers to the constant material parameters object

	this->hnode=new Hook(&pengrid_node_id,1);
	this->helement=new Hook(&pengrid_element_id,1);
	this->hmatpar=new Hook(&pengrid_matpar_id,1);

	//initialize inputs: none needed
	this->inputs=new Inputs();

	//this->parameters: we still can't point to it, it may not even exist. Configure will handle this.
	this->parameters=NULL;
	this->node=NULL;
	this->element=NULL;
	this->matpar=NULL;

	//let's not forget internals
	this->active=0;
	this->zigzag_counter=0;

}
/*}}}*/
/*FUNCTION Pengrid::~Pengrid(){{{1*/
Pengrid::~Pengrid(){
	delete inputs;
	delete hnode;
	delete helement;
	delete hmatpar;
	return;
}
/*}}}1*/
			
/*Object virtual functions definitions:*/
/*FUNCTION Pengrid::Echo {{{1*/
void Pengrid::Echo(void){
	this->DeepEcho();
}
/*}}}1*/
/*FUNCTION Pengrid::DeepEcho{{{1*/
void Pengrid::DeepEcho(void){

	printf("Pengrid:\n");
	printf("   id: %i\n",id);
	printf("   analysis_type: %s\n",EnumToStringx(analysis_type));
	hnode->DeepEcho();
	helement->DeepEcho();
	hmatpar->DeepEcho();
	printf("   active %i\n",this->active);
	printf("   zigzag_counter %i\n",this->zigzag_counter);
	printf("   parameters\n");
	parameters->DeepEcho();
	printf("   inputs\n");
	inputs->DeepEcho();
}
/*}}}*/
/*FUNCTION Pengrid::Id {{{1*/
int    Pengrid::Id(void){ return id; }
/*}}}1*/
/*FUNCTION Pengrid::MyRank {{{1*/
int    Pengrid::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}1*/
#ifdef _SERIAL_
/*FUNCTION Pengrid::Marshall {{{1*/
void  Pengrid::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;
	char* marshalled_inputs=NULL;
	int   marshalled_inputs_size;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Tria: */
	enum_type=PengridEnum;

	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);

	/*marshall Tria data: */
	memcpy(marshalled_dataset,&id,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(marshalled_dataset,&analysis_type,sizeof(analysis_type));marshalled_dataset+=sizeof(analysis_type);
	memcpy(marshalled_dataset,&active,sizeof(active));marshalled_dataset+=sizeof(active);
	memcpy(marshalled_dataset,&zigzag_counter,sizeof(zigzag_counter));marshalled_dataset+=sizeof(zigzag_counter);

	/*Marshall hooks: */
	hnode->Marshall(&marshalled_dataset);
	helement->Marshall(&marshalled_dataset);
	hmatpar->Marshall(&marshalled_dataset);

	/*Marshall inputs: */
	marshalled_inputs_size=inputs->MarshallSize();
	marshalled_inputs=inputs->Marshall();
	memcpy(marshalled_dataset,marshalled_inputs,marshalled_inputs_size*sizeof(char));
	marshalled_dataset+=marshalled_inputs_size;

	/*parameters: don't do anything about it. parameters are marshalled somewhere else!*/

	xfree((void**)&marshalled_inputs);

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
/*FUNCTION Pengrid::MarshallSize {{{1*/
int   Pengrid::MarshallSize(){
	
	return sizeof(id)
		+sizeof(analysis_type)
		+sizeof(active)
		+sizeof(zigzag_counter)
		+hnode->MarshallSize()
		+helement->MarshallSize()
		+hmatpar->MarshallSize()
		+inputs->MarshallSize()
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}*/
/*FUNCTION Pengrid::Demarshall {{{1*/
void  Pengrid::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/

	memcpy(&id,marshalled_dataset,sizeof(id));marshalled_dataset+=sizeof(id);
	memcpy(&analysis_type,marshalled_dataset,sizeof(analysis_type));marshalled_dataset+=sizeof(analysis_type);
	memcpy(&active,marshalled_dataset,sizeof(active));marshalled_dataset+=sizeof(active);
	memcpy(&zigzag_counter,marshalled_dataset,sizeof(zigzag_counter));marshalled_dataset+=sizeof(zigzag_counter);

	/*demarshall hooks: */
	hnode=new Hook(); hnode->Demarshall(&marshalled_dataset);
	helement=new Hook(); helement->Demarshall(&marshalled_dataset);
	hmatpar=new Hook(); hmatpar->Demarshall(&marshalled_dataset);
	
	/*demarshall inputs: */
	inputs=(Inputs*)DataSetDemarshallRaw(&marshalled_dataset); 

	/*parameters: may not exist even yet, so let Configure handle it: */
	this->parameters=NULL;
	this->node=NULL;
	this->element=NULL;
	this->matpar=NULL;

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}*/
#endif
/*FUNCTION Pengrid::Enum {{{1*/
int Pengrid::Enum(void){

	return PengridEnum;
}
/*}}}1*/
/*FUNCTION Icefront::copy {{{1*/
Object* Pengrid::copy() {
	
	Pengrid* pengrid=NULL;

	pengrid=new Pengrid();

	/*copy fields: */
	pengrid->id=this->id;
	pengrid->analysis_type=this->analysis_type;
	if(this->inputs){
		pengrid->inputs=(Inputs*)this->inputs->Copy();
	}
	else{
		pengrid->inputs=new Inputs();
	}
	/*point parameters: */
	pengrid->parameters=this->parameters;

	/*now deal with hooks and objects: */
	pengrid->hnode=(Hook*)this->hnode->copy();
	pengrid->hmatpar=(Hook*)this->hmatpar->copy();
	pengrid->helement=(Hook*)this->helement->copy();

	/*corresponding fields*/
	pengrid->node  =(Node*)pengrid->hnode->delivers();
	pengrid->matpar =(Matpar*)pengrid->hmatpar->delivers();
	pengrid->element=(Element*)pengrid->helement->delivers();

	//let's not forget internals
	pengrid->active=this->active=0;
	pengrid->zigzag_counter=this->zigzag_counter=0;

	return pengrid;

}
/*}}}*/

/*Load virtual functions definitions:*/
/*FUNCTION Pengrid::Configure {{{1*/
void  Pengrid::Configure(Elements* elementsin,Loads* loadsin,Nodes* nodesin,Vertices* verticesin,Materials* materialsin,Parameters* parametersin){

	/*Take care of hooking up all objects for this load, ie links the objects in the hooks to their respective 
	 * datasets, using internal ids and offsets hidden in hooks: */
	hnode->configure(nodesin);
	helement->configure(elementsin);
	hmatpar->configure(materialsin);

	/*Get corresponding fields*/
	node=(Node*)hnode->delivers();
	element=(Element*)helement->delivers();
	matpar=(Matpar*)hmatpar->delivers();

	/*point parameters to real dataset: */
	this->parameters=parametersin;
}
/*}}}1*/
/*FUNCTION Pengrid::SetCurrentConfiguration {{{1*/
void  Pengrid::SetCurrentConfiguration(Elements* elementsin,Loads* loadsin,Nodes* nodesin,Vertices* verticesin,Materials* materialsin,Parameters* parametersin){

}
/*}}}1*/
/*FUNCTION Pengrid::CreateKMatrix {{{1*/
void  Pengrid::CreateKMatrix(Mat Kff, Mat Kfs){

	/*No loads applied, do nothing: */
	return;

}
/*}}}1*/
/*FUNCTION Pengrid::CreatePVector {{{1*/
void  Pengrid::CreatePVector(Vec pf){

	/*No loads applied, do nothing: */
	return;

}
/*}}}1*/
/*FUNCTION Pengrid::PenaltyCreateMatrix {{{1*/
void  Pengrid::PenaltyCreateKMatrix(Mat Kff, Mat Kfs,double kmax){

	/*Retrieve parameters: */
	ElementMatrix* Ke=NULL;
	int analysis_type;
	this->parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	switch(analysis_type){
		case DiagnosticHorizAnalysisEnum: case AdjointHorizAnalysisEnum:
			Ke=PenaltyCreateKMatrixDiagnosticStokes(kmax);
			break;
		case ThermalAnalysisEnum:
			Ke=PenaltyCreateKMatrixThermal(kmax);
			break;
		case MeltingAnalysisEnum:
			Ke=PenaltyCreateKMatrixMelting(kmax);
			break;
		default:
			_error_("analysis %i (%s) not supported yet",analysis_type,EnumToStringx(analysis_type));
	}

	/*Add to global matrix*/
	if(Ke){
		Ke->AddToGlobal(Kff,Kfs);
		delete Ke;
	}
}
/*}}}1*/
/*FUNCTION Pengrid::PenaltyCreatePVector {{{1*/
void  Pengrid::PenaltyCreatePVector(Vec pf,double kmax){

	/*Retrieve parameters: */
	ElementVector* pe=NULL;
	int analysis_type;
	this->parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	switch(analysis_type){
		case ThermalAnalysisEnum:
			pe=PenaltyCreatePVectorThermal(kmax);
			break;
		case MeltingAnalysisEnum:
			pe=PenaltyCreatePVectorMelting(kmax);
			break;
		case DiagnosticHorizAnalysisEnum: case AdjointHorizAnalysisEnum:
			break;
		default:
			_error_("analysis %i (%s) not supported yet",analysis_type,EnumToStringx(analysis_type));
	}

	/*Add to global Vector*/
	if(pe){
		pe->AddToGlobal(pf);
		delete pe;
	}
}
/*}}}1*/
/*FUNCTION Pengrid::InAnalysis{{{1*/
bool Pengrid::InAnalysis(int in_analysis_type){
	if (in_analysis_type==this->analysis_type)return true;
	else return false;
}
/*}}}*/

/*Update virtual functions definitions:*/
/*FUNCTION Pengrid::InputUpdateFromVector(double* vector, int name, int type) {{{1*/
void  Pengrid::InputUpdateFromVector(double* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Pengrid::InputUpdateFromVector(int* vector, int name, int type) {{{1*/
void  Pengrid::InputUpdateFromVector(int* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Pengrid::InputUpdateFromVector(bool* vector, int name, int type) {{{1*/
void  Pengrid::InputUpdateFromVector(bool* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Pengrid::InputUpdateFromVectorDakota(double* vector, int name, int type) {{{1*/
void  Pengrid::InputUpdateFromVectorDakota(double* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Pengrid::InputUpdateFromVectorDakota(int* vector, int name, int type) {{{1*/
void  Pengrid::InputUpdateFromVectorDakota(int* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Pengrid::InputUpdateFromVectorDakota(bool* vector, int name, int type) {{{1*/
void  Pengrid::InputUpdateFromVectorDakota(bool* vector, int name, int type){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Pengrid::InputUpdateFromConstant(double constant, int name) {{{1*/
void  Pengrid::InputUpdateFromConstant(double constant, int name){
	switch(name){

		case MeltingOffsetEnum:
			inputs->AddInput(new DoubleInput(name,constant));
			return;

	}
}
/*}}}*/
/*FUNCTION Pengrid::InputUpdateFromConstant(int constant, int name) {{{1*/
void  Pengrid::InputUpdateFromConstant(int constant, int name){
	/*Nothing updated yet*/
}
/*}}}*/
/*FUNCTION Pengrid::InputUpdateFromConstant(bool constant, int name) {{{1*/
void  Pengrid::InputUpdateFromConstant(bool constant, int name){

	switch(name){

		case ResetPenaltiesEnum:
			if (constant) zigzag_counter=0;
			return;

	}
}
/*}}}*/
/*FUNCTION Pengrid::InputUpdateFromSolution{{{1*/
void  Pengrid::InputUpdateFromSolution(double* solution){
	/*Nothing updated yet*/
}
/*}}}*/		

/*Pengrid management:*/
/*FUNCTION Pengrid::ConstraintActivate {{{1*/
void  Pengrid::ConstraintActivate(int* punstable){

	int analysis_type;

	/*Retrieve parameters: */
	this->parameters->FindParam(&analysis_type,AnalysisTypeEnum);

	if (analysis_type==DiagnosticHorizAnalysisEnum){
		/*No penalty to check*/
		return;
	}
	else if (analysis_type==ThermalAnalysisEnum){
		ConstraintActivateThermal(punstable);
	}
	else if (analysis_type==MeltingAnalysisEnum){
		/*No penalty to check*/
		return;
	}
	else{
		_error_("analysis: %s not supported yet",EnumToStringx(analysis_type));
	}

}
/*}}}1*/
/*FUNCTION Pengrid::ConstraintActivateThermal {{{1*/
void  Pengrid::ConstraintActivateThermal(int* punstable){

	//   The penalty is stable if it doesn't change during to successive iterations.   

	int    found=0;
	const int numnodes=1;
	double pressure;
	double temperature;
	double t_pmp;
	int    new_active;
	int    unstable=0;
	int    reset_penalties=0;
	int    penalty_lock;

	/*recover pointers: */
	Penta* penta=(Penta*)element;
	
	/*check that pengrid is not a clone (penalty to be added only once)*/
	if (node->IsClone()){
		unstable=0;
		*punstable=unstable;
		return;
	}

	//First recover pressure and temperature values, using the element: */
	penta->GetParameterValue(&pressure,node,PressureEnum);
	penta->GetParameterValue(&temperature,node,TemperaturePicardEnum);

	//Recover our data:
	parameters->FindParam(&penalty_lock,ThermalPenaltyLockEnum);
	
	//Compute pressure melting point
	t_pmp=matpar->TMeltingPoint(pressure);

	//Figure out if temperature is over melting_point, in which case, this penalty needs to be activated.

	if (temperature>t_pmp){
		new_active=1;
	}
	else{
		new_active=0;
	}


	//Figure out stability of this penalty
	if (active==new_active){
		unstable=0;
	}
	else{
		unstable=1;
		if(penalty_lock)zigzag_counter++;
	}

	/*If penalty keeps zigzagging more than 5 times: */
	if(penalty_lock){
		if(zigzag_counter>penalty_lock){
			unstable=0;
			active=1;
		}
	}

	//Set penalty flag
	active=new_active;

	//*Assign output pointers:*/
	*punstable=unstable;
}
/*}}}1*/
/*FUNCTION Pengrid::PenaltyCreateKMatrixDiagnosticStokes {{{1*/
ElementMatrix* Pengrid::PenaltyCreateKMatrixDiagnosticStokes(double kmax){
	
	const int numdof = NUMVERTICES *NDOF4;
	double    slope[2];
	double    penalty_offset;
	int       approximation;

	Penta* penta=(Penta*)element;

	/*Initialize Element vector and return if necessary*/
	penta->inputs->GetParameterValue(&approximation,ApproximationEnum);
	if(approximation!=StokesApproximationEnum &&  approximation!=PattynStokesApproximationEnum) return NULL;
	ElementMatrix* Ke=new ElementMatrix(&node,1,this->parameters,StokesApproximationEnum);

	/*Retrieve all inputs and parameters*/
	parameters->FindParam(&penalty_offset,DiagnosticPenaltyFactorEnum);
	penta->GetParameterValue(&slope[0],node,BedSlopeXEnum);
	penta->GetParameterValue(&slope[1],node,BedSlopeYEnum);

	/*Create elementary matrix: add penalty to constrain wb (wb=ub*db/dx+vb*db/dy)*/
	Ke->values[2*NDOF4+0]=-slope[0]*kmax*pow((double)10.0,penalty_offset);
	Ke->values[2*NDOF4+1]=-slope[1]*kmax*pow((double)10.0,penalty_offset);
	Ke->values[2*NDOF4+2]= kmax*pow((double)10,penalty_offset);

	/*Clean up and return*/
	return Ke;
}
/*}}}1*/
/*FUNCTION Pengrid::PenaltyCreateKMatrixMelting {{{1*/
ElementMatrix* Pengrid::PenaltyCreateKMatrixMelting(double kmax){

	const int numdof=NUMVERTICES*NDOF1;
	double pressure,temperature,t_pmp;
	double penalty_factor;

	Penta* penta=(Penta*)element;

	/*check that pengrid is not a clone (penalty to be added only once)*/
	if (node->IsClone()) return NULL;
	ElementMatrix* Ke=new ElementMatrix(&node,1,this->parameters);

	/*Retrieve all inputs and parameters*/
	penta->GetParameterValue(&pressure,node,PressureEnum);
	penta->GetParameterValue(&temperature,node,TemperatureEnum);
	parameters->FindParam(&penalty_factor,ThermalPenaltyFactorEnum);
	
	/*Compute pressure melting point*/
	t_pmp=matpar->GetMeltingPoint()-matpar->GetBeta()*pressure;

	/*Add penalty load*/
	if (temperature<t_pmp){ //If T<Tpmp, there must be no melting. Therefore, melting should be  constrained to 0 when T<Tpmp, instead of using spcs, use penalties
		Ke->values[0]=kmax*pow((double)10,penalty_factor);
	}

	/*Clean up and return*/
	return Ke;
}
/*}}}1*/
/*FUNCTION Pengrid::PenaltyCreateKMatrixThermal {{{1*/
ElementMatrix* Pengrid::PenaltyCreateKMatrixThermal(double kmax){

	const int numdof=NUMVERTICES*NDOF1;
	double    penalty_factor;

	/*Initialize Element matrix and return if necessary*/
	if(!this->active) return NULL;
	ElementMatrix* Ke=new ElementMatrix(&node,NUMVERTICES,this->parameters);

	/*recover parameters: */
	parameters->FindParam(&penalty_factor,ThermalPenaltyFactorEnum);

	Ke->values[0]=kmax*pow((double)10,penalty_factor);

	/*Clean up and return*/
	return Ke;
}
/*}}}1*/
/*FUNCTION Pengrid::PenaltyCreatePVectorMelting {{{1*/
ElementVector* Pengrid::PenaltyCreatePVectorMelting(double kmax){
	
	const int numdof=NUMVERTICES*NDOF1;
	double pressure;
	double temperature;
	double melting_offset;
	double t_pmp;
	double dt,penalty_factor;

	/*recover pointers: */
	Penta* penta=(Penta*)element;

	/*check that pengrid is not a clone (penalty to be added only once)*/
	if (node->IsClone()) return NULL;
	ElementVector* pe=new ElementVector(&node,NUMVERTICES,this->parameters);

	/*Retrieve all inputs and parameters*/
	penta->GetParameterValue(&pressure,node,PressureEnum);
	penta->GetParameterValue(&temperature,node,TemperatureEnum);
	inputs->GetParameterValue(&melting_offset,MeltingOffsetEnum);
	parameters->FindParam(&dt,TimesteppingTimeStepEnum);
	parameters->FindParam(&penalty_factor,ThermalPenaltyFactorEnum);

	/*Compute pressure melting point*/
	t_pmp=matpar->GetMeltingPoint()-matpar->GetBeta()*pressure;

	/*Add penalty load
	  This time, the penalty must have the same value as the one used for the thermal computation
	  so that the corresponding melting can be computed correctly
	  In the thermal computation, we used kmax=melting_offset, and the same penalty_factor*/
	if (temperature<t_pmp){ //%no melting
		pe->values[0]=0;
	}
	else{
		if (dt) pe->values[0]=melting_offset*pow((double)10,penalty_factor)*(temperature-t_pmp)/dt;
		else    pe->values[0]=melting_offset*pow((double)10,penalty_factor)*(temperature-t_pmp);
	}

	/*Clean up and return*/
	return pe;
}
/*}}}1*/
/*FUNCTION Pengrid::PenaltyCreatePVectorThermal {{{1*/
ElementVector* Pengrid::PenaltyCreatePVectorThermal(double kmax){

	const int numdof=NUMVERTICES*NDOF1;
	double pressure;
	double t_pmp;
	double penalty_factor;

	Penta* penta=(Penta*)element;

	/*Initialize Element matrix and return if necessary*/
	if(!this->active) return NULL;
	ElementVector* pe=new ElementVector(&node,1,this->parameters);

	/*Retrieve all inputs and parameters*/
	penta->GetParameterValue(&pressure,node,PressureEnum);
	parameters->FindParam(&penalty_factor,ThermalPenaltyFactorEnum);

	/*Compute pressure melting point*/
	t_pmp=matpar->GetMeltingPoint()-matpar->GetBeta()*pressure;

	pe->values[0]=kmax*pow((double)10,penalty_factor)*t_pmp;

	/*Clean up and return*/
	return pe;
}
/*}}}1*/
/*FUNCTION Pengrid::UpdateInputs {{{1*/
void  Pengrid::UpdateInputs(double* solution){
	_error_("not supported yet!");
}
/*}}}1*/
/*FUNCTION Pengrid::ResetConstraint {{{1*/
void  Pengrid::ResetConstraint(void){
	active=0;
	zigzag_counter=0;
}
/*}}}1*/
