/*!\file: control_core.cpp
 * \brief: core of the control solution 
 */ 
#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../io/io.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./solutions.h"
#include "../modules/modules.h"
#include "../include/include.h"
#include "../solvers/solvers.h"


#ifdef _HAVE_TAO_
#include "tao.h"

/*Local prototype*/
int FormFunctionGradient(TAO_APPLICATION,Vec,double *,Vec,void*);
typedef struct {
	FemModel* femmodel;
} AppCtx;

void controltao_core(FemModel* femmodel){

	/*TAO*/
	int                i,n,info;
	//TaoMethod          method = "tao_blmvm";
	//TaoMethod        method = "tao_lmvm";
	TaoMethod        method = "tao_cg";
	TaoTerminateReason reason;
	TAO_SOLVER         tao;
	TAO_APPLICATION    controlapp;
	Vec                initial_solution = NULL;
	AppCtx             user;
	PetscInt           iter;
	double             ff,gnorm;

	/*Initialize TAO*/
	int argc; char **args; PetscGetArgs(&argc,&args);
	int ierr=TaoInitialize(&argc,&args,(char*)0,"");
	if(ierr) _error_("Could not initialize Tao");

	/*Line search options*/
	info = PetscOptionsSetValue("-tao_ls_stepmax","10e11"); if(info) _error_("STOP"); //does not work
	info = PetscOptionsSetValue("-tao_ls_stepmin","10e5"); if(info) _error_("STOP");    //does not work
	info = PetscOptionsSetValue("-tao_ls_maxfev","8"); if(info) _error_("STOP");

	/*TAO options: http://www.mcs.anl.gov/research/projects/tao/docs/manualpages/solver/TaoSetFromOptions.html*/
	info = PetscOptionsSetValue("-tao_monitor",""); if(info) _error_("STOP");
	info = PetscOptionsSetValue("-tao_gatol","10e-18"); if(info) _error_("STOP");
	info = PetscOptionsSetValue("-tao_max_its","10"); if(info) _error_("STOP");
	info = PetscOptionsSetValue("-tao_max_funcs","20"); if(info) _error_("STOP");

	/*Additional options*/
	//info = PetscOptionsSetValue("-info","/u/astrid-r1b/morlighe/svn/issm/trunk/test/NightlyRun/taolog.txt"); if(info) _error_("STOP");

	/*Initialize argument*/
	user.femmodel=femmodel;

	/*Set up and solve TAO*/
	GetVectorFromInputsx(&initial_solution,femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,MaterialsRheologyBbarEnum,VertexEnum);
	info = TaoCreate(PETSC_COMM_WORLD,method,&tao); if(info) _error_("STOP");
	info = TaoApplicationCreate(PETSC_COMM_WORLD,&controlapp); if(info) _error_("STOP");
	info = TaoAppSetInitialSolutionVec(controlapp,initial_solution);  if(info) _error_("STOP");
	info = TaoAppSetObjectiveAndGradientRoutine(controlapp,FormFunctionGradient,(void*)&user);  if(info) _error_("STOP");
	info = TaoSetOptions(controlapp,tao);  if(info) _error_("STOP");
	info = TaoSolveApplication(controlapp,tao); //if(info) _error_("STOP");

	/*Get solution status*/
	info = TaoGetSolutionStatus(tao,&iter,&ff,&gnorm,0,0,&reason); //CHKERRQ(info);
	switch(reason){ /*http://www.mcs.anl.gov/research/projects/tao/docs/manualpages/solver/TaoGetTerminationReason.html*/
		case TAO_CONVERGED_ATOL:       _printf_(true,"TAO_CONVERGED_ATOL (res <= atol)\n"); break;
		case TAO_CONVERGED_RTOL:       _printf_(true,"TAO_CONVERGED_RTOL (res/res0 <= rtol)\n"); break;
		case TAO_CONVERGED_TRTOL:      _printf_(true,"TAO_CONVERGED_TRTOL (xdiff <= trtol) \n"); break;
		case TAO_CONVERGED_MINF:       _printf_(true,"TAO_CONVERGED_MINF  (f <= fmin)\n"); break;
		case TAO_CONVERGED_USER:       _printf_(true,"TAO_CONVERGED_USER (user defined)\n"); break;
		case TAO_DIVERGED_MAXITS:      _printf_(true,"TAO_DIVERGED_MAXITS (its>maxits)\n"); break;
		case TAO_DIVERGED_NAN:         _printf_(true,"TAO_DIVERGED_NAN (Numerical problems)\n"); break;
		case TAO_DIVERGED_MAXFCN:      _printf_(true,"TAO_DIVERGED_MAXFCN (nfunc > maxnfuncts)\n"); break;
		case TAO_DIVERGED_LS_FAILURE:  _printf_(true,"TAO_DIVERGED_LS_FAILURE (line search failure)\n"); break;
		case TAO_DIVERGED_TR_REDUCTION:_printf_(true,"TAO_DIVERGED_TR_REDUCTION \n"); break;
		case TAO_DIVERGED_USER:        _printf_(true,"TAO_DIVERGED_USER (user defined)\n"); break;
		default: _printf_(true,"unknown reason");
	}
	if (reason<=0){
		_printf_(true,"Try a different TAO method, adjust some parameters, or check the function evaluation routines\n");
		_printf_(true," Iterations: %d,  Function Value: %4.2e, Residual: %4.2e \n",iter,ff,gnorm);
	}
	else{
	 _printf_(true,"TAO found a solution");
	}
	info = TaoView(tao);  if(info) _error_("STOP");

	/*Clean up*/
	info = TaoDestroy(tao);  if(info) _error_("STOP");
	info = TaoAppDestroy(controlapp);  if(info) _error_("STOP");
	VecFree(&initial_solution);

	/* Finalize TAO */
	TaoFinalize();
}

int FormFunctionGradient(TAO_APPLICATION taoapp, Vec X, double *fcn,Vec G,void *userCtx){

	/*Retreive arguments*/
	AppCtx   *user     = (AppCtx*)userCtx;
	FemModel *femmodel = user->femmodel;
	Vec       gradient = NULL;

	/*Temp*/
//	double*   Xserial=NULL;
//	VecToMPISerial(&Xserial,X);
//	printf("X= [%20.20g %20.20g %20.20g]\n",Xserial[0],Xserial[1],Xserial[2]);
	/*End Temp*/

	InputUpdateFromConstantx(femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,SurfaceAbsVelMisfitEnum,InversionCostFunctionEnum);
	InputUpdateFromVectorx(femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,X,MaterialsRheologyBbarEnum,VertexEnum);
	adjointdiagnostic_core(user->femmodel);
	Gradjx(&gradient, femmodel->elements,femmodel->nodes, femmodel->vertices,femmodel->loads, femmodel->materials,femmodel->parameters, MaterialsRheologyBbarEnum);
	VecScale(gradient,10e7);
	//VecScale(gradient,-1.);
	VecCopy(gradient,G);
	CostFunctionx(fcn, femmodel->elements,femmodel->nodes, femmodel->vertices,femmodel->loads, femmodel->materials, femmodel->parameters);

	//printf("X\n");
	//VecView(X,PETSC_VIEWER_STDOUT_SELF);

	//printf("Gradient\n");
	//VecView(G,PETSC_VIEWER_STDOUT_SELF);

	printf("f(x) = %g\n",*fcn);
	return 0;
}
#else
void controltao_core(FemModel* femmodel){
	_error_("TAO not installed");
}
#endif //_HAVE_TAO_ 
