/*! \file
  
  \brief Implementation of ReachConsts::CSReachConsts

  \author Michelle Strout, Barbara Kreaseck
  \version $Id: ReachConstsStandard.cpp,v 1.7 2005/03/18 18:14:16 ntallent Exp $

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/


#include "CSReachConsts.hpp"
#include <Utils/Util.hpp>
#include <OpenAnalysis/Utils/OutputBuilder.hpp>

namespace OA {
  namespace ReachConsts {

static bool debug = false;


//! ===================================================================
//! CSReachConsts methods
//! ===================================================================

CSReachConsts::CSReachConsts(ProcHandle p) 
{
  OA_DEBUG_CTRL_MACRO("DEBUG_CSReachConsts:ALL", debug);
}


//! Return the ConstValBasicInterface of the reaching constant
//! for the given MemRefExpr-AliasTagSet and StmtHandle
//! Returns NULL if no constant value exists for this MemRefExpr-AliasTagSet
//! at this StmtHandle
OA_ptr<ConstValBasicInterface> 
CSReachConsts::getReachConst(StmtHandle stmt,OA_ptr<Alias::AliasTagSet> tset)
{
  // no interpretation/implementation of a result with no specific context, yet
  // could keep a parallel structure that coalesces context results ... later
  // -- cannot just use null CallContext, as that is a legitimate CallContext
  //    for the root proc
  // just asserting for now

  assert("queried CSReachConsts::getReachConst(stmt, tset) with no given context -- not implemented" && false);

  OA_ptr<ConstValBasicInterface> retval;
  retval = NULL;
  return retval;

}

//! Return the ConstValBasicInterface of the reaching constant
//! for the given MemRefExpr-AliasTagSet and CallHandle
//! Returns NULL if no constant value exists for this MemRefExpr-AliasTagSet
//! at this CallHandle
OA_ptr<ConstValBasicInterface> 
CSReachConsts::getReachConst(CallHandle call,OA_ptr<Alias::AliasTagSet> tset)
{
  // no interpretation/implementation of a result with no specific context, yet
  // could keep a parallel structure that coalesces context results ... later
  // -- cannot just use null CallContext, as that is a legitimate CallContext
  //    for the root proc
  // just asserting for now

  assert("queried CSReachConsts::getReachConst(call, tset) with no given context -- not implemented" && false);

  OA_ptr<ConstValBasicInterface> retval;
  retval = NULL;
  return retval;

}



//! Return the ConstValBasicInterface of the reaching constant
//! for the given MemRefExpr-AliasTagSet and StmtHandle and CallContext
//! Returns NULL if no constant value exists for this MemRefExpr-AliasTagSet
//! at this StmtHandle and CallContext
OA_ptr<ConstValBasicInterface> 
CSReachConsts::getReachConst(StmtHandle stmt,OA_ptr<Alias::AliasTagSet> tset,
                             OA_ptr<Alias::CallContext> cc)
{
  OA_ptr<ConstValBasicInterface> retval; retval = NULL;

   if (tset.ptrEqual(0)) {return retval;}

   // wrap stmt and cc
   Alias::StmtHwContext shwc(stmt,cc);

   // look up RCPairSet
   OA_ptr<RCPairSet> rcSet;
   if (mReachConsts.find(shwc) == mReachConsts.end()) {
     // this stmt/cc is not found within mReachConsts ... error?
     assert("getReachConst(stmt,tset,cc): no RCPairSet found for stmt/cc pair\n"
            && false);
   }
   rcSet = mReachConsts[shwc];
   assert(!rcSet.ptrEqual(0));

   OA_ptr<RCPairSetIterator> rcIter;
   rcIter = new RCPairSetIterator(*rcSet);
   bool first = true;
   bool conflict = false;
   OA_ptr<ConstValBasicInterface> tempPtr;
   
   OA_ptr<Alias::AliasTagIterator> tIter;
   tIter = tset->getIterator();
   for (; tIter->isValid()&&!conflict; ++(*tIter)) {
     Alias::AliasTag tag = tIter->current();
     for(rcIter->reset(); rcIter->isValid(); ++(*rcIter)) {
       RCPair rc = rcIter->current();
       if (rc.getTag() == tag) {
         if (first) {
           retval = rc.getConstPtr();
           assert(!retval.ptrEqual(0));
         } else {
           tempPtr = rc.getConstPtr();
           if (tempPtr != retval) {
             conflict = true;
           }
         }
       }
     }
   }

   if (conflict) {
     retval = NULL;
   }

   return retval;
}

//! Return the ConstValBasicInterface of the reaching constant
//! for the given MemRefExpr-AliasTagSet and CallHandle and CallContext
//! Returns NULL if no constant value exists for this MemRefExpr-AliasTagSet
//! at this CallHandle and CallContext
OA_ptr<ConstValBasicInterface> 
CSReachConsts::getReachConst(CallHandle call,OA_ptr<Alias::AliasTagSet> tset,
                             OA_ptr<Alias::CallContext> cc)
{
   OA_ptr<ConstValBasicInterface> retval; retval = NULL;

   if (tset.ptrEqual(0)) {return retval;}

   // wrap call and cc
   Alias::CallHwContext chwc(call,cc);

   // look up RCPairSet
   OA_ptr<RCPairSet> rcSet;
   if (mReachConstsForCallStmt.find(chwc) == mReachConstsForCallStmt.end()) {
     // this stmt/cc is not found within mReachConstsForCallStmt ... error?
     assert("getReachConst(call,tset,cc): no RCPairSet found for call/cc pair\n"
            && false);
   }
   rcSet = mReachConstsForCallStmt[chwc];
   assert(!rcSet.ptrEqual(0));

   OA_ptr<RCPairSetIterator> rcIter;
   rcIter = new RCPairSetIterator(*rcSet);
   bool first = true;
   bool conflict = false;
   OA_ptr<ConstValBasicInterface> tempPtr;
   
   OA_ptr<Alias::AliasTagIterator> tIter;
   tIter = tset->getIterator();
   for (; tIter->isValid()&&!conflict; ++(*tIter)) {
     Alias::AliasTag tag = tIter->current();
     for(rcIter->reset(); rcIter->isValid(); ++(*rcIter)) {
       RCPair rc = rcIter->current();
       if (rc.getTag() == tag) {
         if (first) {
           retval = rc.getConstPtr();
           assert(!retval.ptrEqual(0));
         } else {
           tempPtr = rc.getConstPtr();
           if (tempPtr != retval) {
             conflict = true;
           }
         }
       }
     }
   }

   if (conflict) {
     retval = NULL;
   }

   return retval;
}



//! reset a statement's RCPairSet by sending in a NULL OA_ptr
//! otherwise, stores an RCPairSet that reaches this StmtHandle and CallContext
void CSReachConsts::storeRCPairSet(StmtHandle s, 
                                   OA_ptr<RCPairSet> rcSet,
                                   OA_ptr<Alias::CallContext> cc)
{
  // wrap stmt and context
  Alias::StmtHwContext shwc(s,cc);

  if (rcSet.ptrEqual(0)) {
    mReachConsts[shwc] = new RCPairSet();
  } else {
    mReachConsts[shwc] = (rcSet->clone()).convert<RCPairSet>();
  }
}


//! Return the set of reaching constants for a given stmt and CallContext
OA_ptr<RCPairSetIterator>
CSReachConsts::getRCPairIterator(StmtHandle s, OA_ptr<Alias::CallContext> cc)
{
   // wrap stmt and context
   Alias::StmtHwContext shwc(s,cc);

   if (mReachConsts[shwc].ptrEqual(NULL)) {
        mReachConsts[shwc] = new RCPairSet();
   }

   OA_ptr<RCPairSetIterator> retval;
   retval = new RCPairSetIterator(*(mReachConsts[shwc]));
   return retval;
}




//! reset a call's RCPairSet by sending in a NULL OA_ptr
//! otherwise, stores an RCPairSet that reaches this CallHandle and CallContext
void CSReachConsts::storeRCPairSet(CallHandle c, 
                                   OA_ptr<RCPairSet> rcSet,
                                   OA_ptr<Alias::CallContext> cc)
{
  // wrap call and context
  Alias::CallHwContext chwc(c,cc);

  if (rcSet.ptrEqual(0)) {
    mReachConstsForCallStmt[chwc] = new RCPairSet();
  } else {
    mReachConstsForCallStmt[chwc] = (rcSet->clone()).convert<RCPairSet>();
  }
}


//! Return the set of reaching constants for a given CallHandle and CallContext
OA_ptr<RCPairSetIterator>
CSReachConsts::getRCPairIterator(CallHandle c, OA_ptr<Alias::CallContext> cc)
{
   // wrap call and context
   Alias::CallHwContext chwc(c,cc);

   if (mReachConstsForCallStmt[chwc].ptrEqual(NULL)) {
     mReachConstsForCallStmt[chwc] = new RCPairSet();
   }

   OA_ptr<RCPairSetIterator> retval;
   retval = new RCPairSetIterator(*(mReachConstsForCallStmt[chwc]));
   return retval;
}





//! =============== Output/Debugging =====================
void CSReachConsts::output(IRHandlesIRInterface& ir) const
{
    std::cout << "Please call output with alias::Interface"
              << std::endl;
    assert(0);
}






void CSReachConsts::output(OA::IRHandlesIRInterface &ir,
                                 Alias::Interface& alias) const
{
    sOutBuild->objStart("CSReachConsts");

    sOutBuild->mapStart("mReachConsts", "StmtHwContext", 
                        "OA::OA_ptr<RCPairSet> ");
    std::map<Alias::StmtHwContext, OA::OA_ptr<RCPairSet> >::const_iterator
        reg_mReachConsts_iterator;
    for(reg_mReachConsts_iterator = mReachConsts.begin();
        reg_mReachConsts_iterator != mReachConsts.end();
        reg_mReachConsts_iterator++)
    {
        const Alias::StmtHwContext &first = reg_mReachConsts_iterator->first;
        OA::OA_ptr<RCPairSet> second = reg_mReachConsts_iterator->second;

        sOutBuild->mapEntryStart();

        sOutBuild->mapKeyStart();
        first.output(ir);
        sOutBuild->mapKeyEnd();

        sOutBuild->mapValueStart();
        second->output(ir,alias);
        sOutBuild->mapValueEnd();

        sOutBuild->mapEntryEnd();
    }
    sOutBuild->mapEnd("mReachConsts");


    sOutBuild->mapStart("mReachConstsForCallStmt", "CallHandle", "OA::OA_ptr<RCPairSet> ");
    std::map<Alias::CallHwContext, OA::OA_ptr<RCPairSet> >::const_iterator
        reg_mReachConstsForCallStmt_iterator;
    for(reg_mReachConstsForCallStmt_iterator = mReachConstsForCallStmt.begin();
        reg_mReachConstsForCallStmt_iterator != mReachConstsForCallStmt.end();
        reg_mReachConstsForCallStmt_iterator++)
    {
      const Alias::CallHwContext &first = reg_mReachConstsForCallStmt_iterator->first;
      OA::OA_ptr<RCPairSet> second = reg_mReachConstsForCallStmt_iterator->second;
        sOutBuild->mapEntryStart();

        sOutBuild->mapKeyStart();
        first.output(ir);
        sOutBuild->mapKeyEnd();

        sOutBuild->mapValueStart();
        second->output(ir,alias);
        sOutBuild->mapValueEnd();

        sOutBuild->mapEntryEnd();
    }
    sOutBuild->mapEnd("mReachConstsForCallStmt");


    sOutBuild->objEnd("CSReachConsts");
}







//! incomplete output of info for debugging, just lists stmts
//! and associated set of RCPair that are reaching const defs for the given
void CSReachConsts::dump(std::ostream& os,
                             OA_ptr<OA::IRHandlesIRInterface> ir)
{
    std::map<Alias::StmtHwContext, OA_ptr<RCPairSet> >::iterator mapIter;
    for (mapIter = mReachConsts.begin(); mapIter != mReachConsts.end(); mapIter++) {
        Alias::StmtHwContext shwc = mapIter->first;
        StmtHandle s = shwc.getStmt();
        OA_ptr<Alias::CallContext> cc; cc = shwc.getCallContext();

        os << "StmtHandle(" << s.hval() << ") " << ir->toString(s)
           << std::endl << "\t";
        cc->dump(os,*ir);
        os << "\treachConsts: " << std::endl;
        // iterate over reaching consts for statement s

        OA_ptr<RCPairSet> rcSet = mapIter->second;
        rcSet->dump(os,ir);

        os << std::endl;
    }

    std::map<Alias::CallHwContext, OA_ptr<RCPairSet> >::iterator cIter;
    for (cIter = mReachConstsForCallStmt.begin();
         cIter != mReachConstsForCallStmt.end();
         cIter++) {

        Alias::CallHwContext chwc = cIter->first;
        CallHandle c = chwc.getCall();
        OA_ptr<Alias::CallContext> cc; cc = chwc.getCallContext();

        OA_ptr<RCPairSet> rcSet = cIter->second;
        
        os << "CallHandle ( " << ir->toString(c) << " )\n\t";
        cc->dump(os,*ir);
        os << "\treachConsts:\n";
        rcSet->dump(os,ir);
        os << std::endl;
    }
}






  } // end of ReachConsts namespace
} // end of OA namespace
