/*! \file
  
  \brief Definition of a context-sensitive dataflow algorithm for ICFGs.

  \authors Michelle Strout, Barbara Kreaseck

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#include "ICFGCSDFSolver.hpp"
#include <Utils/Util.hpp>
#include <list>

namespace OA {
  namespace DataFlow {

static bool debug = false;

ICFGCSDFSolver::ICFGCSDFSolver(DFDirectionType pDirection, 
                               ICFGCSDFProblem& prob)
  : mDirection(pDirection), mCSDFProb(prob)
{
    OA_DEBUG_CTRL_MACRO("DEBUG_ICFGCSDFSolver", debug);
    mCCSetPerProc = NULL;
}

void
ICFGCSDFSolver::solve(OA_ptr<ICFG::ICFGInterface> icfg,
                      OA_ptr<Alias::CCSetPerProc> ccResults,
                      DFPImplement algorithm)
{
  mCCSetPerProc = ccResults;

    // remove all mappings of nodes to data flow sets 
    mNodeInSetMap.clear();
    mNodeOutSetMap.clear();
    mNodeInitTransApp.clear();

    // now done during atDGraphNode as needed
    //    mTop = mCSDFProb.initializeTop();

    DataFlow::DGraphSolverDFP::solve(icfg, 
            ((mDirection==Forward) ? DGraph::DEdgeOrg : DGraph::DEdgeRev),
            algorithm);

    /*
    // if forward then return DataFlowSet for exit
    if (mDirection==Forward) {
        OA_ptr<ICFG::ICFGStandard::Node> exitnode = icfg->getExit();
        return mNodeInSetMap[exitnode];

    // if backward then return DataFlowSet for entry
    } else {
        OA_ptr<ICFG::ICFGStandard::Node> entrynode = icfg->getEntry();
        return mNodeOutSetMap[entrynode];
    }
    */
}
  
//========================================================
// implementation of DGraphIterativeDFP callbacks
//========================================================
  
//--------------------------------------------------------
// initialization upcall 
//--------------------------------------------------------


void ICFGCSDFSolver::initialize(OA_ptr<DGraph::DGraphInterface> dg)
{
    OA_ptr<ICFG::ICFGInterface> cfg = dg.convert<ICFG::ICFGInterface>();
    OA_ptr<ICFG::NodesIteratorInterface> nodeIterPtr;
    OA_ptr<CC2DFSMap> ccdfsMapIn;
    OA_ptr<CC2DFSMap> ccdfsMapOut;
    OA_ptr<DataFlowSet> initialDFS;
    OA_ptr<Alias::CallContext> cc;
    OA_ptr<ICFG::NodeInterface> iNode;

    // iterate over all nodes and allocate maps
    if (debug) {
      std::cout << "initializing: maps for ALL nodes\n";
    }
    nodeIterPtr = cfg->getICFGNodesIterator();
    for (  ;nodeIterPtr->isValid(); ++(*nodeIterPtr) ) {
      if (debug) {
	std::cout << "\tNode " << (nodeIterPtr->current())->getId()
		  << std::endl;
      }
      iNode = nodeIterPtr->currentICFGNode();

      mNodeInSetMap[iNode] = new CC2DFSMap;

      mNodeOutSetMap[iNode] = new CC2DFSMap;

      mNodeInitTransApp[iNode] = false;
    }


    // Now, initialize SetMaps for entry nodes

    if (mDirection==Forward) { // forward
      // iterate over entry nodes and call initialization routine
      // that sets up IN DataFlowSets

      nodeIterPtr = cfg->getICFGEntryNodesIterator();
      if (debug) {
        std::cout << "initializing:  looping over entry nodes:\n";
      }
      for ( ;nodeIterPtr->isValid(); ++(*nodeIterPtr) ) {
	iNode = nodeIterPtr->currentICFGNode();
        ProcHandle proc = iNode->getProc();

	if (debug) {
	  std::cout << "\tEntry Node " << iNode->getId() << std::endl;
	}
        
        /* BK: cannot assume that EntryNodes are from subroutines
           that are never called.  This Open64 weirdness with those
           extra Control_Flow Nodes that only have one stmt (a return) 
           and one out-going edge to the ExitNode has come back to
           haunt me ... I HATE THESE ... have I ever told anyone that
           I HATE THESE ????  Argh.
           ==> so the issue is that it is wrong to process these nodes
           with an initialCC of CallHandle(0).  We really don't want to 
           introduce a DataFlowSet with CallHandle(0) into the DataFlow
           going out of these nodes.  (Also, none of the maps in the
           ICFGCSDFProblem are initialized to deal with a CallContext of
           CallHandle(0) on these nodes if they are within a proc that
           gets called from another proc (i.e., if they are within a proc
           that is not a root node within the CallGraph))

           1) Shouldn't we just be able to prune these away early so that
           they never have to haunt us again?

           2) We could just make a call-back to the mCSDFProb to query
           its mCCSetsPerProc for us ...

           3) Looks like we will need to send the CCSetsPerProc into 
           the ICFGCSDFSolver so that the initialization routine here
           can iterate through the CallContexts for the Proc of the
           specific EntryNode.  Actually, this seems easiest for now.

        */
        
        // for each CC for this node's proc:
        OA_ptr<Alias::CallContextSetIterator> ccSetIter;
        ccSetIter = mCCSetPerProc->getCallContextSet(proc);
        for (ccSetIter->reset() ; ccSetIter->isValid(); ++(*ccSetIter)) {
          cc = ccSetIter->current();
        
          initialDFS = mCSDFProb.initializeNodeIN(iNode,cc);
          mNodeInSetMap[iNode]->assign(cc,initialDFS);
          if (debug) {
            std::cout << "\t\tmap size = " << mNodeInSetMap[iNode]->size()
                      << std::endl;
          }
        }
      }

    } else { // backward
      // iterate over exit nodes and call initialization routine
      // that sets up OUT DataFlowSets

      nodeIterPtr = cfg->getICFGExitNodesIterator();
      if (debug) {
        std::cout << "initializing:  looping over exit nodes:\n";
      }

      for ( ;nodeIterPtr->isValid(); ++(*nodeIterPtr) ) {
	iNode = nodeIterPtr->currentICFGNode();
        ProcHandle proc = iNode->getProc();
	if (debug) {
	  std::cout << "\tExit Node " << iNode->getId() << std::endl;
	}
	
        // for each CC for this node's proc:
        OA_ptr<Alias::CallContextSetIterator> ccSetIter;
        ccSetIter = mCCSetPerProc->getCallContextSet(proc);
        for (ccSetIter->reset() ; ccSetIter->isValid(); ++(*ccSetIter)) {
          cc = ccSetIter->current();

          initialDFS = mCSDFProb.initializeNodeOUT(iNode,cc);
          mNodeOutSetMap[iNode]->assign(cc,initialDFS);
          if (debug) {
            std::cout << "\t\tmap size = " << mNodeOutSetMap[iNode]->size()
                      << std::endl;
          }
        }
      }
    }
}


//--------------------------------------------------------
// solver upcalls
//--------------------------------------------------------
bool ICFGCSDFSolver::atDGraphNode( OA_ptr<DGraph::NodeInterface> pNode, 
                                 DGraph::DGraphEdgeDirection pOrient)
{
    bool changed = false;
    OA_ptr<ICFG::NodeInterface> iNode 
      = pNode.convert<OA::ICFG::NodeInterface>();
    
    if (debug) {
      std::cout << "ICFGCSDFSolver::atDGraphNode: ICFG node = ";
      std::cout << iNode->getId() << std::endl;
    }
    
    //-----------------------------------------------------
    // do a meet of all out information from nodes that are
    // predecessors with current input for this node
    // based on the flow direction, edge type,
    // and node type
    // FIXME: Have to meet with current input for this node because
    // of ReachConsts?
    //-----------------------------------------------------
    
    // === BK: not worrying about ReachConsts right now, will revisit this XXX
    /*
      OA_ptr<DataFlowSet> meetPartialResult = mTop->clone();
      if (pOrient==DGraph::DEdgeOrg) {
      meetPartialResult = mCSDFProb.meet(meetPartialResult,mNodeInSetMap[iNode]);
      } else {
      meetPartialResult = mCSDFProb.meet(meetPartialResult,mNodeOutSetMap[iNode]);
      }
    */
    
    // set up iterator for predecessor edges
    OA_ptr<ICFG::EdgesIteratorInterface> predIterPtr;
    if (pOrient==DGraph::DEdgeOrg) {
      predIterPtr = iNode->getICFGIncomingEdgesIterator();
    } else {
      predIterPtr = iNode->getICFGOutgoingEdgesIterator();
    }

    OA_ptr<CC2DFSMap> meetPartialResult;
    if (pOrient==DGraph::DEdgeOrg) {
      if (iNode->num_incoming() == 0) {
        // Then there are no predEdges from which to create meetPartialResult
        meetPartialResult = mNodeInSetMap[iNode]->clone();
      } else {
        meetPartialResult = new CC2DFSMap;
      }
    } else {
      if (iNode->num_outgoing() == 0) {
        // Then there are no predEdges from which to create meetPartialResult
        meetPartialResult = mNodeOutSetMap[iNode]->clone();
      } else {
        meetPartialResult = new CC2DFSMap;
      }
    }


    std::set<OA_ptr<Alias::CallContext> > inCCSet;
    std::set<OA_ptr<Alias::CallContext> >::iterator inCCSetIter;

    std::list<OA_ptr<CC2DFSMap> > inMapList;
    std::list<OA_ptr<CC2DFSMap> >::iterator inMapListIter;

    OA_ptr<CC2DFSMap> inMap;

    // for use with CallNodes
    OA_ptr<std::set<OA_ptr<Alias::CallContext> > > inCallCCSet;

    //----
    // 1) set meetPartialResult to be the Top set for each CC that exists within
    //    the incoming edge CC2DFSMaps, compile set of in-Maps at same time

    for (; predIterPtr->isValid(); ++(*predIterPtr)) {
      OA_ptr<ICFG::EdgeInterface> predEdge = predIterPtr->currentICFGEdge();
      OA_ptr<ICFG::NodeInterface> predNode;
      OA_ptr<DataFlowSet> predSet;
      OA_ptr<DataFlowSet> inSet;
      OA_ptr<Alias::CallContext> predCC;
      OA_ptr<CC2DFSMapIterator> mapIter;
      inMap = new CC2DFSMap;

      if (pOrient==DGraph::DEdgeOrg) {
        OA_ptr<ICFG::NodeInterface> predNode = predEdge->getICFGSource();
        
	switch(predEdge->getType()) {
          
        case (ICFG::CALL_EDGE):
          {
            inCallCCSet = new std::set<OA_ptr<Alias::CallContext> >;
            
            CallHandle call = predEdge->getCall();
            
            // iterate through predecessor Map to call problem method
            mapIter = new CC2DFSMapIterator(*(mNodeOutSetMap[predNode]));
            for ( ; mapIter->isValid(); ++(*mapIter)) {
              CCDFSPair cdPair = mapIter->current();
              predCC = cdPair.getContextPtr();
              
              // remember predCCs for this call
              inCallCCSet->insert(predCC);
              
              // append CallHandle to predCC in a k-context way, giving newCC
              OA_ptr<Alias::CallContext> newCC;
              newCC = predCC->clone();
              newCC->appendForce(call);
              
              predSet = cdPair.getDFSetPtr();
              
              inSet = mCSDFProb.callerToCallee(predEdge->getSourceProc(),
                                               predSet, predCC, call,
                                               predEdge->getSinkProc());
              
              
              // if there was a currSet for this newCC in inMap already,
              // get meet of currSet and inSet
              OA_ptr<DataFlowSet> currSet;
              currSet = inMap->getDFS(newCC);
              if (!currSet.ptrEqual(0)) {
                inSet = mCSDFProb.meet(inSet, currSet, newCC);
              }
              
              inMap->assign(newCC,inSet);
              inCCSet.insert(newCC);
            }
            // store in-coming CCs for this call
            mCall2CallContextInSet[call] = inCallCCSet;
          }
          break;
        case (ICFG::RETURN_EDGE):
          {
            // in Forward mode, iNode must be a RETURN_NODE and predNode must be
            // an EXIT_NODE ... do not want the entire outSet for the predNode, 
            // just those that are assoc. with the CallHandle on the RETURN_EDGE.

            CallHandle call = predEdge->getCall();
            OA_ptr<CC2DFSMap> outMap = mCallOutSetMap[call];
            // these outMap CC's have already been stripped of the call

            if (!outMap.ptrEqual(0)) { 
              // iterate through predecessor Map to call problem method
              mapIter = new CC2DFSMapIterator(*outMap);
              for ( ; mapIter->isValid(); ++(*mapIter)) {
                CCDFSPair cdPair = mapIter->current();
                predCC = cdPair.getContextPtr();
                inCCSet.insert(predCC);
                
                predSet = cdPair.getDFSetPtr();
                
                inSet = mCSDFProb.calleeToCaller(predEdge->getSourceProc(),
                                                 predSet, predCC, 
                                                 predEdge->getCall(),
                                                 predEdge->getSinkProc());
                inMap->assign(predCC,inSet);
              }
            } else {
              if (debug) {
                std::cout << "\t\tprocessing return_edge, no outMap found\n";
              }
            }
          }
          break;
        case (ICFG::CALL_RETURN_EDGE):
          {
            // iterate through predecessor Map to call problem method
            mapIter = new CC2DFSMapIterator(*(mNodeOutSetMap[predNode]));
            for ( ; mapIter->isValid(); ++(*mapIter)) {
              CCDFSPair cdPair = mapIter->current();
              predCC = cdPair.getContextPtr();
              inCCSet.insert(predCC);
              
              predSet = cdPair.getDFSetPtr();
              
              // has same parameter order as callerToCallee() for now
              // but will need to be changed when callToReturn is 
              // revisited.
              inSet = mCSDFProb.callToReturn(predEdge->getSourceProc(),
                                             predSet, predCC, 
                                             predEdge->getCall(),
                                             predEdge->getSinkProc());
              inMap->assign(predCC,inSet);
            }
          }
          break;
        case (ICFG::CFLOW_EDGE):
          {
            // iterate through predecessor Map to call problem method
            mapIter = new CC2DFSMapIterator(*(mNodeOutSetMap[predNode]));
            for ( ; mapIter->isValid(); ++(*mapIter)) {
              CCDFSPair cdPair = mapIter->current();
              predCC = cdPair.getContextPtr();
              inCCSet.insert(predCC);
              
              predSet = cdPair.getDFSetPtr();
              // inSet = predSet;
              inMap->assign(predCC,predSet);
            }
          }
          break;
        } // end of switch statement

        if (debug) {
          std::cout << "performing forward meet with pred node " 
                    << predNode->getId() << std::endl;
        }                                              

      // backward flow
      } else {
        OA_ptr<ICFG::NodeInterface> predNode = predEdge->getICFGSink();
        switch(predEdge->getType()) {
        case (ICFG::CALL_EDGE):
          {
            // in Backward mode, iNode must be a CALL_NODE and predNode must be
            // an ENTRY_NODE ... do not want the entire outSet for the predNode, 
            // just those that are assoc. with the CallHandle on the CALL_EDGE.

            CallHandle call = predEdge->getCall();
            OA_ptr<CC2DFSMap> outMap = mCallOutSetMap[call];
            // these outMap CC's have already been stripped of the call

            if (!outMap.ptrEqual(0)) {
              // iterate through predecessor Map to call problem method
              mapIter = new CC2DFSMapIterator(*outMap);
              for ( ; mapIter->isValid(); ++(*mapIter)) {
                CCDFSPair cdPair = mapIter->current();
                predCC = cdPair.getContextPtr();
                inCCSet.insert(predCC);
                
                predSet = cdPair.getDFSetPtr();
                
                inSet = mCSDFProb.calleeToCaller(predEdge->getSinkProc(),
                                                 predSet, predCC, 
                                                 predEdge->getCall(),
                                                 predEdge->getSourceProc());
                inMap->assign(predCC,inSet);
              }
            }
          }
          break;
        case (ICFG::RETURN_EDGE):
          {
            inCallCCSet = new std::set<OA_ptr<Alias::CallContext> >;
            
            CallHandle call = predEdge->getCall();
            
            // iterate through predecessor Map to call problem method
            mapIter = new CC2DFSMapIterator(*(mNodeInSetMap[predNode]));
            for ( ; mapIter->isValid(); ++(*mapIter)) {
              CCDFSPair cdPair = mapIter->current();
              predCC = cdPair.getContextPtr();

              // remember predCCs for this call
              inCallCCSet->insert(predCC);
              
              // append CallHandle to predCC in a k-context way, giving newCC
              OA_ptr<Alias::CallContext> newCC;
              newCC = predCC->clone();
              newCC->appendForce(call);

              predSet = cdPair.getDFSetPtr();
              
              // use outset for RETURN_NODE in caller
              inSet = mCSDFProb.callerToCallee(predEdge->getSinkProc(),
                                               predSet, predCC, call,
                                               predEdge->getSourceProc());

              // if there was a currSet for this newCC in inMap already,
              // get meet of currSet and inSet
              OA_ptr<DataFlowSet> currSet;
              currSet = inMap->getDFS(newCC);
              if (!currSet.ptrEqual(0)) {
                inSet = mCSDFProb.meet(inSet, currSet, newCC);
              }
              
              inMap->assign(newCC,inSet);
              inCCSet.insert(newCC);
            }
            // store in-coming CCs for this call
            mCall2CallContextInSet[call] = inCallCCSet;
          }
          break;
        case (ICFG::CALL_RETURN_EDGE):
          {
            if (debug) {
              std::cout << "ICFG::CALL_RETURN_EDGE\n";
            }
            // iterate through predecessor Map to call problem method
            mapIter = new CC2DFSMapIterator(*(mNodeInSetMap[predNode]));
            for ( ; mapIter->isValid(); ++(*mapIter)) {
              CCDFSPair cdPair = mapIter->current();
              predCC = cdPair.getContextPtr();
              inCCSet.insert(predCC);
              
              predSet = cdPair.getDFSetPtr();
              
              // has same parameter order as callerToCallee() for now
              // but will need to be changed when callToReturn is 
              // revisited.  Currently, only LocDFSet is used.
              inSet = mCSDFProb.callToReturn(predEdge->getSinkProc(),
                                             predSet, predCC,
                                             predEdge->getCall(),
                                             predEdge->getSourceProc());
              inMap->assign(predCC,inSet);
              if (debug) {
                std::cout << "storing for predCC/inSet: ";
                mCSDFProb.printCallContext(std::cout, predCC);
                mCSDFProb.printDFSet(std::cout, inSet);
              }
                
            }
          }
          break;
        case (ICFG::CFLOW_EDGE):
          {
            // iterate through predecessor Map to call problem method
            mapIter = new CC2DFSMapIterator(*(mNodeInSetMap[predNode]));
            for ( ; mapIter->isValid(); ++(*mapIter)) {
              CCDFSPair cdPair = mapIter->current();
              predCC = cdPair.getContextPtr();
              inCCSet.insert(predCC);
              
              predSet = cdPair.getDFSetPtr();
              
              //inSet = predSet;
              inMap->assign(predCC,predSet);
            }
          }
          break;
        } // end of switch statement

        if (debug) {
          std::cout << "performing backward meet with succ node " 
                    << predNode->getId() << std::endl;
        }                                              

      } // end of else { dir == backward }

      inMapList.push_back(inMap);  // store for later use in step 2 below

    } // end of predNode iteration

    OA_ptr<DataFlowSet> topSet;
    OA_ptr<Alias::CallContext> predCC;

    for (inCCSetIter = inCCSet.begin(); 
	 inCCSetIter != inCCSet.end();
	 ++inCCSetIter)
      {
	predCC = *inCCSetIter;
	topSet = mCSDFProb.initializeTop(predCC);
	meetPartialResult->assign(predCC, topSet);
      }

    //----
    // 2) iterate over inMap's and do meet operation
    for (inMapListIter = inMapList.begin();
	 inMapListIter != inMapList.end();
	 ++inMapListIter)
      {
	meetPartialResult = doMeet(meetPartialResult, *inMapListIter, iNode);
      }

    //---
    // update the appropriate set for this node
    if (pOrient==DGraph::DEdgeOrg) { // forward
      if ( mNodeInSetMap[iNode] != meetPartialResult ) {
        mNodeInSetMap[iNode] = meetPartialResult;
        changed = true;
      }
    } else { // reverse
      if ( mNodeOutSetMap[iNode] != meetPartialResult ) {
        mNodeOutSetMap[iNode] = meetPartialResult;
        changed = true;
      }
    }

    // if the data flowing into this node has changed or if the
    // transfer functions have never been applied, then
    // loop through statements in the CFG node/(Basic Block) 
    // calculating the new node out
    if (debug) {
      std::cout << "\tchanged = " << changed << ", mNITA[iNode]=" 
                << mNodeInitTransApp[iNode] << std::endl;
    }
    if (changed || !mNodeInitTransApp[iNode]) {
      changed = false;  // reuse to determine if there is a change based
                        // on the block transfer function
      mNodeInitTransApp[iNode] = true;
      OA_ptr<CC2DFSMapIterator> mapIter;

      // Forward direction
      if (pOrient==DGraph::DEdgeOrg) {
	OA_ptr<CC2DFSMap> currOutMap;
	currOutMap = new CC2DFSMap;
        OA_ptr<CC2DFSMap> currInMap;
        currInMap = mNodeInSetMap[iNode];
        if (iNode->getType()==ICFG::EXIT_NODE) {
          // exit nodes have no statements, but need to fill the mCallOutSetMap
          // for each CallHandle on an outgoingEdge (should be RETURN_EDGE)
          OA_ptr<CC2DFSMap> outMap;
          OA_ptr<ICFG::EdgesIteratorInterface> succIterPtr;
          succIterPtr = iNode->getICFGOutgoingEdgesIterator();
          for (; succIterPtr->isValid(); ++(*succIterPtr)) {
            OA_ptr<ICFG::EdgeInterface> succEdge = succIterPtr->currentICFGEdge();
            CallHandle call = succEdge->getCall();
            OA_ptr<std::set<OA_ptr<Alias::CallContext> > > inCallCCSet;
            inCallCCSet = mCall2CallContextInSet[call];
            if (inCallCCSet.ptrEqual(0)) {
              // hit exit_node before entry_node
              continue;
            }
            outMap = new CC2DFSMap;
            std::set<OA_ptr<Alias::CallContext> >::iterator inCCCSiter;
            for (inCCCSiter = inCallCCSet->begin();
                 inCCCSiter != inCallCCSet->end();  inCCCSiter++) {
              OA_ptr<Alias::CallContext> inCC, newCC;
              inCC = *inCCCSiter;
              newCC = inCC->clone();
              newCC->appendForce(call);
              OA_ptr<DataFlowSet> dfSet;
              dfSet = currInMap->getDFS(newCC);

              //assert((!dfSet.ptrEqual(0)) && 
              //       "exit_node newCC lookup failure in outSet");
              if (dfSet.ptrEqual(0)) {
                continue; // no data on that newCC yet on the entry side
              }

              //dfSet clone needed here.  ?needed for inCC?
              outMap->assign(inCC->clone(), dfSet->clone());

            }
            mCallOutSetMap[call] = outMap;
            if (debug) {
              std::cout << "For call on Edge #" << succEdge->getId()
                        << " STORING ";
              OA_ptr<CC2DFSMapIterator> defIter;
              defIter = new CC2DFSMapIterator(*outMap);
              for (; defIter->isValid(); ++(*defIter)) {
                CCDFSPair ccdfsPair = defIter->current();
                OA_ptr<Alias::CallContext> cc;
                cc = ccdfsPair.getContextPtr();
                mCSDFProb.printCallContext(std::cout, cc);
                mCSDFProb.printDFSet(std::cout,ccdfsPair.getDFSetPtr());
              }
            }

          }
        } // end of if EXIT_NODE

	mapIter = new CC2DFSMapIterator(*currInMap);
	for ( ; mapIter->isValid(); ++(*mapIter)) {
          CCDFSPair cdPair = mapIter->current();
	  OA_ptr<Alias::CallContext> inCC;
	  inCC = cdPair.getContextPtr();
          if (debug) {
            std::cout << "for CC: ";
            mCSDFProb.printCallContext(std::cout,inCC);
            std::cout << "\n";
          }

	  OA_ptr<DataFlowSet> inSet;
	  inSet = cdPair.getDFSetPtr();
          
	  OA_ptr<DataFlowSet> currOut; 
	  OA_ptr<DataFlowSet> inClone = inSet->clone();
          
	  // call transfer methods based on what kind of node 
	  if (iNode->getType()==ICFG::ENTRY_NODE) {
            currOut = mCSDFProb.entryTransfer(iNode->getProc(), inClone, inCC);
            
	    // otherwise it is a normal node and should apply regular transfer
	    // loop through statements in forward order
	  } else {
            if (iNode->getType()==ICFG::CALL_NODE) {
              OA::CallHandle call = iNode->getCall();
              currOut = mCSDFProb.transfer(iNode->getProc(), inClone,
                                           inCC, call);
            } else  {
              currOut = inClone;
              OA_ptr<CFG::NodeStatementsIteratorInterface> stmtIterPtr 
                = iNode->getNodeStatementsIterator();
              for (; stmtIterPtr->isValid(); ++(*stmtIterPtr)) {
                OA::StmtHandle stmt = stmtIterPtr->current();
                currOut = mCSDFProb.transfer(iNode->getProc(), currOut, 
                                             inCC, stmt);
              }
            }
          }

	  currOutMap->assign(inCC, currOut);

	} 

        // check to see if changed from last time
        if (currOutMap != mNodeOutSetMap[iNode] ) {
          changed = true;
          mNodeOutSetMap[iNode] = currOutMap;
        }
      
      // Reverse direction
      } else { 
	OA_ptr<CC2DFSMap> currInMap;
	currInMap = new CC2DFSMap;
	OA_ptr<CC2DFSMap> currOutMap;
        currOutMap = mNodeOutSetMap[iNode];

        if (iNode->getType()==ICFG::ENTRY_NODE) {
          // entry nodes have no statements, but need to fill the mCallOutSetMap
          // for each CallHandle on an incomingEdge (should be CALL_EDGE)
          OA_ptr<CC2DFSMap> inMap;
          OA_ptr<ICFG::EdgesIteratorInterface> succIterPtr;
          succIterPtr = iNode->getICFGIncomingEdgesIterator();
          for (; succIterPtr->isValid(); ++(*succIterPtr)) {
            OA_ptr<ICFG::EdgeInterface> succEdge = succIterPtr->currentICFGEdge();
            CallHandle call = succEdge->getCall();
            OA_ptr<std::set<OA_ptr<Alias::CallContext> > > inCallCCSet;
            inCallCCSet = mCall2CallContextInSet[call];
            if (inCallCCSet.ptrEqual(0)) {
              // hit entry_node before exit_node
              continue;
            }
            inMap = new CC2DFSMap;
            std::set<OA_ptr<Alias::CallContext> >::iterator inCCCSiter;
            for (inCCCSiter = inCallCCSet->begin();
                 inCCCSiter != inCallCCSet->end();  inCCCSiter++) {
              OA_ptr<Alias::CallContext> inCC, newCC;
              inCC = *inCCCSiter;
              newCC = inCC->clone();
              newCC->appendForce(call);
              if (debug) {
                std::cout << "inCC: ";
                mCSDFProb.printCallContext(std::cout,inCC);
                std::cout << "\nnewCC: ";
                mCSDFProb.printCallContext(std::cout,newCC);
              }

              OA_ptr<DataFlowSet> dfSet;
              dfSet = currOutMap->getDFS(newCC);
              //assert((!dfSet.ptrEqual(0)) && 
              //       "entry_node newCC lookup failure in outSet");
              if (dfSet.ptrEqual(0)) {
                continue;
              }
              //dfSet clone needed here.  ?needed for inCC?
              inMap->assign(inCC->clone(), dfSet->clone()); 
            }
            mCallOutSetMap[call] = inMap;
            if (debug) {
              std::cout << "For call on Edge #" << succEdge->getId()
                        << " STORING ";
              OA_ptr<CC2DFSMapIterator> defIter;
              defIter = new CC2DFSMapIterator(*inMap);
              for (; defIter->isValid(); ++(*defIter)) {
                CCDFSPair ccdfsPair = defIter->current();
                OA_ptr<Alias::CallContext> cc;
                cc = ccdfsPair.getContextPtr();
                mCSDFProb.printCallContext(std::cout, cc);
                mCSDFProb.printDFSet(std::cout,ccdfsPair.getDFSetPtr());
              }
            }

          }
        } // end of if ENTRY_NODE

	mapIter = new CC2DFSMapIterator(*currOutMap);
	for ( ; mapIter->isValid(); ++(*mapIter)) {
          CCDFSPair cdPair = mapIter->current();
	  OA_ptr<Alias::CallContext> outCC;
	  outCC = cdPair.getContextPtr();
          if (debug) {
            std::cout << "for CC: ";
            mCSDFProb.printCallContext(std::cout,outCC);
            std::cout << "\n";
          }

	  OA_ptr<DataFlowSet> outSet;
	  outSet = cdPair.getDFSetPtr();

	  OA_ptr<DataFlowSet> currIn; 
	  OA_ptr<DataFlowSet> outClone = outSet->clone();

        
	  // if it is an exit node call transfer method for exit nodes
	  if (iNode->getType()==ICFG::EXIT_NODE) {
            currIn = mCSDFProb.exitTransfer(iNode->getProc(), outClone, outCC);

	    // otherwise it is a normal node and should apply regular transfer
	  } else {
            if(iNode->getType()==ICFG::CALL_NODE) {
              OA::CallHandle call = iNode->getCall();
              currIn = mCSDFProb.transfer(iNode->getProc(), outClone, outCC, call);
            } else {
              currIn = outClone;
              // loop through statements in reverse order
              OA_ptr<CFG::NodeStatementsRevIteratorInterface> stmtIterPtr 
                = iNode->getNodeStatementsRevIterator();
              if (debug) {
                std::cout << "looping through statements in reverse order:\n";
              }
              for (; stmtIterPtr->isValid(); ++(*stmtIterPtr)) {
                OA::StmtHandle stmt = stmtIterPtr->current();
                currIn = mCSDFProb.transfer(iNode->getProc(), currIn, outCC, stmt);
              }
            }
	  }
          
	  currInMap->assign(outCC, currIn);

	}

        // check to see if changed from last time
        if (currInMap != mNodeInSetMap[iNode] ) {
          changed = true;
          mNodeInSetMap[iNode] = currInMap;
        }
      }

    }
   
    if (debug) {
      std::cout << "ICFGCSDFSolver::atDGraphNode: changed = " << changed 
                << std::endl;
    }
    return changed;
}
  
//--------------------------------------------------------
// finalization upcalls
//--------------------------------------------------------
void 
ICFGCSDFSolver::finalizeNode(OA_ptr<DGraph::NodeInterface> node)
{
}



//--------------------------------------------------------
// debugging upcall 
//--------------------------------------------------------
void ICFGCSDFSolver::dump(std::ostream& os, OA_ptr<IRHandlesIRInterface> ir)
{
    // iterate over all entries in mNodeInSetMap and mNodeOutSetMap
    std::map<OA_ptr<ICFG::NodeInterface>, OA_ptr<CC2DFSMap> >::iterator mapIter;
    for (mapIter=mNodeInSetMap.begin(); mapIter!=mNodeInSetMap.end(); mapIter++)
    {
        OA_ptr<CC2DFSMap> dfset = mapIter->second;
        os << "Node (" << mapIter->first << ") In: ";
        if (dfset.ptrEqual(0)) {
          os << "[ NULL CC2DFSMap ]";
        } else {
          dfset->dump(os,ir);
        }
        os << std::endl;
    }
    for (mapIter=mNodeOutSetMap.begin(); mapIter!=mNodeOutSetMap.end(); mapIter++)
    {
        OA_ptr<CC2DFSMap> dfset = mapIter->second;
        os << "Node (" << mapIter->first << ") Out: ";
        if (dfset.ptrEqual(0)) {
          os << "[ NULL CC2DFSMap ]";
        } else {
          dfset->dump(os,ir);
        }
        os << std::endl;
    }
}

//------------------------------------------------------------
// private helper functions
//------------------------------------------------------------

// assumes that the CallContexts within map2 are within map1
// that is, map2 is contained within map1 wrt CallContext's
OA_ptr<CC2DFSMap> ICFGCSDFSolver::doMeet(OA_ptr<CC2DFSMap> map1, 
                                         OA_ptr<CC2DFSMap> map2,
                                         OA_ptr<ICFG::NodeInterface> iNode)
{
  if (debug) {
    std::cout << "ICFGCSDFSolver::doMeet\n";
    std::cout << "\t";
    if (map1.ptrEqual(0)) {
      std::cout << "map1 NULL\n";
    } else {
      std::cout << "map1 size " << map1->size() << std::endl;
    }
    std::cout << "\t";
    if (map2.ptrEqual(0)) {
      std::cout << "map2 NULL\n";
    } else {
      std::cout << "map2 size " << map2->size() << std::endl;
    }

  }

  OA_ptr<CC2DFSMap> retval;
  retval = new CC2DFSMap;

  OA_ptr<Alias::CallContext> cc;
  OA_ptr<DataFlowSet> dfs1;
  OA_ptr<DataFlowSet> dfs2;
  OA_ptr<DataFlowSet> resDFS;

  OA_ptr<CC2DFSMapIterator> mIter;
  mIter = new CC2DFSMapIterator(*map1);

  for ( ; mIter->isValid(); ++(*mIter)) {
    CCDFSPair cdPair = mIter->current();
    cc = cdPair.getContextPtr();
    dfs1 = cdPair.getDFSetPtr();;
    dfs2 = map2->getDFS(cc);

    // if there is no dfs for cc in map2, call problem's init routine
    if (dfs2.ptrEqual(0)) {
      if (mDirection==Forward) {
	dfs2 = mCSDFProb.initializeNodeIN(iNode,cc);
      } else {
	dfs2 = mCSDFProb.initializeNodeOUT(iNode,cc);
      }
    }

    resDFS = mCSDFProb.meet(dfs1,dfs2, cc);
    retval->assign(cc, resDFS);
  }

  // Now handle every cc in map2 that is not already in map1
  mIter = new CC2DFSMapIterator(*map2);
  for ( ; mIter->isValid(); ++(*mIter)) {
    CCDFSPair cdPair = mIter->current();
    cc = cdPair.getContextPtr();
    dfs2 = cdPair.getDFSetPtr();
    
    // if we already have an entry in retval, continue
    resDFS = retval->getDFS(cc);
    if (!resDFS.ptrEqual(0)) {
      continue;
    }

    // we know there is no dfs in map1 for this cc, so get initial dfs from prob    
    if (mDirection==Forward) {
      dfs1 = mCSDFProb.initializeNodeIN(iNode,cc);
    } else {
      dfs1 = mCSDFProb.initializeNodeOUT(iNode,cc);
    }

    resDFS = mCSDFProb.meet(dfs2,dfs1, cc);
    retval->assign(cc, resDFS);
  }

  return retval;

}

  } // end of DataFlow namespace
}  // end of OA namespace

