/*! \file
  
  \brief Holds a set of CCDFPairs

  \authors Michelle Strout, Barbara Kreaseck
  \version $Id: ReachConstsStandard.hpp,v 1.9 2005/06/10 02:32:05 mstrout Exp $

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#ifndef CCDFSPairSet_hpp
#define CCDFSPairSet_hpp

#include <cassert>
#include <iostream>
#include <set>
#include <OpenAnalysis/IRInterface/IRHandles.hpp>
#include <OpenAnalysis/Utils/OA_ptr.hpp>
#include <OpenAnalysis/DataFlow/DataFlowSet.hpp>
#include <OpenAnalysis/OABase/Annotation.hpp>
#include <OpenAnalysis/Alias/Interface.hpp>

#include "CCDFSPair.hpp"

namespace OA {
  namespace DataFlow {


class CCDFSPairSet;
class CCDFSPairSetIterator;

/*!Holds a set of CCDFSPair.  This is only designed for
 *!ICFGCSDFSolver to use.
 */

class CCDFSPairSet  : public virtual Annotation
{

public:

    //! ===== construction =====
    OA_ptr<CCDFSPairSet> clone() const;


    //! ===== access =====
    void clear() ;
    int  size() const ;
    bool isEmpty() const ;
    bool containsPairCC(OA_ptr<Alias::CallContext>) const ;


    //! ===== output =====
    void output(IRHandlesIRInterface& ir) const;
    void dump(std::ostream &os, OA_ptr<IRHandlesIRInterface> pIR);




    //! ==========================================================
    //! CCDFSPairSet Methods 
    //! ==========================================================


    //! ===== construction =====

    //! non-universal empty set
    CCDFSPairSet();

    //! copy constructor
    CCDFSPairSet(const CCDFSPairSet& other);

    //! assignment operator
    CCDFSPairSet& operator= (const CCDFSPairSet& other);


    //! ===== destruction =====
    ~CCDFSPairSet();

    //! ===== Modification routines by CCDFSPair ====
    void insert(CCDFSPair); // copies, does not clone
    void remove(CCDFSPair);

    //! ===== Needed Set Operations =====
  
    //! intersects the argument with this set, store results in this set
    CCDFSPairSet& intersectEqu(CCDFSPairSet& other);
    //! unions the argument with this set, store results in this set
    CCDFSPairSet& unionEqu(CCDFSPairSet& other);
    //! subtracts the argument from this set, storing results in this set
    CCDFSPairSet& minusEqu(CCDFSPairSet& other);

    //! =============== relationship ====================
    bool operator==(const CCDFSPairSet& other) const;
    bool operator!=(const CCDFSPairSet& other) const;

    //! ================ Output & Debugging =========================
    void output(IRHandlesIRInterface& ir, Alias::Interface &alias) const;


protected:

    //! ===== Members =====  
    OA_ptr<std::set<CCDFSPair> > mSet;

    //! ===== Friends =====
    friend class CCDFSPairSetIterator;
};
    

 





 
//! ==============================================
//! CCDFSPairSet Iterator
//! ==============================================

class CCDFSPairSetIterator {

public:

  //! ===== Constructor/Destructor =====
  CCDFSPairSetIterator (CCDFSPairSet& ccdfsSet);
  ~CCDFSPairSetIterator ();


  //! ===== Access =====
  void operator ++ ();
  bool isValid();
  CCDFSPair current();
  void reset();


private:
  OA_ptr<std::set<CCDFSPair> > mSet;
  OA_ptr<std::set<CCDFSPair>::iterator> mIter;
};



  } // end of DataFlow namespace
} // end of OA namespace




#endif // CCDFSPairSet_hpp
