/*! \file
  
  \brief Holds a map of Call Contexts to DataFlowSets

  \authors Michelle Strout, Barbara Kreaseck
  \version $Id: ReachConstsStandard.hpp,v 1.9 2005/06/10 02:32:05 mstrout Exp $

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#ifndef CC2DFSMap_hpp
#define CC2DFSMap_hpp

#include <cassert>
#include <iostream>
#include <map>
#include <OpenAnalysis/IRInterface/IRHandles.hpp>
#include <OpenAnalysis/Utils/OA_ptr.hpp>
#include <OpenAnalysis/DataFlow/DataFlowSet.hpp>
#include <OpenAnalysis/OABase/Annotation.hpp>
#include <OpenAnalysis/Alias/Interface.hpp>
#include <OpenAnalysis/DataFlow/CCDFSPair.hpp>


namespace OA {
  namespace DataFlow {


class CC2DFSMap;
class CC2DFSMapIterator;

/*!Holds a map of CallContext to DataFlowSets.  This is only designed for
 *!ICFGCSDFSolver to use.
 */

class CC2DFSMap  : public virtual Annotation
{

public:

    //! ==========================================================
    //! CC2DFSMap Methods 
    //! ==========================================================


    //! ===== construction =====

    //! Creates empty map
    CC2DFSMap();

    //! copy constructor
    CC2DFSMap(const CC2DFSMap& other);

    //! ===== destruction =====
    ~CC2DFSMap();

    //! clone routine
    OA_ptr<CC2DFSMap> clone() const;

    //! assignment operator (copies, does not clone)
    CC2DFSMap& operator= (const CC2DFSMap& other);


    //! ===== access =====
    void clear() ;
    int  size() const ;
    bool isEmpty() const ;
    OA_ptr<DataFlowSet> getDFS(OA_ptr<Alias::CallContext> cc) const ;




    // ==================================================
    //   Modification routines 
    // ==================================================

    //! create a mapping for this cc
    //! (assign just copies, does not clone)
    void assign(OA_ptr<Alias::CallContext> cc, OA_ptr<DataFlowSet> dfs);

    //! remove mapping for this cc
    void remove(OA_ptr<Alias::CallContext> cc);


    //! =============== relationship ====================
    bool operator==(const CC2DFSMap& other) const;
    bool operator!=(const CC2DFSMap& other) const;
    bool operator< (const CC2DFSMap& other) const;

    //! ================ Output & Debugging =========================
    void output(IRHandlesIRInterface& ir, Alias::Interface &alias) const;
    void output(IRHandlesIRInterface& ir) const;
    void dump(std::ostream &os, OA_ptr<IRHandlesIRInterface> pIR);


protected:

    //! ===== Members =====  
    OA_ptr<std::map<OA_ptr<Alias::CallContext>, OA_ptr<DataFlowSet> > > mMap;

    //! ===== Friends =====
    friend class CC2DFSMapIterator;
};
    

 





 
//! ==============================================
//! CC2DFSMap Iterator
//! ==============================================

class CC2DFSMapIterator {

public:

  //! ===== Constructor/Destructor =====
  CC2DFSMapIterator (const CC2DFSMap& cc2dfsMap);
  ~CC2DFSMapIterator ();


  //! ===== Access =====
  void operator ++ ();
  bool isValid();
  CCDFSPair current();
  void reset();


private:
  OA_ptr<CC2DFSMap> mcc2dfsMap;
  std::map<OA_ptr<Alias::CallContext>, OA_ptr<DataFlowSet> >::iterator mIter;
};



  } // end of DataFlow namespace
} // end of OA namespace




#endif // CC2DFSMap_hpp
