/*! \file
  
  \brief The AnnotationManager that generates CCSetPerProc

  \authors Barbara Kreaseck

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#include "ManagerCallContexts.hpp"
#include <Utils/Util.hpp>


namespace OA {
  namespace Alias {

static bool debug = false;

ManagerCallContexts::ManagerCallContexts(
    OA_ptr<IRHandlesIRInterface> _ir) : mIR(_ir)
{
    OA_DEBUG_CTRL_MACRO("DEBUG_ManagerCallContexts:ALL", debug);
}

OA_ptr<CCSetPerProc> 
ManagerCallContexts::performAnalysis(OA_ptr<CallGraph::CallGraphInterface> callGraph )
{
  // empty set of parameter bindings that we are going to fill
  OA_ptr<CCSetPerProc> retval;
  retval = new CCSetPerProc;

  if (debug) {
      std::cout << "In ManagerCallContexts::performAnalysis" << std::endl;
  }

  // we really want to start a cascade from the roots of the CallGraph
  // As we generate the CallContexts for a CallGraphNode (i.e., proc), we 
  // can add the CallGraphNodes to the work queue.
  // When we take a CallGraphNode off of the work queue, we can check to see
  // whether all of it's caller nodes have already had their CallContexts created.
  // if not, we can put them back on the queue (or check that the caller nodes missing
  // are on the work queue).
  
  std::list<OA_ptr<CallGraph::NodeInterface> > workQ;
  std::map<OA_ptr<CallGraph::NodeInterface>, bool> nodeDone;

  OA_ptr<CallGraph::NodesIteratorInterface> cNodeIter;
  OA_ptr<CallGraph::NodeInterface> cNode;
  OA_ptr<CallContextSet> ccSetP;
  OA_ptr<CallGraph::EdgesIteratorInterface> ceIter;
  OA_ptr<CallGraph::EdgeInterface> cEdge;

  // Iterate over all the CallGraphNodes in the CallGraph, place roots on workQ
  cNodeIter = callGraph->getCallGraphNodesIterator();
  for (; cNodeIter->isValid(); ++(*cNodeIter)) { 
    cNode = cNodeIter->currentCallGraphNode();
    if (cNode->isCalled()) {
      nodeDone[cNode] = false;
    } else {
      if (debug) {
        std::cout << "Root proc: " << mIR->toString(cNode->getProc()) << "\n";
      }

      // this is a root node, give it CallHandle(0)
      ccSetP = new CallContextSet();
      OA_ptr<CallContext> emptyContext;
      emptyContext = new CallContext(CallHandle(0));
      ccSetP->insert(emptyContext);
      mNodeToSetMap[cNode] = ccSetP;
      nodeDone[cNode] = true;

      // enqueue it's children
      ceIter = cNode->getCallGraphOutgoingEdgesIterator();      
      for (; ceIter->isValid(); ++(*ceIter) ) {
        cEdge = ceIter->currentCallGraphEdge();
        if (debug) {
          OA_ptr<CallGraph::NodeInterface> callee;
          callee = cEdge->getCallGraphSink();
          std::cout << "Enque_ing proc: " << mIR->toString(callee->getProc()) << "\n";
        }
        workQ.push_back(cEdge->getCallGraphSink());
      }
    }
  }

  // process each node whose in-edge-sink-nodes are nodeDone, recyle otherwise
  while (!workQ.empty()) {
    cNode = workQ.front();
    workQ.pop_front();

    if (nodeDone[cNode] == true) {
      continue;
    }

    if (debug) {
      std::cout << "Processing proc: " << mIR->toString(cNode->getProc()) << "\n";
    }
      
    // check all in-edge-sink-nodes for doneness
    bool allIncomingDone = true;
    ceIter = cNode->getCallGraphIncomingEdgesIterator();
    for (; ceIter->isValid() && allIncomingDone ; ++(*ceIter) ) {
        cEdge = ceIter->currentCallGraphEdge();
        OA_ptr<CallGraph::NodeInterface> caller;        
        caller = cEdge->getCallGraphSource();
        if (nodeDone[caller] == false) {
          allIncomingDone = false;
        }
    }

    if (!allIncomingDone) {
      if (debug) {
        std::cout << "NOT allIncomingDone\n";
      }
      // requeue cNode // why not wait for unDone parent to enqueue it?
      //      workQ.push_back(cNode);
      continue;
    } else {
      if (debug) {
        std::cout << "allIncomingDone\n";
      }
    }

    if (debug) {
      std::cout << "Processing in-coming edges\n";
    }
    // create CallContextSet for cNode from incomingEdges
    ccSetP = new CallContextSet();
    ceIter->reset();
    for (; ceIter->isValid() && allIncomingDone ; ++(*ceIter) ) {
        cEdge = ceIter->currentCallGraphEdge();
        if (debug) {
          cEdge->output(*mIR);
          std::cout << std::endl;
        }
        CallHandle call = cEdge->getCallHandle();
        OA_ptr<CallGraph::NodeInterface> caller;        
        caller = cEdge->getCallGraphSource();

        // force append the call to each callcontext of the caller to create
        // call contexts for cNode
        OA_ptr<CallContextSetIterator> callerSetIter;
        callerSetIter = new CallContextSetIterator(*(mNodeToSetMap[caller]));
        for (; callerSetIter->isValid(); ++(*callerSetIter)) {
          OA_ptr<CallContext> callerCC;
          callerCC = callerSetIter->current();
          OA_ptr<CallContext> newCC;
          newCC = callerCC->clone();
          newCC->appendForce(call);
          ccSetP->insert(newCC);
        }
    }

    // record this ccset for cNode
    mNodeToSetMap[cNode] = ccSetP;
    nodeDone[cNode] = true;

    if (debug) {
      std::cout << "Processing out-going edges\n";
    }

    // enqueue all children that are not already done
    ceIter = cNode->getCallGraphOutgoingEdgesIterator();      
    for (; ceIter->isValid(); ++(*ceIter) ) {
      cEdge = ceIter->currentCallGraphEdge();
      if (debug) {
        cEdge->output(*mIR);
      }
      OA_ptr<CallGraph::NodeInterface> callee;
      callee = cEdge->getCallGraphSink();
      if (nodeDone[callee] == false) {
        if (debug) {
          std::cout << "*** ENQUEUED ***\n";
        }
        workQ.push_back(callee);
      }
    }
  }

  if (debug) {
    std::cout << "\n&*&*&*&* FILLING RETVAL &*&*&*&*\n\n";
  }

  // place ccSets into retval
  int numCCs = 0;
  cNodeIter->reset();
  for (; cNodeIter->isValid(); ++(*cNodeIter)) { 
    cNode = cNodeIter->currentCallGraphNode();
    if (debug) {
      cNode->output(*mIR);
      if (nodeDone[cNode] == true) {
        std::cout << "\tDONE\n\n";
      } else {
        std::cout << "\tNOT DONE\n\n";
      }
      std::cout.flush();
    }
    assert("Missing CallContextSet for CallGraph Node\n" 
           && (nodeDone[cNode]==true));
    ProcHandle proc = cNode->getProc();
    ccSetP = mNodeToSetMap[cNode];
    numCCs += ccSetP->size();
    retval->recordCallContextSet(proc, ccSetP);
  }

  retval->setNumCCs(numCCs);

  return retval;

}


  } // end of namespace Alias
} // end of namespace OA
