/*! \file
  
  \brief Declarations of the AnnotationManager that implements CSFIAlias.

  \authors Michelle Strout, Brian White, Barbara Kreaseck

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#ifndef ManagerCSFIAlias_H
#define ManagerCSFIAlias_H

//--------------------------------------------------------------------
#include <cassert>
#include <map>
#include <set>

// OpenAnalysis headers
#include <OpenAnalysis/IRInterface/AliasIRInterface.hpp>
#include <OpenAnalysis/Utils/UnionFindUniverse.hpp>
#include <OpenAnalysis/MemRefExpr/MemRefExprVisitor.hpp>
#include <OpenAnalysis/Alias/CallContext.hpp>
#include <OpenAnalysis/Alias/CCSetPerProc.hpp>
#include <OpenAnalysis/Alias/MREwContext.hpp>
#include <OpenAnalysis/Alias/MRHwContext.hpp>
#include <OpenAnalysis/Alias/Interface.hpp>
#include <OpenAnalysis/CallGraph/CallGraphInterface.hpp>
#include <OpenAnalysis/Alias/OuterRefOpVisitor.hpp>

// for extra functionality outside of ManagerFIAlias class
#include <OpenAnalysis/Alias/ManagerFIAlias.hpp>

namespace OA {
  namespace Alias {


/*! 
    Creates a union-find universe holding sets of aliased <MRE, CallingContext> .
    Uses the FIAlias algorithm described in the Ryder2001 journal paper.
    Note that ManagerCSFIAlias does not satisfy the Alias::Interface.hpp
    interface.  Returning a concrete representation of alias
    relations (via performAnalysis) is now left to subclasses 
    of ManagerCSFIAlias, such as ManagerCSFIAliasAliasTag.
*/
class ManagerCSFIAlias 
{ 
public:
  ManagerCSFIAlias(OA_ptr<AliasIRInterface> _ir);
  ~ManagerCSFIAlias () {}

  OA_ptr<UnionFindUniverse>
  performCSFIAlias(  OA_ptr<CallGraph::CallGraphInterface> callGraph,
		   // OA_ptr<IRProcIterator> procIter,
                  FIAliasImplement implement = ALL_PROCS );

  OA_ptr<IRProcIterator> getAnalyzedProcIter();

protected:
  OA_ptr<AliasIRInterface> mIR;

  // mCCResults gets initialized in ManagerCSFIAliasAliasTag::performAnalysis()
  OA_ptr<Alias::CCSetPerProc> mCCResults;

  //===== helper datastructures and routines
  // mapping of MemRefExpr/Context to unique ids
  std::map<MREwContext,int> mMREwCToID;
  // use a counter to assign unique ids to each MemRefExprWithContext
  int mCount;

  // this will be the default value for this
  // for each partition
  //    map[part][memrefwrapper] = 0;  
  std::map<int,std::map<OA_ptr<MemRefExpr>,int> > mMap;

  // mapping of MemRefExpr to set of MemRefHandles that can
  // be expressed by said MemRefExpr
  //std::map<OA_ptr<MemRefExpr>,std::set<MemRefHandle> > mMREToMemRefHandles;
  std::map<MREwContext,std::set<MemRefHandle> > mMREwCToMemRefHandles;


  // mapping of MemRefExpr to procedures it shows up in
  //std::map<OA_ptr<MemRefExpr>,std::set<ProcHandle> > mMREToProcs;
  std::map<MREwContext,std::set<ProcHandle> > mMREwCToProcs;

  // mapping of MemRefHandles to procedures, each MemRefHandle is only in
  // one procedure
  std::map<MemRefHandle,ProcHandle> mMemRefHandleToProc;

  //! set of formal symbols associated with each procedure
  std::map<ProcHandle,std::set<SymHandle> > mProcToFormalSet;

  //! mapping of callnodes to callcontexts
  //std::map<OA_ptr<CallGraph::NodeInterface>, 
  //	   std::set<OA_ptr<CallContext> > > mCNodeToContextSet;


  //! maps mre to the given proc and memref, also assigns mre a unique ID
  MREwContext recordMREwC( OA_ptr<MemRefExpr> mre, OA_ptr<CallContext> cc, 
                           ProcHandle proc, MemRefHandle memref);
  MREwContext recordMREwC( OA_ptr<MemRefExpr> mre, OA_ptr<CallContext> cc,
                           ProcHandle proc );

  OA_ptr<MemRefExpr> createDeref( OA_ptr<MemRefExpr> mre );

  std::set<MREwContext> 
    allMemRefExprsInSameSet( MREwContext mrewc, 
                             OA_ptr<UnionFindUniverse> ufset);

  //! records all memory reference expressions in a given procedure
  void initMemRefExprs( OA_ptr<CallGraph::NodeInterface> cNode );

  //! records all memory reference expressions in all procedures
  void initMemRefExprs( OA_ptr<CallGraph::CallGraphInterface> callGraph );

  //! increment versions of FIAlias use a worklist.  this does not, so this is a noop.
  void addProcToWorkList(ProcHandle proc);

  //! perform Ryder phase 1 on stmt.
  void doPhase1Iteration(StmtHandle stmt, 
			 OA_ptr<CallGraph::NodeInterface> cnode, 
			 OA_ptr<UnionFindUniverse> ufset);

  //! perform Ryder phase 2
  bool 
  doPhase2Iteration(OA_ptr<UnionFindUniverse> ufset, 
		    std::map<int,std::map<OA_ptr<MemRefExpr>,int> > & map);

  //! perform Ryder phase 3 on the callsite call invoked from caller cnode.
  bool doPhase3Iteration(CallHandle call,
			 OA_ptr<CallGraph::NodeInterface> callernode,
			 OA_ptr<CallGraph::NodeInterface> calleenode,
			 OA_ptr<UnionFindUniverse> ufset, 
			 std::map<int,std::map<OA_ptr<MemRefExpr>,int> > & map);

  //! A set of reachable procs that have been analyzed.
  std::set<ProcHandle> mAnalyzedProcs;

  //! Perform FIAlias using Ryder's original algorithm that analyzes all procs.
  OA_ptr<UnionFindUniverse>
  performCSFIAliasAllProcs(OA_ptr<CallGraph::CallGraphInterface> cgraph);

  //! Perform FIAlias using a modification that only visits reachable procs.
  OA_ptr<UnionFindUniverse>
  performCSFIAliasReachableProcs( OA_ptr<CallGraph::CallGraphInterface> cgraph);

  friend class RecordMREwCsVisitor;
  friend class AnalyzedProcIterator;

private: // helper functions
  void outputMREwCsInSet(int setID, 
        OA_ptr<UnionFindUniverse> ufset, 
        std::map<int,std::map<OA_ptr<MemRefExpr>,int> > & map  );

  //! merger routine as defined in Ryder paper
  void merge(int part1, int part2, OA_ptr<UnionFindUniverse> ufset, 
           std::map<int,std::map<OA_ptr<MemRefExpr>,int> > & map  );

  //! The list of procedures to be analyzed.
  std::set<ProcHandle> mWorklist;

  //! The flavor of FIAlias--examine all procs or only reachable procs.
  FIAliasImplement mImplement;
};


  } // end of Alias namespace
} // end of OA namespace

#endif
