/*! \file
  
  \brief The AnnotationManager that generates union-find sets of MREs, where
         each MRE in the set may alias other MREs in the set.

  \authors Michelle Strout, Brian White, Barbara Kreaseck

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#include "ManagerCSFIAlias.hpp"
#include <Utils/Util.hpp>

namespace OA {
  namespace Alias {
    
static bool debug = false;
static bool debugStats = false;

/*!
   Visitor over memory reference expressions that determines
   whether this MRE needs context.
 */

class MRENeedsContextQVisitor : public virtual MemRefExprVisitor {
public:
  MRENeedsContextQVisitor()
  {  
    mNeedsContext = false; // conservative???
  }
  ~MRENeedsContextQVisitor() {}

  bool getNeedsContext() { return mNeedsContext; }
  void reset() { mNeedsContext = false; }
  
  void visitNamedRef(NamedRef& ref) { 
    if (ref.isStrictlyLocal()) {
      mNeedsContext = true; // without this, strictly locals act as globals (or static variables)
    } else {
      mNeedsContext = false; // globals?
    }
  }

  void visitUnnamedRef(UnnamedRef& ref) {
    mNeedsContext = true;
  }
  void visitUnknownRef(UnknownRef& ref) {
    mNeedsContext = true;
  }
  void visitAddressOf(AddressOf& ref) {
    // do not record the addressOf, but record its child
    OA_ptr<MemRefExpr> mre = (ref.getMemRefExpr())->clone();
    if (!mre.ptrEqual(0)) { // BK: when does an AddressOf have a NULL getMemRefExpr()??
      mre->acceptVisitor(*this); 
    }
  }
  void visitDeref(Deref& ref) { 
    mNeedsContext = true;
  }
  // default handling of more specific SubSet specificiations
  void visitSubSetRef(SubSetRef& ref) {
    // child memory reference will determine
    OA_ptr<MemRefExpr> mre = ref.getMemRefExpr();
    if (!mre.ptrEqual(0)) { // BK: when does a SubSetRef have a NULL getMemRefExpr()??
      mre->acceptVisitor(*this); 
    }
  }
private:
  bool mNeedsContext;
};


OA_ptr<IRProcIterator> ManagerCSFIAlias::getAnalyzedProcIter()
{
  OA_ptr<AnalyzedProcIterator> iterator;
  iterator = new AnalyzedProcIterator(mAnalyzedProcs);
  return iterator;
}

/*!
*/
ManagerCSFIAlias::ManagerCSFIAlias( OA_ptr<AliasIRInterface> _ir) : mIR(_ir),
    mCount(1)
{
    OA_DEBUG_CTRL_MACRO("DEBUG_ManagerCSFIAlias:ALL", debug);
    OA_DEBUG_CTRL_MACRO("DEBUG_ManagerCSFIAlias_Stats:ALL", debugStats);
}

//!  Only incremental versions of FIAlias (e.g., FIAliasReachable)
//!  use a worklist.  Therefore, this is a noop by default.
void ManagerCSFIAlias::addProcToWorkList(ProcHandle proc)
{
  if ( mImplement == REACHABLE_PROCS ) {
      if ( mAnalyzedProcs.find(proc) != 
           mAnalyzedProcs.end() ) {
          return;
      }

      if ( mWorklist.find(proc) != 
           mWorklist.end() ) {
          return;
      }

      mWorklist.insert(proc);
  }
}


void
ManagerCSFIAlias::doPhase1Iteration(StmtHandle stmt,
				    OA_ptr<CallGraph::NodeInterface> cnode,
				    OA_ptr<UnionFindUniverse> ufset)
{
  // Perform Ryder Phase 1 on each statement for each callcontext
  
  ProcHandle currProc = cnode->getProc();
  
  // set-up iterator over callcontext set for this callnode
  OA_ptr<CallContextSetIterator> ccIter;
  ccIter = mCCResults->getCallContextSet(currProc);
  
  // for each ptrassignpair
  OA_ptr<PtrAssignPairStmtIterator> pairIter
    = mIR->getPtrAssignStmtPairIterator(stmt);
  for ( ; pairIter->isValid(); (*pairIter)++ ) {
    
    OA_ptr<MemRefExpr> target = pairIter->currentTarget();        
    OA_ptr<MemRefExpr> source = pairIter->currentSource();        
    
    // map *target and *source to an id if they don't already have one
    OA_ptr<MemRefExpr> sourceDeref, targetDeref;
    
    // make a deref versions of target and source mre
    targetDeref = createDeref( target );
    sourceDeref = createDeref( source );
    
    // for each CallContext for this CallNode
    ccIter->reset();
    for (; ccIter->isValid(); ++(*ccIter)) {
      OA_ptr<CallContext> cctext = ccIter->current();

      MREwContext targetMREwC = recordMREwC(targetDeref, cctext, currProc);
      MREwContext sourceMREwC = recordMREwC(sourceDeref, cctext, currProc);
      
      if (debug) {
        std::cout << "\tsourceDerefMREwContext = ";
        sourceMREwC.output(*mIR);
        std::cout << "\ttargetDerefMREwContext = ";
        targetMREwC.output(*mIR);
        std::cout << "\t===> Union ( "
                  << ufset->Find(mMREwCToID[targetMREwC]) << ", into "
                  << ufset->Find(mMREwCToID[sourceMREwC]) << " )"
                  << std::endl;
      }
      
      // then union the sets with *target and *source 
      ufset->Union( ufset->Find(mMREwCToID[targetMREwC]),
                    ufset->Find(mMREwCToID[sourceMREwC]),
                    ufset->Find(mMREwCToID[sourceMREwC]) );
      
    } // iteration over contexts 
    
  } // iteration over ptr assign pairs
  
}

/*! Generates aliases due to secondary effects.  For example, if *x and y
 *  alias, then so should **x and *y.
 *  Returns true if there has been a change.
 */
bool
ManagerCSFIAlias::doPhase2Iteration(OA_ptr<UnionFindUniverse> ufset,
                                    std::map<int,std::map<OA_ptr<MemRefExpr>,int> > & map)
{
  bool changed = false;
  
  // for each memref
  std::map<MREwContext,int>::iterator mreMapIter;
  for (mreMapIter=mMREwCToID.begin(); 
       mreMapIter!=mMREwCToID.end(); mreMapIter++ ) 
    {
      MREwContext mrewc = mreMapIter->first;
      OA_ptr<MemRefExpr> mre = mrewc.getMRE();
      OA_ptr<CallContext> cctext = mrewc.getCallContext();
      
      if (debug) {
        std::cout << std::endl << "\tmrewc = ";
        mrewc.output(*mIR);
        std::cout << "\t\tFind(mMREwCToID[mrewc]) = " 
                  << ufset->Find(mMREwCToID[mrewc]) << std::endl;
      }
      
      // only do the following for real MREs, if the MRE is part of
      // an addressOf operation then the outermost MRE doesn't involve
      // any memory being accessed
      if(mre->isaRefOp()) {
        OA_ptr<RefOp> refop = mre.convert<RefOp>();
        if(!refop->isaAddressOf()) {
          OuterRefOpVisitor outerRefVisitor;
          mre->acceptVisitor(outerRefVisitor);
          OA_ptr<RefOp> justRefop = outerRefVisitor.getOuterRefOp();
          OA_ptr<MemRefExpr> innerMRE = refop->getMemRefExpr();
          
          OA_ptr<CallContext> emptyContext; emptyContext = new CallContext();
          MRENeedsContextQVisitor ncVisitor;
          OA_ptr<CallContext> useContext;

          innerMRE->acceptVisitor(ncVisitor);
          if (ncVisitor.getNeedsContext()) {
            useContext = cctext;
          } else {
            useContext = emptyContext;
          }

          MREwContext innerMREwC(innerMRE,useContext);
          MREwContext justRefopwC(justRefop,cctext);
          
          if (debug) {
            std::cout << "\t\tinnerMRE = ";
            if (! innerMRE.ptrEqual(0) ) {
              innerMRE->output(*mIR);
            } else { 
              std::cout << "<null MRE>" << std::endl;
            }
            std::cout << "\t\tFind( innerMREwC ) = "
                      << ufset->Find(mMREwCToID[innerMREwC])
                      << std::endl;
            std::cout << "\t\tjustRefop = ";
            if (! justRefop.ptrEqual(0) ) {
              justRefop->output(*mIR);
            } else { 
              std::cout << "<null MREF>" << std::endl;
            }
          }
          
          // if find(memref)!=find( map[find(innerMRE)][justRefop] )
          
          int setID = ufset->Find(map[ufset->Find(mMREwCToID[innerMREwC])][justRefop] );
          if (debug) {
            std::cout << "setID = " << setID << "\n";
          }
          if ( ufset->Find(mMREwCToID[mrewc]) != setID ) {
            changed = true;
            // if map[find(memref->getMemRef)][refop] is not in a part yet
            if (setID == 0) {
              if (debug) {
                std::cout << "setID = 0\n";
              }
              
              // then put it in the one for memref
              map[ufset->Find(mMREwCToID[innerMREwC])][justRefop]
                = ufset->Find(mMREwCToID[mrewc]);
              // else
            } else {
              merge(ufset->Find(mMREwCToID[mrewc]), setID, ufset, map);
            }
          }
        }// does not have address taken
      } // is a refop
    } // over mre
  return changed;
}
    

/*! Generates aliases due to parameter bindings at function calls.  More
 *  Can arise when secondary aliasing affects discover more possible aliasing
 *  involving function pointers.
 *  Returns true if there has been a change.
 */
bool
ManagerCSFIAlias::doPhase3Iteration(CallHandle call, 
				    OA_ptr<CallGraph::NodeInterface> callernode,
				    OA_ptr<CallGraph::NodeInterface> calleenode,
				    OA_ptr<UnionFindUniverse> ufset,
				    std::map<int,std::map<OA_ptr<MemRefExpr>,int> > & map)
{
  bool changed = false;
  
  ProcHandle callerProc = callernode->getProc();
  
  OA_ptr<CallContextSetIterator> ccIter;
  ccIter = mCCResults->getCallContextSet(callerProc);

  // set-up iterator over callcontext set for this callernode
  OA_ptr<CallContext> cctxt;
  cctxt = new CallContext(call);

  // get the mre for the function call (eg. NamedRef('foo'))
  OA_ptr<MemRefExpr> callMRE = mIR->getCallMemRefExpr(call);
  
  ccIter->reset();
  for (; ccIter->isValid(); ++(*ccIter)) {
    OA_ptr<CallContext> callerContext = ccIter->current();;
    MREwContext callMREwC(callMRE,callerContext);
  
    // will be set of all MREs that are aliased with this callMRE
    std::set<MREwContext> aliasedMREwCs;
  
    aliasedMREwCs.insert(callMREwC);
  
    // if it is a NamedRef then it can only be referring
    // to one procedure (assumption one-to-one mapping
    // between symbols for procedures and procedure handles
    // for procedures
    // If it isn't a nameRef then we must have a function ptr
    if (! callMRE->isaNamed()) {
      
      assert("CSFIAlias doesn't currently work with function ptrs\n" && false);
      
      // -- FIXME CSFIAlias is only designed for Fortran front-end, so
      //    not handling function ptrs yet.  If/when we try to handle function ptrs
      //    will need to figure out how to get the CallContexts for other 
      //    aliased MREwContext that represent aliases to the callMRE which matches
      //    the callernode. ??? not sure about this ... ???
      
      // gather all the MREs that are in the same set as the callMRE
      // NOTICE: only do this if is not a Named, NamedRefs for calls
      // aren't actually given an MREToID
      aliasedMREwCs  = allMemRefExprsInSameSet( callMREwC, ufset );
    }
  
  
    // Previously, we iterated over params and then over
    // aliases to procedures potentially invoked at the call site.
    // I am inverting this.  We visit potential aliases to
    // discover new static resolution of indirect calls.  For
    // each of these, newly discovered, we need to add it to 
    // the worklist.  Doing this in the opposite order would
    // not allow us to visit functions without params or without
    // pointer params that are only indirectly invoked.  BSW 10/27/06
    
    // Note that for non-incremental version of ManagerFIAlias
    // (i.e., those other than ManagerFIAliasReachable)
    // there is no concept of a worklist.  Hence addProcToWorkList
    // is a no-op.  Nevertheless, the above-mentioned loop
    // rearrangement does not cause any inefficiencies.
    
    // for all mres in the same set
    std::set<MREwContext>::iterator mrewcIter; 
    for (mrewcIter=aliasedMREwCs.begin(); mrewcIter!=aliasedMREwCs.end();
         mrewcIter++ )
      {
        MREwContext mrewC = *mrewcIter;
        OA_ptr<MemRefExpr> possCallMRE = mrewC.getMRE();
        OA_ptr<CallContext> callerContext = mrewC.getCallContext();
        
        // create callee context from caller context + call
        OA_ptr<CallContext> calleeContext = callerContext->clone();
        calleeContext->appendForce(call);
        
        OA_ptr<NamedRef> namedProc;
        
        if (debug) {
          std::cout << "Aliased call mre: ";
          possCallMRE->output(*mIR);
        }
        
        // if the mre is a NamedRef then
        if (possCallMRE->isaNamed()) {
          namedProc = possCallMRE.convert<NamedRef>();
          if (debug) {
            std::cout << " (Named)" << std::endl;
          }
        } else {
          if (debug) {
            std::cout << " (Unnamed)" << std::endl;
          }
          continue;
        }
        
        // get the symbol for the function
        SymHandle sym = namedProc->getSymHandle();
        
        // get the procedure handle for that symbol
        ProcHandle proc = mIR->getProcHandle(sym);
        
        // check that the symbol actually has an associated procedure
        if (proc==ProcHandle(0)) { continue; }
        
        // We have a statically resolvable function that may
        // be invoked at this callsite.  Add it to the
        // worklist to be analyzed, if it isn't already pending
        // and hasn't already been analyzed.
        addProcToWorkList(proc);
        
        // for each implicit ptrassignpair due to formal/actual binding
        OA_ptr<ParamBindPtrAssignIterator> pairIter
          = mIR->getParamBindPtrAssignIterator(call);
        for ( ; pairIter->isValid(); (*pairIter)++ ) {
          
          int formalId = pairIter->currentFormalId();
          OA_ptr<MemRefExpr> actualMRE = pairIter->currentActual();
          
          
          if (debug) {
            std::cout << "Call mre: ";
            callMRE->output(*mIR);
            std::cout << " formal id: " << formalId << std::endl;
          }
          
          // get the formal symbol for the param binding
          // we are handling
          SymHandle formalSym = mIR->getFormalSym(proc,formalId);
          // if the signature doesn't match then don't process
          if (formalSym==SymHandle(0)) { continue; }
          
          // make mre for the formal
          // FIXME: since already do this, could save time
          // by memoizing in a map[proc][formalID]
          // Create a namedRef for the formal.
          bool addressTaken = false;
          bool fullAccuracy = true;
          // DEF since formal = actual.
          MemRefExpr::MemRefType mrType = MemRefExpr::DEF;
          OA_ptr<MemRefExpr> formalmre; 

          // formal parameter NamedRefs are isLocal == true
          formalmre = new NamedRef(mrType, formalSym, true);
          
          // Thinking of Using new IRInterface::createNamedRef() below
          // FIXME???
          // formalmre will be MemRefExpr::USE
          // is this an error???

          // formalmre = mIR->createNamedRef(formalSym);

          //if (formalmre->isaNamed()) {
          //  OA_ptr<NamedRef> mre = formalmre.convert<NamedRef>();
          //  assert(mre->isStrictlyLocal());
          //} else {
          //  assert("formalmre is not a NamedRef" && false);
          //}
          
          // create a dereference for the formalref and actualref
          OA_ptr<MemRefExpr> targetDeref = createDeref( formalmre );
          OA_ptr<MemRefExpr> sourceDeref = createDeref( actualMRE );
          
          MREwContext sourceMREwC = recordMREwC(sourceDeref, callerContext,
                                                callerProc);
          MREwContext targetMREwC = recordMREwC(targetDeref, calleeContext,
                                                proc);

          if (debug) {
            std::cout << "ParamBindPtrAssign: ";
            std::cout << "\tsourceDerefMREwContext = ";
            sourceDeref->output(*mIR);
            std::cout << "\ttargetDerefMREwContext = ";
            targetDeref->output(*mIR);
            std::cout << "\t===> merge ( "
                      << ufset->Find(mMREwCToID[targetMREwC]) << ", into "
                      << ufset->Find(mMREwCToID[sourceMREwC]) << " )"
                      << std::endl;
          }
            
          // union the sets they are in
          if( ufset->Find(mMREwCToID[targetMREwC]) != 
              ufset->Find(mMREwCToID[sourceMREwC]) )  
            {
              changed=true;
              
              // then merge the sets for the target and source deref
              merge( ufset->Find(mMREwCToID[targetMREwC]),
                     ufset->Find(mMREwCToID[sourceMREwC]),
                     ufset, map);
            }
          
          
        } // paramBindPtrAssigns
        
      } // over possible procedures for function call
    
  } // for each Call Context for the CallerNode
  
  return changed;
}

OA_ptr<UnionFindUniverse>
ManagerCSFIAlias::performCSFIAlias( OA_ptr<CallGraph::CallGraphInterface> cGraph,
				    FIAliasImplement implement )
{
  OA_ptr<UnionFindUniverse> ufset;
  
  mImplement = implement;
  if ( mImplement == ALL_PROCS ) {
    ufset = performCSFIAliasAllProcs(cGraph);
  } else { 
    ufset = performCSFIAliasReachableProcs(cGraph);
  }
  
  if ( debug ) {
    OA_ptr<IRProcIterator> analyzedProcIter = getAnalyzedProcIter();
    int numAnalyzedProcs = 0;
    for(analyzedProcIter->reset(); analyzedProcIter->isValid(); ++(*analyzedProcIter)) {
      ++numAnalyzedProcs;
    }
    
    std::cout << "ManagerCSFIAlias::performCSFIAlias analyzed "
              << numAnalyzedProcs << " procs." << std::endl;
  }
  
  return ufset;
}

OA_ptr<UnionFindUniverse>
ManagerCSFIAlias::performCSFIAliasAllProcs( OA_ptr<CallGraph::CallGraphInterface> cgraph)
{
  if ( debug ) {
    std::cout << "performCSFIAliasAllProcs" << std::endl;
  }
  
  // ptr to union find datastructure
  OA_ptr<UnionFindUniverse> ufset;
  
  // map each MemRefExpr to a unique id and count them all
  initMemRefExprs(cgraph);
  if(debug) {
    std::cout << "Program has " << mCount << " MREwContexts " 
              << " with Context-level " << CallContext::getMaxDegree()
              << std::endl;
  }
  
  // declare the union-find datastructure
  // 1 for each memrefExpr we have seen so far and each one
  // could get a deref added in the worse case, the extra
  // 1 is for zero
  ufset = new UnionFindUniverse(2*mCount+1);
  
  // for all the stmts in the whole program
  // Iterate over all the procedures in the CallGraph
  
  // Iterate over all the CallGraphNodes in the CallGraph
  OA_ptr<CallGraph::NodesIteratorInterface> cNodeIter;
  cNodeIter = cgraph->getCallGraphNodesIterator();
  
  double time = clock();
  
  for (; cNodeIter->isValid(); ++(*cNodeIter)) { 
    OA_ptr<CallGraph::NodeInterface> cNode;
    cNode = cNodeIter->currentCallGraphNode();
    
    ProcHandle currProc = cNode->getProc();
    
    // Add this procedure to the set of those that are reachable 
    // and have already been analyzed (or are about to be analyzed,
    // as with currProc).
    // CSFIAlias visits all procs; this makes more sense for 
    // CSFIAliasReachable.
    mAnalyzedProcs.insert(currProc);
    
    if (debug) {
      std::cout << "\nPhase 1 Iterations over stmts in Proc: " 
                << mIR->toString(currProc) << "\n\n";
    } 
    
    // Iterate over the statements of this procedure
    OA_ptr<IRStmtIterator> stmtIterPtr = mIR->getStmtIterator(currProc);
    // Iterate over the statements of this block adding procedure references
    for ( ; stmtIterPtr->isValid(); ++(*stmtIterPtr)) {
      
      StmtHandle stmt = stmtIterPtr->current();
      
      if (debug) {
        std::cout << "===========================" << std::endl << "stmt = ";
        mIR->dump(stmt,std::cout);
        std::cout << std::endl;
      }
      
      doPhase1Iteration(stmt, cNode, ufset);
      
    } // over stmts
  } // over procedures
  if(debugStats) {
    printf("Time to do phase1 = %lf\n", 
           (clock() - time) / (1.0 * CLOCKS_PER_SEC));
  }
  
  // MMS, 1/24/08, this used to be where map was declared, 
  // I renamed it mMap and made it a protected member variable
  // so that the subclasses could use this info
  
  bool changed = true;
  int changes = 0;
  
  while (changed) {
    //// Phase 2 in CSFIAlias algorithm plus some additions for SubSetRefs
    // do this until there are no more changes
    
    // Please note that 'phases' are a bit of a misnomer since we
    // iterate over phases 2 and 3 together-- as opposed to 
    // separately iterating over phase 2, then phase 3.  Doing them
    // separately appears to be a bug in Fig 5 of the Ryder 2001 paper.
    
    if (debug) { 
      std::cout << std::endl 
                << "======== Beginning of Phase 2 while loop =====" 
                << std::endl; 
    }
    
    time = clock();
    changed = doPhase2Iteration(ufset, mMap);
    
    if(debugStats) {
      printf("Time to do phase2 = %lf\n", 
             (clock() - time) / (1.0 * CLOCKS_PER_SEC));
    }
    
    
    //// Phase 3 in CSFIAlias: accounting for calls through function ptrs
    ////                     and non-function ptr calls
    
    if (debug) { 
      std::cout << std::endl 
                << "======== Beginning of Phase 3 while loop =====" 
                << std::endl; 
    }
    
    // for all the stmts in the whole program
    // Iterate over all the procedures in the program
    time = clock();
    
    // repeat phase 3 until it reaches a fix point
    bool phase3change = false;
    OA_ptr<CallGraph::EdgesIteratorInterface> ceIter;
    OA_ptr<CallGraph::NodeInterface> callerNode;
    OA_ptr<CallGraph::NodeInterface> calleeNode;
    do {
      for (cNodeIter->reset() ; cNodeIter->isValid(); ++(*cNodeIter)) { 
        callerNode = cNodeIter->currentCallGraphNode();
	
        ProcHandle currProc = callerNode->getProc();
	
        // for each out-going edge from the call node: (BK2/2/09)
        //  --  since call graph was created from FIAlias, these
        //     edges may be overly conservative.  Should revisit this
        //     once CSFIAlias is working ... Maybe re-build call graph???
        //  -- also, since we are initially writing this for use with
        //     Fortran front-end, we should not be hitting function ptrs.
        //     FIXME for use with nonFortran as needed
	
        ceIter = callerNode->getCallGraphOutgoingEdgesIterator();
        OA_ptr<CallGraph::EdgeInterface> outEdge;
        
        for (; ceIter->isValid(); ++(*ceIter) ) {
          outEdge = ceIter->currentCallGraphEdge();
          CallHandle call = outEdge->getCallHandle();
          calleeNode = outEdge->getCallGraphSink();
	  
          if (debug) {
            std::cout << "Call site: " << mIR->toString(call) << std::endl;
          }
          
          phase3change = doPhase3Iteration(call, callerNode, calleeNode, 
                                           ufset, mMap);
          changed = (changed || phase3change);
        } // out-going call edges
      } // over call nodes
      
      if(debugStats) {
        printf("Time to do phase3 = %lf\n", 
               (clock() - time) / (1.0 * CLOCKS_PER_SEC));
      }
    }while(phase3change);
    
    changes++;
  } // iterating to fixed point 
  
  if(debugStats) {
    printf("Performed %d iterations to reach a fixed point\n", changes);
  }
  
  return ufset;
}
    
OA_ptr<UnionFindUniverse>
ManagerCSFIAlias::performCSFIAliasReachableProcs( OA_ptr<CallGraph::CallGraphInterface> cgraph)
{
  if ( debug ) {
    std::cout << "performCSFIAliasReachableProcs" << std::endl;
  }
  
  // will come back to this when AllProcs is working
  assert(true && "ReachableProcs is not implemented in ManagerCSFIAlias");
  
  // ptr to union find datastructure
  OA_ptr<UnionFindUniverse> ufset;
  
  /*
    
  // Maintain a list of procedures known to be reachable, but
  // which have not yet been processed.  Those procs in the
  // procIter are considered a seed of reachable functions passed
  // by the caller.  In the common case, we expect procIter to 
  // hold only main.
  // Iterate over all the CallGraphNodes in the CallGraph
  OA_ptr<CallGraph::NodesIteratorInterface> cNodeIter;
  cNodeIter = callGraph->getCallGraphNodesIterator();
  
  for (; cNodeIter->isValid(); ++(*cNodeIter)) { 
  OA_ptr<CallGraph::NodeInterface> cNode;
  cNode = cNodeIter->currentCallGraphNode();
  
  ProcHandle currProc = cNode->getProc();
  addProcToWorkList(currProc);
  }
  
    // BK got to here, and went on elsewhere ...




    // Keep a list of ambiguous/indirect calls that require
    // alias analysis for resolution.  The ProcHandle is the
    // caller.
    std::set<std::pair<CallHandle,ProcHandle> > indirect_calls;

    // ptr to union find datastructure
    OA_ptr<UnionFindUniverse> ufset;

    // declare the union-find datastructure
    // CSFIAlias would have found all MREs in the program
    // at this point and would know how large to make the
    // UnionFindUniverse.  We analyze procedures on demand 
    // and do not have this information.  We could analyze
    // main to seed the size of the universe, but this
    // would complicate the loop below.  Instead, just
    // set it to something arbitrary and let it grow.
    int initialSize = 1024;
    ufset = new UnionFindUniverse(2*initialSize+1);

    // MMS, 1/24/08, this used to be where map was declared, 
    // I renamed it mMap and made it a protected member variable
    // so that the subclasses could use this info

    bool changed = true;

    unsigned int iter_count = 0;
    unsigned int stmts_analyzed = 0;

    while ( changed ) {

        // Iterate over the worlist
        while ( mWorklist.begin() != mWorklist.end() ) {

            ProcHandle currProc = *(mWorklist.begin());
            mWorklist.erase(currProc);

            // Add this procedure to the set of those that are reachable 
            // and have already been analyzed (or are about to be analyzed,
            // as with currProc).
            mAnalyzedProcs.insert(currProc);

            // Map each MemRefExpr in this proc to a unique id and 
            // count them all.
            initMemRefExprs(currProc);

            // Iterate over the statements of this procedure.
            OA_ptr<IRStmtIterator> stmtIterPtr = 
                mIR->getStmtIterator(currProc);
            // Iterate over the statements of this block, adding 
            // procedure references.
            for ( ; stmtIterPtr->isValid(); ++(*stmtIterPtr)) {

	        stmts_analyzed++;
                StmtHandle stmt = stmtIterPtr->current();
                if (debug) {
                    std::cout << "===========================" 
                              << std::endl 
                              << "stmt = ";
                    mIR->dump(stmt,std::cout);
                    std::cout << std::endl;
                }

                doPhase1Iteration(stmt, currProc, ufset);

                // End Ryder phase 1.

                // Visit each call in the current proc.  If the
                // target is Named (i.e., resolvable), 
                // then add it to the worklist.  [However, notice
                // that this happens in doPhase3Iteration.] 
                // If the target is not a Named mre, 
		// then it is an indirect call--
                // add it to the list of indirect_calls.
                // In any case, perform phase 2b-- i.e., param
                // bindings.

                // For all the callsites in the stmt
                OA_ptr<IRCallsiteIterator> callIter = mIR->getCallsites(stmt);
                for ( ; callIter->isValid(); (*callIter)++ ) {
                    CallHandle call = callIter->current();

                    if (debug) {
                        std::cout << "Call site: " 
                                  << mIR->toString(call) 
                                  << std::endl;
                    }

                    // Get the mre for the function call (eg. NamedRef('foo'))
                    OA_ptr<MemRefExpr> callMRE = mIR->getCallMemRefExpr(call);

                    if (!callMRE->isaNamed()) {
                        // The call is not a named MRE-- it must be an
                        // indirect call.
                        indirect_calls.insert(std::pair<CallHandle,ProcHandle>(
                            call,currProc));
                    }

		    // Perform param binding-- i.e., phase 2b.
                   changed = (changed || 
                             doPhase3Iteration(call, currProc, ufset, mMap));

                } // End iteration over call sites

            } // over stmts

        } // over worklist
        
        //// Phase 2 in CSFIAlias algorithm plus some additions for SubSetRefs
        // do this until there are no more changes
      
        // Please note that 'phases' are a bit of a misnomer since we
        // iterate over phases 2 and 3 together-- as opposed to 
        // separately iterating over phase 2, then phase 3.  Doing them
        // separately appears to be a bug in Fig 5 of the Ryder 2001 paper.

        if (debug) { 
            std::cout << std::endl 
		      << "======== Beginning of Phase 2 while loop =====" 
		      << std::endl; 
        }

        changed = doPhase2Iteration(ufset, mMap);
        
        //// Phase 3 in CSFIAlias: accounting for calls through function ptrs
        ////                     and non-function ptr calls
      
        if (debug) { 
            std::cout << std::endl 
                      << "======== Beginning of Phase 3 while loop =====" 
                      << std::endl; 
        }

        // Ryder phase 3/CGO phase 2b performs param binding.
        // Notice that we have already done this above.  However,
	// an indirect call site may have acquired a static
        // resolution through alias analysis.  Therefore, revisit
        // all indirect call sites and (re-)apply param binding.

        for ( std::set<std::pair<CallHandle,ProcHandle> >::iterator it =
              indirect_calls.begin();
              it != indirect_calls.end(); ++it)
        {

            CallHandle call = it->first;
            ProcHandle caller = it->second;

            changed = (changed || doPhase3Iteration(call, caller, ufset, mMap));
            
        } // call sites
        ++iter_count;
    } // iterating to fixed point 

    */

    return ufset;
}

/*!
   Private method for associating an mre with a particular procedure.
*/
MREwContext
ManagerCSFIAlias::recordMREwC(OA_ptr<MemRefExpr> mre, OA_ptr<CallContext> cc,
                              ProcHandle proc )
{
  OA_ptr<CallContext> emptyContext; emptyContext = new CallContext();
  MRENeedsContextQVisitor ncVisitor;
  OA_ptr<CallContext> useContext;

  mre->acceptVisitor(ncVisitor);
  if (ncVisitor.getNeedsContext()) {
    useContext = cc;
  } else {
    useContext = emptyContext;
  }

  MREwContext mrewc(mre, useContext);

  if (debug) {
    std::cout << "In recordMRE ..." << std::endl;
    std::cout << "\tmrewc = ";
    mrewc.output(*mIR);
    std::cout << "\tproc = " << mIR->toString(proc) << std::endl;
  }
  
  // check if it doesn't already have an id in the range 1..
  if (mMREwCToID[mrewc] == 0) {
    if (debug) {
       std::cout << "*.** Getting new mCount value *.**\n";
    }    
    mMREwCToID[mrewc] = mCount++;
  }
  
  if (debug) {
    std::cout << "\tID = " << mMREwCToID[mrewc] << std::endl;
  }
  
  mMREwCToProcs[mrewc].insert(proc);

  return mrewc;
}

/*!
   Private method for associating an mre with a particular procedure and
   MemRefHandle.  Also associates MemRefHandle with given procedure.
*/
MREwContext
ManagerCSFIAlias::recordMREwC( OA_ptr<MemRefExpr> mre, OA_ptr<CallContext> cc,
                               ProcHandle proc, MemRefHandle memref)
{
  // each MemRefHandle can only show up in one procedure
  mMemRefHandleToProc[memref] = proc;
  
  OA_ptr<CallContext> emptyContext; emptyContext = new CallContext();
  MRENeedsContextQVisitor ncVisitor;
  OA_ptr<CallContext> useContext;

  mre->acceptVisitor(ncVisitor);
  if (ncVisitor.getNeedsContext()) {
    useContext = cc;
  } else {
    useContext = emptyContext;
  }

  MREwContext mrewc(mre, useContext);

  if (debug) {
    std::cout << "In recordMRE ..." << std::endl;
    std::cout << "\tmrewc = ";
    mrewc.output(*mIR);
    std::cout << "\tproc = " << mIR->toString(proc) << std::endl;
    std::cout << "\tmemref = ";
    mIR->dump(memref,std::cout);
  }

  // check if it doesn't already have an id in the range 1..
  if (mMREwCToID[mrewc] == 0) {
    if (debug) {
       std::cout << "**** Getting new mCount value ****\n";
    }
    mMREwCToID[mrewc] = mCount++;
  }
  
  if (debug) {
    std::cout << "\tID = " << mMREwCToID[mrewc] << std::endl;
  }
  
  // keep track of which MemRefHandles an MREwC expresses and which procs
  // it is found in
  mMREwCToMemRefHandles[mrewc].insert(memref);
  mMREwCToProcs[mrewc].insert(proc);

  return mrewc;
}

//! creates a dereferenced version of the given mre
OA_ptr<MemRefExpr>
ManagerCSFIAlias::createDeref( OA_ptr<MemRefExpr> mre )
{
  
  OA_ptr<MemRefExpr> retval;

/*
    // if this is an addressOf then remove the addressOf
    if (mre->isaRefOp()) {
      OA_ptr<RefOp> refOp = mre.convert<RefOp>();
      if(refOp->isaAddressOf()) {
         retval = mre->clone();
         retval = refOp->getMemRefExpr();
         return retval;
      }
    }
*/
  int numDerefs = 1;
  OA::OA_ptr<OA::Deref> deref_mre;
  OA::OA_ptr<OA::MemRefExpr> nullMRE;
  
  deref_mre = new OA::Deref(
                            OA::MemRefExpr::USE,
                            nullMRE,
                            numDerefs);
  retval = deref_mre->composeWith(mre->clone());
  
  return retval;

}


std::set<MREwContext > 
ManagerCSFIAlias::allMemRefExprsInSameSet( MREwContext mrewc,
    OA_ptr<UnionFindUniverse> ufset)
{
  std::set<MREwContext> retval;
  
  int setID = ufset->Find(mMREwCToID[mrewc]);
  
  std::map<MREwContext,int>::iterator mapIter;
  for (mapIter=mMREwCToID.begin(); mapIter!=mMREwCToID.end(); mapIter++)
    {
      MREwContext mwc = mapIter->first;
      if (debug) {
        std::cout << "allMREwContextInSameSet: mrewc = ";
        mwc.output(*mIR);
      }
      if ( ufset->Find(mMREwCToID[mwc]) == setID ) {
        retval.insert(mwc);
      }
    }
  
  return retval;
}

/*!
   Visitor over memory reference expressions that grabs all sub 
   MREs and makes sure that they are recorded.

   NOTE that we are treating the non-address of namedref as a sub
   MRE to the named ref that has its address set.
   Actually need this for all MREs so doing it outside of this visitor.
 */

class RecordMREwCsVisitor : public virtual MemRefExprVisitor {
public:
  RecordMREwCsVisitor(ManagerCSFIAlias &manager, ProcHandle proc,
                      OA_ptr<CallContextSetIterator> ccsetIter)
    : mManager(manager), mProc(proc), mCCIter(ccsetIter)
  {   
  }
  ~RecordMREwCsVisitor() {}
  
  void visitNamedRef(NamedRef& ref) { 
    OA_ptr<MemRefExpr> mre = ref.clone();
    mCCIter->reset();
    for ( ; mCCIter->isValid(); ++(*mCCIter)) {
      mManager.recordMREwC(mre, mCCIter->current(), mProc); 
    }
  }
  void visitUnnamedRef(UnnamedRef& ref) {
    OA_ptr<MemRefExpr> mre = ref.clone();
    mCCIter->reset();
    for ( ; mCCIter->isValid(); ++(*mCCIter)) {
      mManager.recordMREwC(mre, mCCIter->current(), mProc); 
    }
  }
  void visitUnknownRef(UnknownRef& ref) {
    OA_ptr<MemRefExpr> mre = ref.clone();
    mCCIter->reset();
    for ( ; mCCIter->isValid(); ++(*mCCIter)) {
      mManager.recordMREwC(mre, mCCIter->current(), mProc); 
    }
  }
  void visitAddressOf(AddressOf& ref) {
    // do not record the addressOf, but record its child
    OA_ptr<MemRefExpr> mre = ref.getMemRefExpr();
    if (!mre.ptrEqual(0)) { 
      mre->acceptVisitor(*this); 
    }
  }
  void visitDeref(Deref& ref) { 
    // record self
    OA_ptr<MemRefExpr> mref = ref.clone();
    mCCIter->reset();
    for ( ; mCCIter->isValid(); ++(*mCCIter)) {
      mManager.recordMREwC(mref, mCCIter->current(), mProc); 
    }
    // must also visit child memory reference
    OA_ptr<MemRefExpr> mre = ref.getMemRefExpr();
    if (!mre.ptrEqual(0)) {
      mre->acceptVisitor(*this); 
    }
  }
  // default handling of more specific SubSet specificiations
  void visitSubSetRef(SubSetRef& ref) {
    // record self
    OA_ptr<MemRefExpr> mref = ref.clone();
    mCCIter->reset();
    for ( ; mCCIter->isValid(); ++(*mCCIter)) {
      mManager.recordMREwC(mref, mCCIter->current(), mProc); 
    }
    // must also visit child memory reference
    OA_ptr<MemRefExpr> mre = ref.getMemRefExpr();
    if (!mre.ptrEqual(0)) { 
      mre->acceptVisitor(*this); 
    }
  }
private:
  ManagerCSFIAlias& mManager;
  ProcHandle mProc;
  OA_ptr<CallContextSetIterator> mCCIter;
};




void 
ManagerCSFIAlias::initMemRefExprs( OA_ptr<CallGraph::NodeInterface> cNode )
{
  ProcHandle currProc = cNode->getProc();
  
  // 1) get all CallContexts for this cNode 
  
  OA_ptr<CallContextSetIterator> ccIter;
  ccIter = mCCResults->getCallContextSet(currProc);
  
  if (debug) {
    std::cout << "IN initMemRefExpr for proc: " << mIR->toString(currProc)
              << " with CallContexts: " <<std::endl;
    ccIter->reset();
    for ( ; ccIter->isValid(); ++(*ccIter)) {
      std::cout << "\t" << (ccIter->current())->toString(*mIR) << "\n";
    }
  }

  // 2) now proceed as FIAlias, but have set of CallContext for recording
  
  // visitor that can record all mres and sub mres within an mre for
  // all CallContexts for the currProc associated with this cNode.
  RecordMREwCsVisitor recMREVisitor(*this, currProc, ccIter);
  
  // Iterate over the statements of this procedure
  OA_ptr<IRStmtIterator> stmtIterPtr = mIR->getStmtIterator(currProc);
  // Iterate over the statements of this block adding procedure references
  for ( ; stmtIterPtr->isValid(); ++(*stmtIterPtr)) {
    
    StmtHandle stmt = stmtIterPtr->current();
    if (debug) {
      std::cout << std::endl << "stmt = ";
      mIR->dump(stmt,std::cout);
    }
    if (debug) {
      std::cout << "\n>> MemRefHandles to MREs\n";
    }
    
    // map each memory reference expr to a unique number in the range 1..
    OA_ptr<MemRefHandleIterator> mrIterPtr = mIR->getAllMemRefs(stmt);
    for (; mrIterPtr->isValid(); (*mrIterPtr)++ ) {
      
      MemRefHandle memref = mrIterPtr->current();
      
      // get the memory reference expressions for this handle
      OA_ptr<MemRefExprIterator> mreIterPtr 
        = mIR->getMemRefExprIterator(memref);
      
      // for each mem-ref-expr associated with this memref
      for (; mreIterPtr->isValid(); (*mreIterPtr)++) {
        OA_ptr<OA::MemRefExpr> mre = mreIterPtr->current();
        
        // associate this mre with the particular MemRefHandle
        // for all CallContexts
        ccIter->reset();
        for ( ; ccIter->isValid(); ++(*ccIter)) {
          recordMREwC(mre, ccIter->current(), currProc, memref);
        }
        
        // the recordMREs visitor will get all subrefs
        // and not associate them with a MemRefHandle
          
        mre->acceptVisitor(recMREVisitor);

      } 
      
    } // loop over MemRefHandles
    
    // there might be some MemRefExprs in PtrAssignPairs that are
    // not associated with a MemRefHandle
    // for each ptrassignpair
    
    if (debug) {
      std::cout << "\n>>> PtrAssignPairs\n\n";
    }
    

    OA_ptr<PtrAssignPairStmtIterator> pairIter
      = mIR->getPtrAssignStmtPairIterator(stmt);
    for ( ; pairIter->isValid(); (*pairIter)++ ) {
      OA_ptr<MemRefExpr> target = pairIter->currentTarget();
      target->acceptVisitor(recMREVisitor);
      
      OA_ptr<MemRefExpr> source = pairIter->currentSource();        
      source->acceptVisitor(recMREVisitor);
    }
    
    if (debug) {
      std::cout << "\n>>>> ParamBindPtrAssigns\n\n";
    }
    
    // there can be some implicit MemRefExprs in ParamBindPtrAssigns
    // for all the callsites in the stmt
    OA_ptr<IRCallsiteIterator> callIter = mIR->getCallsites(stmt);
    for ( ; callIter->isValid(); (*callIter)++ ) {
      CallHandle call = callIter->current();
      
      // get the mre for the function call (eg. NamedRef('foo'))
      OA_ptr<MemRefExpr> callMRE = mIR->getCallMemRefExpr(call);
      callMRE->acceptVisitor(recMREVisitor);
      
      // for each implicit ptrassignpair due to formal/actual binding
      OA_ptr<ParamBindPtrAssignIterator> pairIter
        = mIR->getParamBindPtrAssignIterator(call);
      for ( ; pairIter->isValid(); (*pairIter)++ ) {
        OA_ptr<MemRefExpr> actualMRE = pairIter->currentActual();        
        
          actualMRE->acceptVisitor(recMREVisitor);
        
      }
    }
    
    
  } // loop over stmts
  
  if (debug) {
    std::cout << "\n>>>>> Formal parameters\n\n";
  }
  
  // loop over formals for this procedure and recordMREs for each formal
  // also add this formal SymHandle to the set of formals for currProc
  int formalCount = 0;
  SymHandle formalSym;
  while ( (formalSym=mIR->getFormalSym(currProc,formalCount)) 
          != SymHandle(0) ) 
    {
      // put ian set of formals for currProc
      mProcToFormalSet[currProc].insert(formalSym);
      
      // make mre for the formal
      // Create a namedRef for the formal.
      bool addressTaken = false;
      bool fullAccuracy = true;
      // DEF since formal = actual.
      MemRefExpr::MemRefType mrType = MemRefExpr::DEF;
      OA_ptr<MemRefExpr> formalmre; 

      // formal parameter NamedRefs are isLocal == true
      formalmre = new NamedRef(mrType, formalSym, true);

      // Thinking of Using new IRInterface::createNamedRef() below
      // FIXME???
      // formalmre will be MemRefExpr::USE
      // is this an error???

      //formalmre = mIR->createNamedRef(formalSym);


      // recordMRE for formal
      // update mMREToMemRefHandle, mMREToProc, and mMREToID
      // and updates mCount of all memory references
      
      formalmre->acceptVisitor(recMREVisitor);
      
      // increment the count used within this loop
      formalCount++;
    }
}

void 
ManagerCSFIAlias::initMemRefExprs( OA_ptr<CallGraph::CallGraphInterface> callGraph )
{
    
  // Iterate over all the CallGraphNodes in the CallGraph
  OA_ptr<CallGraph::NodesIteratorInterface> cNodeIter;
  cNodeIter = callGraph->getCallGraphNodesIterator();
  
  for (; cNodeIter->isValid(); ++(*cNodeIter)) { 
    OA_ptr<CallGraph::NodeInterface> cNode;
    cNode = cNodeIter->currentCallGraphNode();
    initMemRefExprs(cNode);
  }

}

/*! An implementation of the merge function in the Ryder2001 description
 *  of FIAlias.
 */
void ManagerCSFIAlias::merge(int part1, int part2, 
        OA_ptr<UnionFindUniverse> ufset, 
        std::map<int,std::map<OA_ptr<MemRefExpr>,int> > & map  ) 
{
  int part1_find = ufset->Find(part1);
  int part2_find = ufset->Find(part2);
  
  if (debug) {
    std::cout << "ManagerCSFIAlias::merge (" << part1 << ", " << part2
              << ", ... ) with ... " << part1_find << ", "
              << part2_find << "\n\n";
  }

  if (part1_find != part2_find) {
    std::map<OA_ptr<MemRefExpr>,int> &old1 = map[part1_find];
    std::map<OA_ptr<MemRefExpr>,int> &old2 = map[part2_find];
    ufset->Union(part1_find, part2_find, part2_find);
    
    if (debug) {
      std::cout << "\t===> Union ( "
                << part1_find << ", " << part2_find << " )" << std::endl;
    }
    
    // loop over all of the mres that are now in the newly unioned set
    // I think the fastest way to do this is to look over the mres
    // in the two old sets.  Otherwise I have to loop over all the MREs
    // and figure out which ones are in the new set.
    std::map<OA_ptr<MemRefExpr>,int>::iterator mapIter;
    part1_find = ufset->Find(part1);
    for (mapIter = old1.begin(); mapIter != old1.end(); mapIter++) {
      OA_ptr<MemRefExpr> mre = mapIter->first;
      if ( old1[mre]==0 && old2[mre]==0 ) {
        map[part1_find][mre] = 0;
      } else if (old2[mre]==0) {
        map[part1_find][mre] = old1[mre];
      } else if (old1[mre]==0) {
        map[part1_find][mre] = old2[mre];
      } else {
        merge(old1[mre], old2[mre], ufset, map);
        //! Need to update the map once ufsets are modified
        map[part1_find][mre] = ufset->Find(old1[mre]);
      }
    }
    for (mapIter = old2.begin(); mapIter != old2.end(); mapIter++) {
      OA_ptr<MemRefExpr> mre = mapIter->first;
      if ( old1[mre]==0 && old2[mre]==0 ) {
        map[part1_find][mre] = 0;
      } else if (old2[mre]==0) {
        map[part1_find][mre] = old1[mre];
      } else if (old1[mre]==0) {
        map[part1_find][mre] = old2[mre];
      } else {
        merge(old1[mre], old2[mre], ufset, map);
        //! Need to update the map once ufses are modified
        map[part1_find][mre] = ufset->Find(old1[mre]);
      }
    }
  }

}

void ManagerCSFIAlias::outputMREwCsInSet(int setID, 
        OA_ptr<UnionFindUniverse> ufset, 
        std::map<int,std::map<OA_ptr<MemRefExpr>,int> > & map  ) 
{
  std::cout << "All mrewcs in setID = " << setID << std::endl;
  std::map<MREwContext,int>::iterator mapIter;
  for (mapIter=mMREwCToID.begin(); mapIter!=mMREwCToID.end(); mapIter++) {
    if ( ufset->Find(mMREwCToID[mapIter->first]) == ufset->Find(setID) ) {
      MREwContext mymrewc = mapIter->first;
      mymrewc.output(*mIR);
    }
  }
}

  } // end of namespace Alias
} // end of namespace OA

