/*! \file
  
  \brief Implementation of MRHwContext

  \authors Barbara Kreaseck
  \version $Id$

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#include "MRHwContext.hpp"

namespace OA {
  namespace Alias {


// *******************************************************
// Beginning of MRHwContext code
// *******************************************************


MRHwContext::MRHwContext(MemRefHandle mrh, OA_ptr<CallContext> cc)
{
  mMRH = mrh;
  mCallContext = cc;
}

MRHwContext::MRHwContext(OA_ptr<MRHwContext> other)
{
  mMRH = other->mMRH;
  mCallContext = other->mCallContext;

  // to clone or not to clone ... =)
  /*
  mMRH = other->mMRH;
  mCallContext = (other->mCallContext)->clone();
  */
}

MRHwContext::~MRHwContext()
{

}

MemRefHandle MRHwContext::getMRH()
{
  return mMRH;
}

OA_ptr<CallContext> MRHwContext::getCallContext()
{
  return mCallContext;
}

bool MRHwContext::operator ==(const MRHwContext& other) const
{
  if (mMRH == other.mMRH) { 
    if (mCallContext.ptrEqual(0)) {
      if ((other.mCallContext).ptrEqual(0)) {
        return true;
      } else {
        // BK: if one's CallContext is NULL and the other is not NULL for same MRE
        // then shouldn't this assert?  Seems like if we are using NULL for 
        // NamedRefs (local vars), then we shouldn't see a NULL and a nonNULL 
        // for the same MRE ... hmmm ... but we are making MREs for formals 
        return false;
      }
    } else {
      if ((other.mCallContext).ptrEqual(0)) {
        // BK: see comment above
        return false;
      } else {
        // both are non-NULL
        if ((*mCallContext)==(*(other.mCallContext))) {
          return true;
        } else {
          return false;
        }
      }
    }
  }
  return false;
}
  
bool MRHwContext::operator <(const MRHwContext& other) const
{
  if (mMRH < other.mMRH) {
    return true;
  } 
  else if (mMRH == other.mMRH) {
    if (mCallContext.ptrEqual(0) || (other.mCallContext).ptrEqual(0)) {
      // at least one is NULL
      if (mCallContext.ptrEqual(0)) {
        if ((other.mCallContext).ptrEqual(0)) {
          // both NULL
          return false;
        } else {
          // mCallContext is NULL, and other is not
          return true; // NULL < any other address
        }
      } else {
        // other is NULL, and mCallContext is not
        return false; // any address ( NOT < ) NULL
      }
    } else {
      // both are non-NULL
      return ((*mCallContext) < (*(other.mCallContext)));
    } 
  }
  return false; 
}

OA_ptr<MRHwContext> MRHwContext::clone()
{
  OA_ptr<MRHwContext> result;
  if (mCallContext.ptrEqual(0)) {
    result = new MRHwContext(mMRH,mCallContext);
  } else {
    result = new MRHwContext(mMRH,mCallContext->clone());
  }
  return result;
}

MRHwContext& MRHwContext::operator=( MRHwContext& other)
{
  // This copies, not clones.  
  mMRH = other.mMRH;
  mCallContext = other.mCallContext; // OK to copy NULL here
  return *this;
}

void MRHwContext::output(OA::IRHandlesIRInterface& ir) const
{
  sOutBuild->objStart("MRHwContext");
  sOutBuild->fieldStart("mMRH");
  sOutBuild->outputIRHandle(mMRH, ir);
  sOutBuild->fieldEnd("mMRH");
  sOutBuild->fieldStart("mCallContext");
  if (mCallContext.ptrEqual(0)) {
    sOutBuild->field("mCallContext","null");
  } else {
    mCallContext->output(ir);
  }
  sOutBuild->fieldEnd("mCallContext");
  sOutBuild->objEnd("MRHwContext");
}

void MRHwContext::output(OA::IRHandlesIRInterface& ir,
			 const Interface& aliasResults) const
{
  output(ir);
}

void MRHwContext::dump(std::ostream& os,
		       IRHandlesIRInterface& ir,
		       Interface& aliasResults)
{
  os << "[ MRHwContext: \n";
  os << ir.toString(mMRH);
  if (mCallContext.ptrEqual(0)) {
    os << "\tCallContext(null)\n";
  } else {
    mCallContext->dump(os,ir);
  }
  os << "\n]";
}

  } // end of Alias Namespace
} // end of OA Namespace
