/*! \file
  
  \brief Interface for CallContext sets and iterator.

  \authors Barbara Kreaseck
  \version $Id$

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#ifndef CallContextSet_H
#define CallContextSet_H

#include <OpenAnalysis/Utils/Iterator.hpp>
#include <OpenAnalysis/Utils/OutputBuilder.hpp>
#include <OpenAnalysis/Alias/CallContext.hpp>


namespace OA {
  namespace Alias {

class Interface;
class CallContextSet;

//! Iterator over CallContexts in a CallContextSet.
class CallContextSetIterator : public Iterator {
  public:

  //! Construct an iterator given CallContextSet
  CallContextSetIterator(const CallContextSet& ccSet);
  
  //! Construct an iterator over an empty set. (isValid() will always return
  //! false).
  CallContextSetIterator();
  
  ~CallContextSetIterator();
  
  // ************************************************************
  //  From interface
  // ************************************************************
  OA_ptr<CallContext> current();
  
  bool isValid() const;
  void operator++ ();
  void reset();

  private:
  OA_ptr<std::set<OA_ptr<CallContext> > > mSet;
  std::set<OA_ptr<CallContext> >::iterator mIter;
};



//! Set of CallContexts.
class CallContextSet : public virtual Annotation {
  public:
    //************************************************************
    // Constructors / destructors
    //************************************************************
  
    /*! Construct a new CallContextSet given a set of CallContexts */
    CallContextSet(OA_ptr<std::set<OA_ptr<CallContext> > > ccSet);

    /*! Construct an empty set of CallContexts. */
    CallContextSet();

    //! copy constructor
    CallContextSet(CallContextSet& other);

    //! Return a ptr to a clone
    OA_ptr<CallContextSet> clone();

    //************************************************************
    // Queries
    //************************************************************

    //! return an iterator over all tags in the set
    OA_ptr<CallContextSetIterator> getIterator() const;

    //! return true if the passed CallContext is included in the set
    bool contains(OA_ptr<CallContext> cc);

    //! Return the number of elements contained in this set
    int size();

    //************************************************************
    // Modification
    //************************************************************

    //! Insert a new CallContext into the set
    void insert(OA_ptr<CallContext> cc);

    //! Remove all CallContexts from the set
    void clear();

    //************************************************************
    // Output
    //************************************************************
    virtual void output(OA::IRHandlesIRInterface& ir) const;
    virtual void output(
        OA::IRHandlesIRInterface& ir,
        Alias::Interface &aliasResults) const;
    virtual void dump(
        std::ostream& os,
        OA::IRHandlesIRInterface& ir);

  friend class CallContextSetIterator;
  private:
  OA_ptr<std::set<OA_ptr<CallContext> > > mCCSet;

};




} } // end namespaces 

#endif
