/*! \file
  
  \brief Contains implementation of CallContextSet class and iterator class.

  \authors Barbara Kreaseck
  \version $Id$

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#include <OpenAnalysis/Alias/CallContextSet.hpp>

#include <OpenAnalysis/Alias/Interface.hpp>
#include <OpenAnalysis/Utils/OutputBuilder.hpp>

namespace OA {
  namespace Alias {

CallContextSetIterator::CallContextSetIterator(const CallContextSet& ccSet) 
  : mSet(ccSet.mCCSet) {
  mIter = mSet->begin();
}

CallContextSetIterator::CallContextSetIterator() {
  mSet = NULL;
}

CallContextSetIterator::~CallContextSetIterator() { }

OA_ptr<CallContext> CallContextSetIterator::current() {
  if (isValid()) {
    return *mIter;
  } else {
    assert("CallContextSetIterator: called current() with no set\n" && false);
  }
}

bool CallContextSetIterator::isValid() const {
  if (mSet.ptrEqual(0)) {
    return false;
  }
  return (mIter != mSet->end());
}

void CallContextSetIterator::operator++() {
  if (isValid()) {
    ++mIter;
  }
}

void CallContextSetIterator::reset() {
  if (!mSet.ptrEqual(0)) {
    mIter = mSet->begin();
  }
}

// ---------------------------------------------
//  CallContextSet functions
// ---------------------------------------------


CallContextSet::CallContextSet(OA_ptr<std::set<OA_ptr<CallContext> > > ccSet) {
  mCCSet = new std::set<OA_ptr<CallContext> >;
  if (!ccSet.ptrEqual(0)) {
    std::set<OA_ptr<CallContext> >::iterator mIter;
    mIter = ccSet->begin();
    for ( ; mIter != ccSet->end(); ++mIter) {
      mCCSet->insert((*mIter)->clone());
    }
  }
}


CallContextSet::CallContextSet()
{
  mCCSet = new std::set<OA_ptr<CallContext> >;
}

//! copy constructor
CallContextSet::CallContextSet(CallContextSet& other)
{
  mCCSet = new std::set<OA_ptr<CallContext> >;
  OA_ptr<CallContextSetIterator> sIter;
  sIter = other.getIterator();
  for ( ; sIter->isValid(); ++(*sIter)) {
    mCCSet->insert((sIter->current())->clone());
  }
}


//! Return a ptr to a clone
OA_ptr<CallContextSet> CallContextSet::clone()
{
    OA_ptr<CallContextSet> retval;
    retval = new CallContextSet(*this);
    return retval;
}

OA_ptr<CallContextSetIterator> CallContextSet::getIterator() const {
    OA_ptr<CallContextSetIterator> iter;
    iter = new CallContextSetIterator(*this);
    return iter;
}

bool CallContextSet::contains(OA_ptr<CallContext> cc) {
  return (mCCSet->find(cc) != mCCSet->end());
}

int CallContextSet::size() {
    return mCCSet->size();
}


void CallContextSet::insert(OA_ptr<CallContext> cc) {
    mCCSet->insert(cc);
}

void CallContextSet::clear() {
    mCCSet->clear();
}

void CallContextSet::output(OA::IRHandlesIRInterface& ir) const {
    sOutBuild->objStart("CallContextSet");
    sOutBuild->listStart();
    OA_ptr<CallContextSetIterator> sIter;
    sIter = getIterator();
    for ( ; sIter->isValid(); ++(*sIter)) {
      sOutBuild->listItemStart();
      (sIter->current())->output(ir);
      sOutBuild->listItemEnd();
    }
    sOutBuild->listEnd();
    sOutBuild->objEnd("CallContextSet");
}

void CallContextSet::output(
    OA::IRHandlesIRInterface& ir,
    Alias::Interface &aliasResults) const
{
  output(ir);
}

void CallContextSet::dump(
    std::ostream& os,
    OA::IRHandlesIRInterface& ir)
{
    OA_ptr<CallContextSetIterator> sIter;

    os << "[";
    sIter = getIterator();
    for(; sIter->isValid(); ++(*sIter)) {
      (sIter->current())->dump(os,ir);
    }
    os << "]";
}

} } // end namespaces


