/*! \file
  
  \brief Implementation of Alias::CCSetPerProc

  \author Barbara Kreaseck

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/


#include <OpenAnalysis/Utils/OutputBuilder.hpp>
//#include <OpenAnalysis/Alias/CCSetPerProc.hpp>
#include "CCSetPerProc.hpp"

namespace OA {
  namespace Alias {

CCSetPerProc::CCSetPerProc() 
{ 
  mNumCCs = 0;
}

int CCSetPerProc::getNumCallContexts() {
  return mNumCCs;
}

OA_ptr<CallContextSetIterator> 
CCSetPerProc::getCallContextSet(ProcHandle proc)
{
  OA_ptr<CallContextSetIterator> retval;

  if (mProcToCCSet.find(proc)!=mProcToCCSet.end()) {
    OA_ptr<CallContextSet> ccSet;
    ccSet = mProcToCCSet[proc];
    if (ccSet.ptrEqual(NULL)) {
      assert("CCSetPerProc: no CallContextSet found for this proc" 
             && false);
    }
    retval = new CallContextSetIterator(*ccSet);
  } else {
    assert("CCSetPerProc: no entry in map for this proc" && false);
  }

  return retval;
}

//! return size of CallContextSet for given proc
int CCSetPerProc::getCCSetSize(ProcHandle proc)
{
  if (mProcToCCSet.find(proc) != mProcToCCSet.end()) {
    return (mProcToCCSet[proc])->size();
  }

  //assert?
  return 0;
}


//*****************************************************************
// Construction methods 
//*****************************************************************

void CCSetPerProc::recordCallContextSet(ProcHandle proc, 
                                 OA_ptr<CallContextSet> ccset)
{
    mProcToCCSet[proc] = ccset->clone();
}

void CCSetPerProc::setNumCCs(int n)
{
  mNumCCs = n;
}

//*****************************************************************
// Annotation Interface
//*****************************************************************
void CCSetPerProc::output(IRHandlesIRInterface &ir) const
{
  sOutBuild->mapStart("mProcToCCSets","ProcHandle","CallContextSet"); {
    std::map<ProcHandle, OA_ptr<CallContextSet> >::const_iterator mapIter;
    for (mapIter = mProcToCCSet.begin(); 
         mapIter != mProcToCCSet.end(); mapIter++) {
      ProcHandle proc = mapIter->first;
      OA_ptr<CallContextSet> ccSet = mapIter->second;
      sOutBuild->mapEntryStart(); {
        sOutBuild->mapKeyStart(); {
          sOutBuild->outputIRHandle(proc,ir);
        } sOutBuild->mapKeyEnd();
        sOutBuild->mapValueStart(); {
          sOutBuild->objStart("CCSet"); {
            ccSet->output(ir);
          } sOutBuild->objEnd("CCSet");
        } sOutBuild->mapValueEnd();
      } sOutBuild->mapEntryEnd();
    }

    sOutBuild->fieldStart("mNumCCs"); {
      std::ostringstream oss; oss << mNumCCs;
      sOutBuild->outputString(oss.str());
    }sOutBuild->fieldEnd("mNumCCs");

  } sOutBuild->mapEnd("mProcToCCSets");
}

//*****************************************************************
// Output
//*****************************************************************

//! incomplete output of info for debugging
void CCSetPerProc::dump(std::ostream& os, IRHandlesIRInterface &ir)
{
    os << "============================ CCSetPerProc" << std::endl;

    os << "\nTotal CallContexts = " << mNumCCs << "\n\n";

    // iterate over all procs we know about
    std::map<ProcHandle, 
             OA_ptr<CallContextSet> >::iterator mapIter;
    for (mapIter = mProcToCCSet.begin(); 
         mapIter != mProcToCCSet.end(); mapIter++) {
      ProcHandle proc = mapIter->first;
      OA_ptr<CallContextSet> ccSet = mapIter->second;

      os << "ProcHandle(" << ir.toString(proc) << ")" 
           << std::endl << "\tccSet: " << std::endl;
        ccSet->dump(os,ir);
    }
}



  } // end of Alias namespace
} // end of OA namespace
