/*! \file
  
  \brief Declarations of an AnnotationManager that generates an
         ActivePerStmt with CSResults by doing Vary analysis and 
         using results of CSAlias and CSUseful analysis.

  \authors Michelle Strout, Barbara Kreaseck

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#ifndef ManagerICFGCSVaryActive_H
#define ManagerICFGCSVaryActive_H

//--------------------------------------------------------------------
#include <cassert>

// OpenAnalysis headers
#include <OpenAnalysis/IRInterface/ActivityIRInterface.hpp>
#include <OpenAnalysis/DataFlow/ICFGCSDFProblem.hpp>
#include <OpenAnalysis/DataFlow/ICFGCSDFSolver.hpp>
#include <OpenAnalysis/Activity/InterUseful.hpp>
#include <OpenAnalysis/Activity/ActivePerStmt.hpp>
#include <OpenAnalysis/Alias/Interface.hpp>
#include <OpenAnalysis/Alias/CCSetPerProc.hpp>
#include <OpenAnalysis/ExprTree/CollectMREVisitor.hpp>
#include <OpenAnalysis/MemRefExpr/MemRefExprVisitor.hpp>

namespace OA {
  namespace Activity {

/*! 
   Creates ActivePerStmt with CSResults, which can be queried for the sets
   of AliasTags that are InActive and/or OutActive for each stmt+ callcontext
*/
class ManagerICFGCSVaryActive : public virtual DataFlow::ICFGCSDFProblem
{ 
public:
  ManagerICFGCSVaryActive(OA_ptr<ActivityIRInterface> _ir);
  ~ManagerICFGCSVaryActive () {}

  OA_ptr<ActivePerStmt> 
  performAnalysis(OA_ptr<ICFG::ICFGInterface> icfg,
                  OA_ptr<Alias::Interface> Alias, // expects CSAlias
                  OA_ptr<Alias::CCSetPerProc> ccResults,
                  OA_ptr<InterUseful> interUseful,
                  DataFlow::DFPImplement algorithm);


private:
  //========================================================
  // helper routines
  //========================================================
  OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > 
  calculateActive(
        OA_ptr<DataFlow::DFSetBitImplIterator<Alias::AliasTag> > varyIter,
        OA_ptr<DataFlow::DFSetBitImplIterator<Alias::AliasTag> > usefulIter,
        StmtHandle stmt, OA_ptr<Alias::CallContext> cc);

  //========================================================
  // implementation of ICFGDFProblem interface
  //========================================================
  //--------------------------------------------------------
  // initialization callbacks
  //--------------------------------------------------------

  //! Return an initialized top set
  OA_ptr<DataFlow::DataFlowSet>  
  initializeTop(OA_ptr<Alias::CallContext> cc);
  
  //! Should generate an in and out DataFlowSet for node
  OA_ptr<DataFlow::DataFlowSet> 
  initializeNodeIN(OA_ptr<ICFG::NodeInterface> n,
                   OA_ptr<Alias::CallContext> cc);

  OA_ptr<DataFlow::DataFlowSet> 
  initializeNodeOUT(OA_ptr<ICFG::NodeInterface> n,
                    OA_ptr<Alias::CallContext> cc);
 
  //--------------------------------------------------------
  // solver callbacks 
  //--------------------------------------------------------
  
  //! OK to modify set1 and return it as result, because solver
  //! only passes a tempSet in as set1
  OA_ptr<DataFlow::DataFlowSet>
  meet(OA_ptr<DataFlow::DataFlowSet> set1, 
       OA_ptr<DataFlow::DataFlowSet> set2,
       OA_ptr<Alias::CallContext> cc); 

  //! OK to modify in set and return it again as result because
  //! solver clones the BB in sets
  OA_ptr<DataFlow::DataFlowSet> 
  transfer(ProcHandle proc, OA_ptr<DataFlow::DataFlowSet> in, 
           OA_ptr<Alias::CallContext> cc, OA::StmtHandle stmt);


  OA_ptr<DataFlow::DataFlowSet> 
  transfer(ProcHandle proc, OA_ptr<DataFlow::DataFlowSet> in,
           OA_ptr<Alias::CallContext> cc, OA::CallHandle call);


  
  //! transfer function for the entry node of the given procedure
  //! should manipulate incoming data-flow set in any special ways
  //! for procedure and return outgoing data-flow set for node
  OA_ptr<DataFlow::DataFlowSet> 
  entryTransfer(ProcHandle proc, OA_ptr<DataFlow::DataFlowSet> in,
                OA_ptr<Alias::CallContext> cc);

  //! transfer function for the exit node of the given procedure
  //! should manipulate outgoing data-flow set in any special ways
  //! for procedure and return incoming data-flow set for node
  OA_ptr<DataFlow::DataFlowSet> 
  exitTransfer(ProcHandle proc, OA_ptr<DataFlow::DataFlowSet> out,
               OA_ptr<Alias::CallContext> cc);

  //! Propagate a data-flow set from caller to callee
  OA_ptr<DataFlow::DataFlowSet> 
  callerToCallee(ProcHandle caller, OA_ptr<DataFlow::DataFlowSet> dfset,
                 OA_ptr<Alias::CallContext> cc,
                 CallHandle call, ProcHandle callee);
  
  //! Propagate a data-flow set from callee to caller
  OA_ptr<DataFlow::DataFlowSet> 
  calleeToCaller(ProcHandle callee, OA_ptr<DataFlow::DataFlowSet> dfset,
                 OA_ptr<Alias::CallContext> cc,
                 CallHandle call, ProcHandle caller);

 //! Propagate a data-flow set from call node to return node
 OA_ptr<DataFlow::DataFlowSet> 
 callToReturn(ProcHandle caller, OA_ptr<DataFlow::DataFlowSet> dfset, 
              OA_ptr<Alias::CallContext> cc,
              CallHandle call, ProcHandle callee);

  //! --------------------------------------------------------
  //! solver callbacks for DEBUGGING
  //! --------------------------------------------------------

  //! ===== Only the problem itself knows what type of DataFlowSet is being
  //! used, and how to print it ... ICFGCSSolver does not know that info
  void printDFSet(std::ostream& os, OA_ptr<DataFlow::DataFlowSet>);

  //! ===== ICFGCSSolver cannot print CallContexts ... (no IRInterface)
  //! so, ICFGCSDFProblem needs to do it.
  void printCallContext(std::ostream& os, OA_ptr<Alias::CallContext> cc);

private:
  OA_ptr<ActivePerStmt> mActive;
  OA_ptr<InterUseful> mInterUseful;
  OA_ptr<ActivityIRInterface> mIR;
  OA_ptr<Alias::Interface> mCSAlias;
  int mMaxNumTags;
  OA_ptr<DataFlow::ICFGCSDFSolver> mCSSolver;
  OA_ptr<Alias::CCSetPerProc> mCCSetPerProc;
  OA_ptr<ICFG::ICFGInterface> mICFG;

  //! Helper routine for precomputing data-flow sets for each statement.
  void 
  precalcDefsAndUses(OA_ptr<ICFG::ICFGInterface> icfg);
  
  OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> >  
  calcDefs(OA::OA_ptr<OA::AssignPairIterator> aPairIter,
           OA_ptr<Alias::CallContext> cc);
  
  OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> >        
  calcMustDefs(OA::OA_ptr<OA::AssignPairIterator> aPairIter,
               OA_ptr<Alias::CallContext> cc);
  
  OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > 
  calcUses(OA::OA_ptr<OA::AssignPairIterator> aPairIter,
           OA_ptr<Alias::CallContext> cc);
  
  // Places to keep the precomputing data-flow sets.
  // FIXME: copied from ManagerICFGCSUseful, ripe for refactoring
  std::map<Alias::StmtHwContext,
           OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > > mStmtwCToDefs;
  std::map<Alias::StmtHwContext,
           OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > > mStmtwCToMustDefs;
  std::map<Alias::StmtHwContext,
           OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > > mStmtwCToUses;

  std::map<Alias::CallHwContext,
           OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > > mCallwCToDefs;
  std::map<Alias::CallHwContext,
           OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > > mCallwCToMustDefs;
  std::map<Alias::CallHwContext,
           OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > > mCallwCToUses;

};

  } // end of Activity namespace
} // end of OA namespace

#endif
