/*! \file
  
  \brief Declarations of an AnnotationManager that generates an InterUseful
         using the ICFGCSDFSolver for Context Sensitive results.

  \authors Michelle Strout, Barbara Kreaseck

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#ifndef ManagerICFGCSUseful_H
#define ManagerICFGCSUseful_H

//--------------------------------------------------------------------
#include <cassert>

// OpenAnalysis headers
#include <OpenAnalysis/IRInterface/ActivityIRInterface.hpp>
#include <OpenAnalysis/DataFlow/ICFGCSDFProblem.hpp>
#include <OpenAnalysis/DataFlow/ICFGCSDFSolver.hpp>
//#include <OpenAnalysis/Activity/ICFGDep.hpp>
#include <OpenAnalysis/Activity/InterUseful.hpp>
#include <OpenAnalysis/Alias/Interface.hpp>
#include <OpenAnalysis/Alias/CCSetPerProc.hpp>
#include <OpenAnalysis/ExprTree/CollectMREVisitor.hpp>
#include <OpenAnalysis/MemRefExpr/MemRefExprVisitor.hpp>

namespace OA {
  namespace Activity {

/*! 
    Creates InterUseful, which can be queried for a CSUseful for
    each procedure.  
*/
class ManagerICFGCSUseful : public virtual DataFlow::ICFGCSDFProblem
{ 
public:
  ManagerICFGCSUseful(OA_ptr<ActivityIRInterface> _ir);
  ~ManagerICFGCSUseful () {}

  OA_ptr<InterUseful> 
  performAnalysis(OA_ptr<ICFG::ICFGInterface> icfg,
                  OA_ptr<Alias::Interface> Alias,// should be a CSAlias
                  OA_ptr<Alias::CCSetPerProc> ccResults,
                  DataFlow::DFPImplement algorithm);


  class sLocalMREVisitor : public OA::MemRefExprVisitor {
      public:
         sLocalMREVisitor();
         ~sLocalMREVisitor();
         bool isLocal();
         void visitNamedRef(OA::NamedRef& ref);
         void visitUnnamedRef(OA::UnnamedRef& ref);
         void visitUnknownRef(OA::UnknownRef& ref);
         void visitDeref(OA::Deref& ref);
         void visitAddressOf(OA::AddressOf& ref);
         void visitSubSetRef(OA::SubSetRef& ref);
      private:
         bool sLocal;
  };


private:
  //========================================================
  // implementation of ICFGCSDFProblem interface
  //========================================================
  //--------------------------------------------------------
  // initialization callbacks
  //--------------------------------------------------------

  //! Return an initialized top set
  OA_ptr<DataFlow::DataFlowSet>
  initializeTop(OA_ptr<Alias::CallContext> cc);
  
  //! Should generate an in and out DataFlowSet for node
  OA_ptr<DataFlow::DataFlowSet> 
  initializeNodeIN(OA_ptr<ICFG::NodeInterface> n,
                   OA_ptr<Alias::CallContext> cc);

  OA_ptr<DataFlow::DataFlowSet> 
  initializeNodeOUT(OA_ptr<ICFG::NodeInterface> n,
                    OA_ptr<Alias::CallContext> cc);
 
  //--------------------------------------------------------
  // solver callbacks 
  //--------------------------------------------------------
  
  //! ===== Ok to modify set1 and set2 ===== 
  OA_ptr<DataFlow::DataFlowSet> 
  meet(OA_ptr<DataFlow::DataFlowSet> set1, 
       OA_ptr<DataFlow::DataFlowSet> set2,
       OA_ptr<Alias::CallContext> cc); 

  OA_ptr<DataFlow::DataFlowSet> 
  transfer(ProcHandle proc, OA_ptr<DataFlow::DataFlowSet> in, 
           OA_ptr<Alias::CallContext> cc, OA::StmtHandle stmt);


  OA_ptr<DataFlow::DataFlowSet>
  transfer(ProcHandle proc, OA_ptr<DataFlow::DataFlowSet> in,
           OA_ptr<Alias::CallContext> cc, OA::CallHandle call);

  
  //! transfer function for the entry node of the given procedure
  //! should manipulate incoming data-flow set in any special ways
  //! for procedure and return outgoing data-flow set for node
  OA_ptr<DataFlow::DataFlowSet> 
  entryTransfer(ProcHandle proc, OA_ptr<DataFlow::DataFlowSet> in,
                OA_ptr<Alias::CallContext> cc);

  //! transfer function for the exit node of the given procedure
  //! should manipulate outgoing data-flow set in any special ways
  //! for procedure and return incoming data-flow set for node
  OA_ptr<DataFlow::DataFlowSet> 
  exitTransfer(ProcHandle proc, OA_ptr<DataFlow::DataFlowSet> out,
               OA_ptr<Alias::CallContext> cc);

  //! Propagate a data-flow set from caller to callee
  OA_ptr<DataFlow::DataFlowSet> 
  callerToCallee(ProcHandle caller, OA_ptr<DataFlow::DataFlowSet> dfset, 
                 OA_ptr<Alias::CallContext> cc, 
                 CallHandle call, ProcHandle callee);
  
  //! Propagate a data-flow set from callee to caller
  OA_ptr<DataFlow::DataFlowSet> 
  calleeToCaller(ProcHandle callee, OA_ptr<DataFlow::DataFlowSet> dfset,
                 OA_ptr<Alias::CallContext> cc,
                 CallHandle call, ProcHandle caller);

  //! Propagate a data-flow set from call node to return node
  OA_ptr<DataFlow::DataFlowSet> 
  callToReturn(ProcHandle caller, OA_ptr<DataFlow::DataFlowSet> dfset,
               OA_ptr<Alias::CallContext> cc,
               CallHandle call, ProcHandle callee);

  //! --------------------------------------------------------
  //! solver callbacks for DEBUGGING
  //! --------------------------------------------------------

  //! ===== Only the problem itself knows what type of DataFlowSet is being
  //! used, and how to print it ... ICFGCSSolver does not know that info
  void printDFSet(std::ostream& os, OA_ptr<DataFlow::DataFlowSet>);

  //! ===== ICFGCSSolver cannot print CallContexts ... (no IRInterface)
  //! so, ICFGCSDFProblem needs to do it.
  void printCallContext(std::ostream& os, OA_ptr<Alias::CallContext> cc);

private:
  OA_ptr<InterUseful> mInterUseful;
  OA_ptr<ActivityIRInterface> mIR;
  OA_ptr<Alias::Interface> mCSAlias;
  int mMaxNumTags;
  OA_ptr<DataFlow::ICFGCSDFSolver> mCSSolver;
  OA_ptr<ICFG::ICFGInterface> mICFG;
  OA_ptr<Alias::CCSetPerProc> mCCSetPerProc;
  std::map<ProcHandle,OA_ptr<CSUseful> > mCSUsefulMap;

  //! Helper routine for precomputing data-flow sets for each statement.
  void precalcDefsAndUses(OA_ptr<ICFG::ICFGInterface> icfg);

  OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> >  
  calcDefs(OA_ptr<OA::AssignPairIterator> aPairIter,
           OA_ptr<Alias::CallContext> cc);

  OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> >        
  calcMustDefs(OA_ptr<OA::AssignPairIterator> aPairIter,
               OA_ptr<Alias::CallContext> cc);

  OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > 
  calcUses(OA_ptr<OA::AssignPairIterator> aPairIter,
           OA_ptr<Alias::CallContext> cc);
   
  // Places to keep the precomputing data-flow sets.
  std::map<Alias::StmtHwContext,
           OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > > mStmtwCToDefs;
  std::map<Alias::StmtHwContext,
           OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > > mStmtwCToMustDefs;
  std::map<Alias::StmtHwContext,
           OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > > mStmtwCToUses;

  std::map<Alias::CallHwContext,
           OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > > mCallwCToDefs;
  std::map<Alias::CallHwContext,
           OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > > mCallwCToMustDefs;
  std::map<Alias::CallHwContext,
           OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > > mCallwCToUses;

};

  } // end of Activity namespace
} // end of OA namespace

#endif
