/*! \file
  
  \brief Declarations of the AnnotationManager that generates Activity
         analysis results for each procedure using interprocedural results
         and context sensitivity.

  \authors Michelle Strout, Barbara Kreaseck

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#ifndef ManagerICFGCSActive_h
#define ManagerICFGCSActive_h

//--------------------------------------------------------------------
// OpenAnalysis headers

//#include <OpenAnalysis/Activity/ManagerActiveStandard.hpp>
//#include <OpenAnalysis/Activity/ManagerICFGDep.hpp>
#include <OpenAnalysis/Activity/ManagerICFGCSUseful.hpp>
#include <OpenAnalysis/Activity/ManagerICFGCSVaryActive.hpp>
#include <OpenAnalysis/Activity/ActivePerStmt.hpp>
#include <OpenAnalysis/IRInterface/ActivityIRInterface.hpp>
#include <OpenAnalysis/Activity/InterActiveFortran.hpp>
#include <OpenAnalysis/CallGraph/CallGraphInterface.hpp>
#include <OpenAnalysis/Alias/Interface.hpp>
#include <OpenAnalysis/Alias/CCSetPerProc.hpp>
#include <OpenAnalysis/Alias/ProcHwContext.hpp>


namespace OA {
  namespace Activity {


/*! 
   Generates InterActive.
*/
class ManagerICFGCSActive : public virtual DataFlow::ICFGCSDFProblem {
public:
  ManagerICFGCSActive(OA_ptr<Activity::ActivityIRInterface> _ir);
  ~ManagerICFGCSActive () {}

  OA_ptr<InterActiveFortran> performAnalysis(
          OA_ptr<ICFG::ICFGInterface> icfg,
          OA_ptr<Alias::Interface> csalias,
          OA_ptr<Alias::CCSetPerProc> ccResults,
          DataFlow::DFPImplement algorithm);

private:
  //========================================================
  // implementation of ICFGDFProblem interface
  //========================================================
  //--------------------------------------------------------
  // initialization callbacks
  //--------------------------------------------------------

  //! Return an initialized top set
  OA_ptr<DataFlow::DataFlowSet>  
  initializeTop(OA_ptr<Alias::CallContext> cc);
  
  //! Should generate an in and out DataFlowSet for node
  OA_ptr<DataFlow::DataFlowSet> 
  initializeNodeIN(OA_ptr<ICFG::NodeInterface> n,
                   OA_ptr<Alias::CallContext> cc);
  OA_ptr<DataFlow::DataFlowSet> 
  initializeNodeOUT(OA_ptr<ICFG::NodeInterface> n,
                    OA_ptr<Alias::CallContext> cc);
 
  //--------------------------------------------------------
  // solver callbacks 
  //--------------------------------------------------------
  
  //! OK to modify set1 and return it as result, because solver
  //! only passes a tempSet in as set1
  OA_ptr<DataFlow::DataFlowSet> meet(OA_ptr<DataFlow::DataFlowSet> set1, 
                                     OA_ptr<DataFlow::DataFlowSet> set2,
                                     OA_ptr<Alias::CallContext> cc); 

  //! OK to modify in set and return it again as result because
  //! solver clones the BB in sets
  OA_ptr<DataFlow::DataFlowSet> 
  transfer(ProcHandle proc, OA_ptr<DataFlow::DataFlowSet> in,
           OA_ptr<Alias::CallContext> cc, OA::StmtHandle stmt);
 

  OA_ptr<DataFlow::DataFlowSet>
  transfer(ProcHandle proc, OA_ptr<DataFlow::DataFlowSet> in,
           OA_ptr<Alias::CallContext> cc, OA::CallHandle call);


 
  //! transfer function for the entry node of the given procedure
  //! should manipulate incoming data-flow set in any special ways
  //! for procedure and return outgoing data-flow set for node
  OA_ptr<DataFlow::DataFlowSet> 
  entryTransfer(ProcHandle proc, OA_ptr<DataFlow::DataFlowSet> in,
                OA_ptr<Alias::CallContext> cc);

  //! transfer function for the exit node of the given procedure
  //! should manipulate outgoing data-flow set in any special ways
  //! for procedure and return incoming data-flow set for node
  OA_ptr<DataFlow::DataFlowSet> 
  exitTransfer(ProcHandle proc, OA_ptr<DataFlow::DataFlowSet> out,
               OA_ptr<Alias::CallContext> cc);

  //! Propagate a data-flow set from caller to callee
  OA_ptr<DataFlow::DataFlowSet> 
  callerToCallee(ProcHandle caller, OA_ptr<DataFlow::DataFlowSet> dfset,
                 OA_ptr<Alias::CallContext> cc,
                 CallHandle call, ProcHandle callee);
  
  //! Propagate a data-flow set from callee to caller
  OA_ptr<DataFlow::DataFlowSet> 
  calleeToCaller(ProcHandle callee, OA_ptr<DataFlow::DataFlowSet> dfset, 
                 OA_ptr<Alias::CallContext> cc,
                 CallHandle call, ProcHandle caller);

 //! Propagate a data-flow set from call node to return node
 OA_ptr<DataFlow::DataFlowSet> 
 callToReturn(ProcHandle caller, OA_ptr<DataFlow::DataFlowSet> dfset,
              OA_ptr<Alias::CallContext> cc,
              CallHandle call, ProcHandle callee);

  //! --------------------------------------------------------
  //! solver callbacks for DEBUGGING
  //! --------------------------------------------------------

  //! ===== Only the problem itself knows what type of DataFlowSet is being
  //! used, and how to print it ... ICFGCSSolver does not know that info
  void printDFSet(std::ostream& os, OA_ptr<DataFlow::DataFlowSet>);

  //! ===== ICFGCSSolver cannot print CallContexts ... (no IRInterface)
  //! so, ICFGCSDFProblem needs to do it.
  void printCallContext(std::ostream& os, OA_ptr<Alias::CallContext> cc);

private: // member variables

  OA_ptr<Activity::ActivityIRInterface> mIR;
  std::map<ProcHandle,OA_ptr<ActiveStandard> > mActiveMap;

  // need set of active alias tags per procedure
  std::map<Alias::ProcHwContext,OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > > mActiveSet;

  OA_ptr<Alias::Interface> mCSAlias;
  int mMaxTags;
  OA_ptr<DataFlow::ICFGCSDFSolver> mCSSolver;
  OA_ptr<ActivePerStmt> mCSActive;

};

  } // end of Activity namespace
} // end of OA namespace

#endif
