/*! \file
  
  \brief The AnnotationManager that generates InterActive INTERprocedurally
         with context sensitivity.

  \authors Michelle Strout, Barbara Kreaseck

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#include <iostream>
#include <OpenAnalysis/Activity/ManagerICFGCSActive.hpp>
#include <Utils/Util.hpp>

#include <sys/time.h>

namespace OA {
  namespace Activity {

static bool debug = false;

ManagerICFGCSActive::ManagerICFGCSActive(
    OA_ptr<Activity::ActivityIRInterface> _ir) : mIR(_ir)
{
    OA_DEBUG_CTRL_MACRO("DEBUG_ManagerICFGCSActive:ALL", debug);
    mCSSolver = new DataFlow::ICFGCSDFSolver(DataFlow::ICFGCSDFSolver::Backward,*this);
}

/*!
*/
OA_ptr<InterActiveFortran> 
ManagerICFGCSActive::performAnalysis(
        OA_ptr<ICFG::ICFGInterface> icfg,
        OA_ptr<Alias::Interface> csalias,
        OA_ptr<Alias::CCSetPerProc> ccResults,
        DataFlow::DFPImplement algorithm)
{
  OA_ptr<InterActiveFortran> retval;
  retval = new InterActiveFortran();

  if (debug) { csalias->output(*mIR); }

  //! ManagerICFGCSUseful does useful analysis and determines which
  //! AliasTags are useful coming Into a stmt and Outof a stmt
  if (debug) {
      std::cout << "Calling csusefulman->performAnalysis() ...\n";
      std::cout.flush();
  }

  OA_ptr<ManagerICFGCSUseful> csusefulman;
  csusefulman = new ManagerICFGCSUseful(mIR);
  OA_ptr<InterUseful> csInterUseful;
  csInterUseful = csusefulman->performAnalysis(icfg, csalias, ccResults,
                                               algorithm);
  retval->setNumIterUseful(csInterUseful->getNumIter());
  if (debug) { csInterUseful->output(*mIR); }
 
  
  // ManagerICFGCSVaryActive does vary analysis and determines which
  // locations are active coming Into a stmt and Outof a stmt
  if (debug) {
      std::cout << "Calling csvaryman->performAnalysis() ...\n";
      std::cout.flush();
  }

  OA_ptr<ManagerICFGCSVaryActive> csvaryman;
  csvaryman = new ManagerICFGCSVaryActive(mIR);
  OA_ptr<ActivePerStmt> csActive;
  csActive= csvaryman->performAnalysis(icfg, csalias, ccResults,
                                       csInterUseful, algorithm);

  retval->setNumIterVary(csActive->getNumIter());
  if (debug) { csActive->output(*mIR); }
  
  
  //-------------------------------------------------------------
  // Do backward dataflow analysis to determine which def memrefs
  // and stmts are active
  
  // store results that will be needed in callbacks
  mCSAlias = csalias;
  mMaxTags = mCSAlias->getMaxAliasTag().val();
  mCSActive = csActive;

  // call iterative data-flow solver for ICFG
  mCSSolver->solve(icfg,ccResults,algorithm);

  // iterate over all active alias tags and generate the set of active
  // SymHandles
  // for each procedure
  std::map<Alias::ProcHwContext, OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > >::iterator mapIter;
  for (mapIter=mActiveSet.begin(); mapIter!=mActiveSet.end(); mapIter++) {
    Alias::ProcHwContext phwc = mapIter->first;
    ProcHandle proc = phwc.getProc();
    OA_ptr<Alias::CallContext> cc = phwc.getCallContext();

    // get alias and active results for current procedure
    OA_ptr<ActiveStandard> active = mActiveMap[proc];

    OA_ptr<DataFlow::DFSetBitImplIterator<Alias::AliasTag> > tagIter;
    tagIter = new DataFlow::DFSetBitImplIterator<Alias::AliasTag>(mapIter->second);
    for ( ; tagIter->isValid(); (*tagIter)++ ) {
        active->insertTag(tagIter->current(),cc);
        OA_ptr<MemRefExprIterator> mreIter
            = mCSAlias->getMemRefExprIterator(tagIter->current());
     
        for( ; mreIter->isValid(); ++(*mreIter) ) {
            OA_ptr<MemRefExpr> mre 
                = mreIter->current();
            if(mre->isaNamed()) {
              active->insertSym(mre.convert<NamedRef>()->getSymHandle(),cc);
            }
            // ASSUMPTION: doing this for FORTRAN reference params
            // Also subseting I guess.
            if(mre->isaRefOp()) {
               while(!mre->isaNamed()) {
                   mre = mre.convert<RefOp>()->getMemRefExpr();
               }
               active->insertSym(mre.convert<NamedRef>()->getSymHandle(),cc);
            }
        }
    }

  }

  // assign activity results for each procedure into InterActive

  std::map<ProcHandle,OA_ptr<ActiveStandard> >::iterator pmapIter;
  for (pmapIter=mActiveMap.begin(); pmapIter!=mActiveMap.end(); pmapIter++) {
    retval->mapProcToActive(pmapIter->first,pmapIter->second);

    if (debug) {
      std::cout << "Mapping ActiveStandard for Proc " 
                << mIR->toString(pmapIter->first)
                << " ============\n";
      (pmapIter->second)->dump(std::cout, mIR);
      std::cout << std::endl;
    }
  }

  // how many bytes are active?
  // iterate over all symbols to determine the size
  OA_ptr<SymHandleIterator> symIter = retval->getActiveSymIterator();
  int bytes = 0;

  for ( ; symIter->isValid(); (*symIter)++ ) {
    SymHandle sym = symIter->current();

    if (!(mIR->isRefParam(sym))) {
      bytes += mIR->getSizeInBytes(sym);
    }

    if (true) {
      std::cout << "ManagerICFGCSActive: sym = " << mIR->toString(sym)
                << ", size = <" << mIR->getSizeInBytes(sym)
                << ">, bytes = " << bytes;
      if (mIR->isRefParam(sym)) {
        std::cout << ", refParam";
      }
      std::cout << std::endl;
    }

  }

  retval->setActiveSizeInBytes(bytes);

  // how many iterations did this take?
  if (debug) 
    {
      std::cout << "ICFGCSActive took " << mCSSolver->getNumIter()
                << " iterations.\n";
    }

  retval->setNumIterActive(mCSSolver->getNumIter());

  return retval;
}

//========================================================
// implementation of ICFGDFProblem interface
//========================================================
//--------------------------------------------------------
// initialization callbacks
//--------------------------------------------------------

/*!
   Data-flow set passed around on the icfg is a
   LocDFSet.  The top value for this is an empty set.
*/
OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSActive::initializeTop(OA_ptr<Alias::CallContext> cc)
{
  OA_ptr<OA::DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
  retval = new OA::DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxTags); 
  return retval;
}

OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSActive::initializeNodeIN(OA_ptr<ICFG::NodeInterface> n,
                                      OA_ptr<Alias::CallContext> cc)
{
  OA_ptr<OA::DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
  retval = new OA::DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxTags);
  return retval;
}

OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSActive::initializeNodeOUT(OA_ptr<ICFG::NodeInterface> n,
                                       OA_ptr<Alias::CallContext> cc)
{
  OA_ptr<OA::DataFlow::DFSetBitImpl<Alias::AliasTag> > retval;
  retval = new OA::DataFlow::DFSetBitImpl<Alias::AliasTag>(mMaxTags);
  return retval;
}


//--------------------------------------------------------
// solver callbacks 
//--------------------------------------------------------
  
//! OK to modify set1 and return it as result, because solver
//! only passes a tempSet in as set1
OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSActive::meet(OA_ptr<DataFlow::DataFlowSet> set1, 
                          OA_ptr<DataFlow::DataFlowSet> set2,
                          OA_ptr<Alias::CallContext> cc)
{
    // just return in set
    return set1;
}

//! OK to modify in set and return it again as result because
//! solver clones the BB in sets
OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSActive::transfer(ProcHandle proc,
                              OA_ptr<DataFlow::DataFlowSet> out,
                              OA_ptr<Alias::CallContext> cc,
                              OA::StmtHandle stmt)
{  
    // will be storing activity results for stmt and memrefs in
    // ActiveStandard for current procedure
    if (mActiveMap[proc].ptrEqual(0)) {
        mActiveMap[proc] = new ActiveStandard(proc);
    }
    // get alias and active results for current procedure
    OA_ptr<ActiveStandard> active = mActiveMap[proc];

    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > inActiveSet, outActiveSet;

    // Get the InActive Set for Statement for this call context
    inActiveSet = mCSActive->getInActiveSet(stmt,cc);
    // Get the OutActive Set for the Statement for this call context
    outActiveSet = mCSActive->getOutActiveSet(stmt,cc);

    //BK: How are CallStmts handled correctly here. ==> see transfer(CallHandle ...
    //    The correct inActiveSet for a stmt that has a CallHandle should be the
    //    inSet for the CallNode.  However, the CallStmt is now stored in the
    //    ReturnNode, and we cannot use the inSet for the ReturnNode, as it is
    //    the meet of all of the outSets of the ExitNodes that have ReturnEdges
    //    incoming to the ReturnNode. Hmmm... leaving this for another day...
    //    FIXME?

    
    Alias::ProcHwContext phwc(proc,cc);

    // insert both of the above sets into the active alias tags for
    // this procedure
    if (mActiveSet[phwc].ptrEqual(0)) {
      mActiveSet[phwc] = new DataFlow::DFSetBitImpl<Alias::AliasTag>(*inActiveSet);
      mActiveSet[phwc]->unionEqu(*outActiveSet);
    } else {
      mActiveSet[phwc]->unionEqu(*inActiveSet);
      mActiveSet[phwc]->unionEqu(*outActiveSet);
    }





    // FIXME: MMS, 8/4/08, for now not going to have any active statements
    //if((!inActiveSet->isEmpty()) && (!diffSet->isEmpty())) {
    //    active->insertStmt(stmt);
   // }

/* moved to performAnalysis
   OA_ptr<DataFlow::DFSetBitImplIterator<Alias::AliasTag> > tagIter;
    tagIter = new DataFlow::DFSetBitImplIterator<Alias::AliasTag>(inActiveSet);
    for ( ; tagIter->isValid(); (*tagIter)++ ) {
        active->insertTag(tagIter->current());
        OA_ptr<MemRefExprIterator> mreIter
            = mAlias->getMemRefExprIterator(tagIter->current());
     
        for( ; mreIter->isValid(); ++(*mreIter) ) {
            OA_ptr<MemRefExpr> mre 
                = mreIter->current();
            if(mre->isaNamed()) {
               active->insertSym(mre.convert<NamedRef>()->getSymHandle());
            }
            if(mre->isaRefOp()) {
               while(!mre->isaNamed()) {
                   mre = mre.convert<RefOp>()->getMemRefExpr();
               }
               active->insertSym(mre.convert<NamedRef>()->getSymHandle());
            }
        }
    }

    diffSet->minusEqu(*inActiveSet);
    tagIter = new DataFlow::DFSetBitImplIterator<Alias::AliasTag>(diffSet);
    for ( ; tagIter->isValid(); (*tagIter)++ ) {
        active->insertTag(tagIter->current());
        OA_ptr<MemRefExprIterator> mreIter
            = mAlias->getMemRefExprIterator(tagIter->current());
        for( ; mreIter->isValid(); ++(*mreIter) )
        {
           OA_ptr<MemRefExpr> mre
               = mreIter->current();
           if(mre->isaNamed()) {
              active->insertSym(mre.convert<NamedRef>()->getSymHandle());
           }
           if(mre->isaRefOp()) {
              while(!mre->isaNamed()) {
                   mre = mre.convert<RefOp>()->getMemRefExpr(); 
              }
              active->insertSym(mre.convert<NamedRef>()->getSymHandle());
           }
        }
    }
*/
    if (debug) {
        std::cout << "In transfer, stmt(hval=" << stmt.hval() << ")= ";
        std::cout << mIR->toString(stmt) << std::endl;
        std::cout << "With cc: ";
        cc->output(*mIR);
        std::cout << "\tInActive = ";
        inActiveSet->output(*mIR);
        std::cout << "\tOutActive = ";
        outActiveSet->output(*mIR);
        std::cout << std::endl;
    }

    return out;
}

//! OK to modify in set and return it again as result because
//! solver clones the BB in sets
OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSActive::transfer(ProcHandle proc,
                              OA_ptr<DataFlow::DataFlowSet> in,
                              OA_ptr<Alias::CallContext> cc,
                              OA::CallHandle call)
{  

  // BK: Shouldn't we need to do something here? Or is it all subsumed
  //     within the transfer over the statement that contains the call???

  return in;
}


/*!
   Will get the indep vars if this procedure has any and add them
   to incoming LocDFSet
*/
OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSActive::entryTransfer(ProcHandle proc, 
                                   OA_ptr<DataFlow::DataFlowSet> in,
                                   OA_ptr<Alias::CallContext> cc)
{
    return in;
}

/*!
*/
OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSActive::exitTransfer(ProcHandle proc, 
                                  OA_ptr<DataFlow::DataFlowSet> out,
                                  OA_ptr<Alias::CallContext> cc)
{
    return out;
}


//! Propagate a data-flow set from caller to callee
OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSActive::callerToCallee(ProcHandle caller,
                                    OA_ptr<DataFlow::DataFlowSet> dfset,
                                    OA_ptr<Alias::CallContext> cc,
                                    CallHandle call, ProcHandle callee)
{
  // no data flow for this analysis ... returning in set
  return dfset;
}
  
//! Propagate a data-flow set from callee to caller
OA_ptr<DataFlow::DataFlowSet> 
ManagerICFGCSActive::calleeToCaller(ProcHandle callee,
                                    OA_ptr<DataFlow::DataFlowSet> dfset,
                                    OA_ptr<Alias::CallContext> cc,
                                    CallHandle call, ProcHandle caller)
{
  // no data flow for this analysis ... returning in set
  return dfset;
}

//! Propagate a data-flow set from call node to return node
OA_ptr<DataFlow::DataFlowSet>
ManagerICFGCSActive::callToReturn(ProcHandle caller,
                                  OA_ptr<DataFlow::DataFlowSet> dfset,
                                  OA_ptr<Alias::CallContext> cc,
                                  CallHandle call, ProcHandle callee)
{
  // no data flow for this analysis ... returning in set
  return dfset;
}


void 
ManagerICFGCSActive::printCallContext(std::ostream& os, 
                                      OA_ptr<Alias::CallContext> cc)
{
  cc->dump(os,*mIR);
}


void 
ManagerICFGCSActive::printDFSet(std::ostream& os, 
                                OA_ptr<DataFlow::DataFlowSet> dfs)
{
    OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > dfsRecast;
    dfsRecast = dfs->clone().convert<DataFlow::DFSetBitImpl<Alias::AliasTag> >();
    dfsRecast->output(*mIR);
}


  } // end of namespace Activity
} // end of namespace OA
