/*! \file
  
  \brief Maps StmtHandles and CallContexts (and CallHandles and CallContexts)
         to AliasTagSets that are in- or out-useful.

  \authors Michelle Strout, Barbara Kreaseck

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#ifndef CSUseful_hpp
#define CSUseful_hpp

#include <cassert>
#include <iostream>
#include <map>
#include <set>
#include <vector>
#include <OpenAnalysis/Utils/OA_ptr.hpp>
#include <OpenAnalysis/IRInterface/IRHandles.hpp>
#include <OpenAnalysis/OABase/Annotation.hpp>
//#include <OpenAnalysis/DataFlow/DataFlowSetImpl.hpp>
#include <OpenAnalysis/DataFlow/DFSetBitImpl.hpp>
#include <OpenAnalysis/Alias/Interface.hpp>
#include <OpenAnalysis/Alias/StmtHwContext.hpp>
#include <OpenAnalysis/Alias/CallHwContext.hpp>

namespace OA {
  namespace Activity {


class CSUseful : public virtual Annotation {
  public:
    CSUseful(ProcHandle p); 
    ~CSUseful() {}


    //! ===== StmtHandle =====

    //! Return an iterator that will iterate over the set of AliasTags
    //! that are in InUseful set for stmt/CallContext
    //! (above the stmt, code-wise)
    OA_ptr<DataFlow::DFSetBitImplIterator<Alias::AliasTag> >
    getInUsefulIterator(StmtHandle s, OA_ptr<Alias::CallContext> cc);

    //! Return an iterator that will iterate over the AliasTagSet
    //! that are in OutUseful set for stmt/CC (below the stmt, code-wise)
    OA_ptr<DataFlow::DFSetBitImplIterator<Alias::AliasTag> > 
    getOutUsefulIterator(StmtHandle s, OA_ptr<Alias::CallContext> cc);


    //! ===== CallHandle =====
    
    //! Return an iterator that will iterate over the AliasTagSet
    //! that are in InUseful set for call/CC (above the call, code-wise)
    OA_ptr<DataFlow::DFSetBitImplIterator<Alias::AliasTag> >
    getInUsefulIterator(CallHandle c, OA_ptr<Alias::CallContext> cc);

    //! Return an iterator that will iterate over the AliasTagSet
    //! that are in OutUseful set for call/CC (below the call, code-wise)
    OA_ptr<DataFlow::DFSetBitImplIterator<Alias::AliasTag> >
    getOutUsefulIterator(CallHandle c, OA_ptr<Alias::CallContext> cc);
 

    //! access the number of iterations over all nodes for Useful
    int getNumIter();

    //*****************************************************************
    // Construction methods 
    //*****************************************************************


    //! copy into the InUseful set for a statement
    void copyIntoInUseful(StmtHandle s, OA_ptr<Alias::CallContext> cc,
                          OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > dfset);

    //! copy into the OutUseful set for a statement
    void copyIntoOutUseful(StmtHandle s, OA_ptr<Alias::CallContext> cc,
                           OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > dfset);

    //! copy into the CallOutUseful set for a call
    void copyIntoCallInUseful(CallHandle c, OA_ptr<Alias::CallContext> cc,
                              OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > dfset);

    //! copy into the CallOutUseful set for a call
    void copyIntoCallOutUseful(CallHandle c, OA_ptr<Alias::CallContext> cc,
                               OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > dfset);

    //!  set the number of iterations over all nodes for Useful
    void setNumIter(int n);

      
    //*****************************************************************
    // Output
    //*****************************************************************

    //! incomplete output of info for debugging, just lists stmts
    //! and associated set of InUseful locations
    void dump(std::ostream& os, OA_ptr<IRHandlesIRInterface> ir);

    //*****************************************************************
    // Annotation Interface
    //*****************************************************************
    void output(IRHandlesIRInterface &ir) const;

  private:
    // data members
      std::map<Alias::StmtHwContext,OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > >  mInUseful;
      std::map<Alias::StmtHwContext,OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > >  mOutUseful;
      std::map<Alias::CallHwContext,OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > >  mCallInUseful;
      std::map<Alias::CallHwContext,OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> > >  mCallOutUseful;
    //OA_ptr<DataFlow::DFSetBitImpl<Alias::AliasTag> >  mDepTagSet;
    int mNumIter;

};

  } // end of Activity namespace
} // end of OA namespace

#endif

