/*! \file
  
  \brief Maps procedures to a set of locations, statements, and memory
         references in the stmt that are active, w/poss. CS results as well

  \authors Michelle Strout, Barbara Kreaseck
  \version $Id: ActiveStandard.hpp,v 1.6 2005/06/10 02:32:01 mstrout Exp $

  Copyright (c) 2002-2005, Rice University <br>
  Copyright (c) 2004-2005, University of Chicago <br>
  Copyright (c) 2006, Contributors <br>
  All rights reserved. <br>
  See ../../../Copyright.txt for details. <br>
*/

#ifndef ActiveStandard_hpp
#define ActiveStandard_hpp

#include <cassert>
#include <iostream>
#include <list>
#include <map>
#include <OpenAnalysis/Utils/OA_ptr.hpp>
#include <OpenAnalysis/IRInterface/IRHandles.hpp>
#include <OpenAnalysis/DataFlow/DataFlowSetImpl.hpp>
#include <OpenAnalysis/Alias/Interface.hpp>
#include <OpenAnalysis/Alias/CallContext.hpp>

namespace OA {
  namespace Activity {


class ActiveStandard {
  public:
    ActiveStandard(ProcHandle p);
    ~ActiveStandard() {}

    //! Indicates whether Context-Sensitive results are available
    bool hasCSResults();

    //! Return an iterator for set of active symbols
    //!FIXME: not really possible unless we know all visible symbols
    //!because could have the Unknown location
    //!be active and under that circumstance all symbols are active
    OA_ptr<SymHandleIterator>
    getActiveSymIterator();
    OA_ptr<SymHandleIterator> 
    getActiveSymIterator(OA_ptr<Alias::CallContext>cc);

    //! FIXME: will eventually need a routine to indicate what statements
    //! dynamically allocate memory that is active
    //getActiveUnnamedIterator

    //! Indicate whether the given symbol is active or not
    bool isActive(SymHandle sym);
    bool isActive(SymHandle sym, OA_ptr<Alias::CallContext> cc);

    //! Indicate whether the given stmt is active or not
    bool isActive(StmtHandle stmt);
    bool isActive(StmtHandle stmt, OA_ptr<Alias::CallContext> cc);

    //! Indicate whether the given memref is active or not
    bool isActive(MemRefHandle memref);
    bool isActive(MemRefHandle memref, OA_ptr<Alias::CallContext> cc);
    
    //! Return an iterator for set of active stmts
    OA_ptr<StmtHandleIterator> 
    getActiveStmtIterator();
    OA_ptr<StmtHandleIterator> 
    getActiveStmtIterator(OA_ptr<Alias::CallContext> cc);

    //! Return an iterator for set of active memory references
    OA_ptr<MemRefHandleIterator> 
    getActiveMemRefIterator();
    OA_ptr<MemRefHandleIterator> 
    getActiveMemRefIterator(OA_ptr<Alias::CallContext> cc);

    //! Get an iterator over active locations
    //OA_ptr<LocIterator> getActiveLocsIterator();
    OA_ptr<OA::DataFlow::DataFlowSetImplIterator<Alias::AliasTag> > 
    getActiveTagsIterator();
    OA_ptr<OA::DataFlow::DataFlowSetImplIterator<Alias::AliasTag> > 
    getActiveTagsIterator(OA_ptr<Alias::CallContext> cc);

    //*****************************************************************
    // Construction methods 
    //*****************************************************************

    //! insert an active location
    //void insertLoc(OA_ptr<Location> loc);
    void insertTag(Alias::AliasTag tag);
    void insertTag(Alias::AliasTag tag, OA_ptr<Alias::CallContext> cc);

    //! insert an active Stmt
    void insertStmt(StmtHandle stmt); 
    void insertStmt(StmtHandle stmt, OA_ptr<Alias::CallContext> cc); 

    //! insert an active MemRef
    void insertMemRef(MemRefHandle memref);
    void insertMemRef(MemRefHandle memref, OA_ptr<Alias::CallContext> cc);

    //! insert an active SymHandle
    void insertSym(SymHandle sym);
    void insertSym(SymHandle sym, OA_ptr<Alias::CallContext> cc);
    
    //! For construction of InterActive
    //bool getUnknownLocActive() { return mUnknownLocActive; }

    //*****************************************************************
    // Output
    //*****************************************************************

    //! incomplete output of info for debugging, just lists 
    //! the contents of the three active sets
    void dump(std::ostream& os, OA_ptr<IRHandlesIRInterface> ir);

  private:
    // data members

    OA_ptr<OA::DataFlow::DataFlowSetImpl<Alias::AliasTag> > mActiveTagSet;
    std::map<OA_ptr<Alias::CallContext>, OA_ptr<OA::DataFlow::DataFlowSetImpl<Alias::AliasTag> > > mCCToActiveTagSetMap;

    OA_ptr<std::set<SymHandle> > mActiveSymSet;
    std::map<OA_ptr<Alias::CallContext>, OA_ptr<std::set<SymHandle> > > mCCToActiveSymSetMap;
    bool mUnknownLocActive;
    bool mHasCSResults;

    OA_ptr<std::set<StmtHandle> > mActiveStmtSet;
    std::map<OA_ptr<Alias::CallContext>, OA_ptr<std::set<StmtHandle> > > mCCToActiveStmtSetMap;

    OA_ptr<std::set<MemRefHandle> > mActiveMemRefSet;
    std::map<OA_ptr<Alias::CallContext>, OA_ptr<std::set<MemRefHandle> > > mCCToActiveMemRefSetMap;

};

//! An iterator over Symbols
class ActiveSymIterator : public virtual OA::SymHandleIterator,
                          public OA::IRHandleSetIterator<OA::SymHandle>
{
public:
  ActiveSymIterator(OA::OA_ptr<std::set<OA::SymHandle> > pList) 
    : OA::IRHandleSetIterator<OA::SymHandle>(pList) {}
  ~ActiveSymIterator() {}

  void operator++() { OA::IRHandleSetIterator<OA::SymHandle>::operator++(); }
  bool isValid() const
    { return OA::IRHandleSetIterator<OA::SymHandle>::isValid(); }
  OA::SymHandle current() const
    { return OA::IRHandleSetIterator<OA::SymHandle>::current(); }
  void reset() { OA::IRHandleSetIterator<OA::SymHandle>::current(); }
};


//! An iterator over Memory References 
class ActiveMemRefIterator : public virtual OA::MemRefHandleIterator,
                             public OA::IRHandleSetIterator<OA::MemRefHandle>
{
public:
  ActiveMemRefIterator(OA::OA_ptr<std::set<OA::MemRefHandle> > pList) 
    : OA::IRHandleSetIterator<OA::MemRefHandle>(pList) {}
  ~ActiveMemRefIterator() {}

  void operator++() { OA::IRHandleSetIterator<OA::MemRefHandle>::operator++(); }
  bool isValid() const
    { return OA::IRHandleSetIterator<OA::MemRefHandle>::isValid(); }
  OA::MemRefHandle current() const
    { return OA::IRHandleSetIterator<OA::MemRefHandle>::current(); }
  void reset() { OA::IRHandleSetIterator<OA::MemRefHandle>::current(); }
};


//! An iterator over stmts 
class ActiveStmtIterator : public virtual OA::IRStmtIterator,
                           public OA::IRHandleSetIterator<OA::StmtHandle>
{
public:
  ActiveStmtIterator(OA::OA_ptr<std::set<OA::StmtHandle> > pList) 
    : OA::IRHandleSetIterator<OA::StmtHandle>(pList) {}
  ~ActiveStmtIterator() {}

  void operator++() { OA::IRHandleSetIterator<OA::StmtHandle>::operator++(); }
  bool isValid() const
    { return OA::IRHandleSetIterator<OA::StmtHandle>::isValid(); }
  OA::StmtHandle current() const
    { return OA::IRHandleSetIterator<OA::StmtHandle>::current(); }
  void reset() { OA::IRHandleSetIterator<OA::StmtHandle>::current(); }
};



  } // end of Activity namespace
} // end of OA namespace

#endif

