#ifndef autoTuningSupport_INCLUDED
#define autoTuningSupport_INCLUDED
/*!
 *
 * This namespace contains the  modules for the empirical tuning system
 * It reads into the xml file generated by HPCToolKit and
 *    annotates the AST with performance metrics from the file 
 *
 *  We expect two kinds of xml input from HPCToolkit
 *  1. Raw statement level data without any propagation
 *  2. Correlated data which is already propagated by HPCToolKit, with scope level metrics 
 *
 *  We also accept the line-by-line profiling information generated by GNU gprof -l -L 
 *
 * Liao, 1/28/2009
 */

// rose-hpct headers 
#include "rosehpct/rosehpct.hh"

// Outliner headers
#include "Outliner.hh"

// Generic headers
#include <iostream>
#include <string>
#include <map>
#include <set>
#include <vector>

namespace autoTuning
{
  //! Some internal flags controlled by command line options
  extern bool aggressive_triage;
  extern bool enable_debug;
  extern bool triage_only; // stop right after the code triage phase, do not proceed to outline them
  extern float triage_threshold; // threshold for sum of top statements' execution percentage numbers
  const float DEFAULT_THRESHOLD = 0.8; 

  //! autoTuning specific command line processing
  void autotuning_command_processing(std::vector<std::string>&argvList);

  //! Code triage based on performance annotations
  void code_triage(std::set<SgLocatedNode*>& candidateStmts, std::set<SgForStatement*>& candidateSgLoops);

  //! Check if a node has a performance metric matching a given metric name
  RoseHPCT::MetricAttr * checkMetricAttribute (SgNode* node, const std::string& metric_name);

  //! Generate a file metrics map from profile IR's file nodes
  std::map<std::string, std::map< std::string, double > >
    generateFileMetricsMap(std::set<const RoseHPCT::IRNode *> profFileNodes);

  //! Find the name of the time-consuming source file given a set of profile file nodes
  std::string findHottestFile(std::map<std::string, std::map< std::string, double > > fileMetrics);

  //! Collect available metric names
  std::vector <std::string> collectMetricNames(std::map<std::string, std::map< std::string, double > > fileMetrics);

  //! Generate a node-metrics map for SgLocatedNode within a SgFile
  std::map<SgNode*, std::map< std::string, double > > 
    generateNodeMetricsMap(SgFile* file, std::vector <std::string > metricNames);

  //! Find the hottest non-scope statement within a file from a hot statement
  // This makes sense since HPCToolkit generates accumulative flat profiling metrics for 
  // each statements. 
  // Metrics of statements within loops already consider loop iterations(and function calls)
  SgStatement* findHottestStatement(SgFile* file, std::map<SgNode*, std::map< std::string, double > > nodeMetricsMap);

  //! Find a target loop from a hot statement
  SgForStatement* findTargetLoop(SgNode* stmt);
}

#endif
