// This example program shows how to write out a binary file representing the AST.

#include "rose.h"

// This is OK if not declared in a header file
using namespace std;

// Example of user defined attribute
class UserDefinedAttribute : public AstAttribute
   {
     public:
       // Function interface for supporting File IO. 
       // Required if user-defined attributes are to be reconstructed in File input.
       // But not defined as pure virtual functions in the AstAttribute base class.
          virtual string attribute_class_name() { return "UserDefinedAttribute"; }
          virtual UserDefinedAttribute* constructor() { return new UserDefinedAttribute(); }
          virtual char* packed() { return (char*) this; }
          virtual int packed_size() { return 0; }
          virtual void unpacked( char* ptr ) { }
   };

int
main ( int argc, char * argv[] )
   {
     SgProject* project = frontend(argc,argv);
     ROSE_ASSERT (project != NULL);

  // Get the file name of the source file input on the command line (a.out is default if none is specified)
     string fileName =  project->get_outputFileName();

  // This macro registers any user-defined classes derived from AstAttribute so that
  // they can be rebuilt by mechanism to read the file.  This must be called at any
  // point before the AST_FILE_IO::readASTFromFile() is called.  It is a macro because
  // we need to build the type and the name of the type.
     REGISTER_ATTRIBUTE_FOR_FILE_IO(UserDefinedAttribute);

  // This phase:
  // 1) builds the lists of numbers of IR nodes for each type
  // 2) sets the free pointer in each IR node to a global id number (so that the pointers can be reconstructed)
  // 3) builds list of static data held in the IR nodes (builtin types, global function type symbol table, root of AST)
  // 4) registers all astAttribute objects (base classes of user defined objects attached to the AST)
     AST_FILE_IO::startUp( project ) ;

  // This phase:
  // 1) write out lists of static data
  // 2) for each memory pool (each IR node type)
  //     a) loop over each IR node for selected type (SgStatement, etc.)
  //        *) copy data from each IR node to an entry in a contiguous array 
  //           of associated StorageClass nodes (classes generated by ROSETTA)
  //        *) write out all astArrtibute information (calls user-defined function to pack data)
  //     b) write out single the storage class array (fastest possible binary file I/O)

     AST_FILE_IO::writeASTToFile ( fileName + ".binaryAST" );

  // Delete the memory pools, removing the entire AST from memory (and resetting all static data pointers).
     AST_FILE_IO::clearAllMemoryPools();

  // Rebuild the AST from the save binary file
  // 1) Rebuild the memory pools sizes for the IR nodes from the information saved 
  //    in the file (only applies to AST read in from empty memory pools)
  // 2) Extends the memory pools to hold correct number of allocated IR nodes from AST in file
  // 3) for each memory pool (each IR node type)
  //    a) Read the stored storage classes.
  //    b) Read the static data for each IR node type
  //    c) Loop over array of storage classes
  //       *) rebuild each IR node
  //       *) convert global id values to pointer addresses in memory pools
  //    d) delete the StorageClass array 
  //    e) Save the static data for the IR node type (save in the
  //       AST_FILE_IO::staticDataMemberManagementClass (or some name like that))
     project = (SgProject*) (AST_FILE_IO::readASTFromFile ( fileName + ".binaryAST" ) );

  // Run all the AST consistancy tests
     AstTests::runAllTests(project);

  // Generate the source code from the AST and compile it
     return backend(project);
   }

