function md=setflowequation(md,varargin)
%SETELEMENTSTYPE - associate a solution type to each element
%
%   This routine works like plotmodel: it works with an even number of inputs
%   'hutter','macayeal','pattyn','stokes' and 'fill' are the possible options
%   that must be followed by the corresponding exp file or flags list
%   It can either be a domain file (argus type, .exp extension), or an array of element flags. 
%   If user wants every element outside the domain to be 
%   setflowequationd, add '~' to the name of the domain file (ex: '~Pattyn.exp');
%   an empty string '' will be considered as an empty domain
%   a string 'all' will be considered as the entire domain
%   You can specify the type of coupling, 'penalties' or 'tiling', to use with the input 'coupling'
%
%   Usage:
%      md=setflowequation(md,varargin)
%
%   Example:
%      md=setflowequation(md,'pattyn','Pattyn.exp','macayeal',md.mask.elementonfloatingice,'fill','hutter');
%      md=setflowequation(md,'pattyn','Pattyn.exp',fill','hutter','coupling','tiling');

%some checks on list of arguments
if ((nargin<2) | (nargout~=1)),
	error('setflowequation error message');
end

%Find_out what kind of coupling to use
options=pairoptions(varargin{:});
coupling_method=getfieldvalue(options,'coupling','tiling');
if (~strcmpi(coupling_method,'tiling') & ~strcmpi(coupling_method,'penalties')),
	error('coupling type can only be: tiling or penalties');
end

[hutterflag macayealflag pattynflag stokesflag filltype]=recover_areas(md,varargin{:});

%Flag the elements that has not been flagged as filltype
if strcmpi(filltype,'hutter'),
	hutterflag(find(~macayealflag & ~pattynflag))=1;
elseif strcmpi(filltype,'macayeal'),
	macayealflag(find(~hutterflag & ~pattynflag & ~stokesflag))=1;
elseif strcmpi(filltype,'pattyn'),
	pattynflag(find(~hutterflag & ~macayealflag & ~stokesflag))=1;
end

%check that each element has at least one flag
if any(hutterflag+ macayealflag+pattynflag+stokesflag==0),
	error('setflowequation error message: elements type not assigned, must be specified')
end

%check that each element has only one flag
if any(hutterflag+ macayealflag+pattynflag+stokesflag>1),
	disp('setflowequation warning message: some elements have several types, higher order type is used for them')
	hutterflag(find(hutterflag & macayealflag))=0;
	hutterflag(find(hutterflag & pattynflag))=0;
	macayealflag(find(macayealflag & pattynflag))=0;
end

%Check that no pattyn or stokes for 2d mesh
if (md.dim==2),
	if any(stokesflag | pattynflag)
		error('setflowequation error message: stokes and pattyn elements no allowed in 2d mesh, extrude it first')
	end
end

%Stokes can only be used alone for now:
if any(stokesflag) &any(hutterflag),
	error('setflowequation error message: stokes cannot be used with any other model for now, put stokes everywhere')
end

%add in model who is who
md.flowequation.element_equation=zeros(md.numberofelements,1);

%1: Hutter elements
nodeonhutter=zeros(md.numberofnodes,1);
nodeonhutter(md.elements(find(hutterflag),:))=1;
md.flowequation.element_equation(find(hutterflag))=1;

%2: MacAyeal elements
nodeonmacayeal=zeros(md.numberofnodes,1);
nodeonmacayeal(md.elements(find(macayealflag),:))=1;
md.flowequation.bordermacayeal=nodeonmacayeal;
md.flowequation.element_equation(find(macayealflag))=2;

%3: Pattyn elements
nodeonpattyn=zeros(md.numberofnodes,1);
nodeonpattyn(md.elements(find(pattynflag),:))=1;
md.flowequation.borderpattyn=nodeonpattyn;
md.flowequation.element_equation(find(pattynflag))=3;

%4: Stokes elements
%First modify stokesflag to get rid of elements contrained everywhere (spc + border with pattyn or macayeal)
if any(stokesflag),
	nodeonstokes=zeros(md.numberofnodes,1);
	nodeonstokes(md.elements(find(stokesflag),:))=1;
	fullspcnodes=double((~isnan(md.diagnostic.spcvx)+~isnan(md.diagnostic.spcvy)+~isnan(md.diagnostic.spcvz))==3 | (nodeonpattyn & nodeonstokes));         %find all the nodes on the boundary of the domain without icefront
	fullspcelems=double(sum(fullspcnodes(md.elements),2)==6);         %find all the nodes on the boundary of the domain without icefront
	stokesflag(find(fullspcelems))=0;
end
nodeonstokes=zeros(md.numberofnodes,1);
nodeonstokes(md.elements(find(stokesflag),:))=1;
md.flowequation.borderstokes=nodeonstokes;
md.flowequation.element_equation(find(stokesflag))=4;

%Then complete with NoneApproximation or the other model used if there is no stokes
if any(stokesflag), 
	if any(pattynflag), %fill with pattyn
		pattynflag(~stokesflag)=1;
		nodeonpattyn(md.elements(find(pattynflag),:))=1;
		md.flowequation.borderpattyn=nodeonpattyn;
		md.flowequation.element_equation(find(~stokesflag))=3;
	elseif any(macayealflag), %fill with macayeal
		macayealflag(~stokesflag)=1;
		nodeonmacayeal(md.elements(find(macayealflag),:))=1;
		md.flowequation.bordermacayeal=nodeonmacayeal;
		md.flowequation.element_equation(find(~stokesflag))=2;
	else %fill with none 
	%5: None elements (non Stokes)
		md.flowequation.element_equation(find(~stokesflag))=0;
	end
end

%Now take care of the coupling between MacAyeal and Pattyn
md.diagnostic.vertex_pairing=[];
nodeonmacayealpattyn=zeros(md.numberofnodes,1);
nodeonpattynstokes=zeros(md.numberofnodes,1);
nodeonmacayealstokes=zeros(md.numberofnodes,1);
if strcmpi(coupling_method,'penalties'),
	%Create the border nodes between Pattyn and MacAyeal and extrude them
	numnodes2d=md.numberofnodes2d;
	numlayers=md.numlayers;
	bordernodes2d=find(nodeonpattyn(1:numnodes2d) & nodeonmacayeal(1:numnodes2d)); %Nodes connected to two different types of elements

	%initialize and fill in penalties structure
	if ~isnan(bordernodes2d),
		penalties=[];
		for	i=1:numlayers-1,
			penalties=[penalties; [bordernodes2d bordernodes2d+md.numberofnodes2d*(i)]];
		end
		md.diagnostic.vertex_pairing=penalties;
	end
elseif strcmpi(coupling_method,'tiling'),
	if any(macayealflag) & any(pattynflag), %coupling macayeal pattyn
		%Find node at the border
		nodeonmacayealpattyn(find(nodeonmacayeal & nodeonpattyn))=1;
		%Macayeal elements in contact with this layer become MacAyealPattyn elements
		matrixelements=ismember(md.elements,find(nodeonmacayealpattyn));
		commonelements=sum(matrixelements,2)~=0;
		commonelements(find(pattynflag))=0; %only one layer: the elements previously in macayeal
		macayealflag(find(commonelements))=0; %these elements are now macayealpattynelements
		macayealpattynflag=zeros(md.numberofelements,1);
		macayealpattynflag(find(commonelements))=1;
		nodeonmacayeal=zeros(md.numberofnodes,1);
		nodeonmacayeal(md.elements(find(macayealflag),:))=1;
		md.flowequation.bordermacayeal=nodeonmacayeal;

		%Create MacaAyealPattynApproximation where needed
		md.flowequation.element_equation(find(macayealpattynflag))=5;

		%Now recreate nodeonmacayeal and nodeonmacayealpattyn
		nodeonmacayealpattyn(md.elements(find(macayealpattynflag),:))=1;
	elseif any(pattynflag) & any(stokesflag), %coupling pattyn stokes
		%Find node at the border
		nodeonpattynstokes(find(nodeonpattyn & nodeonstokes))=1;
		%Stokes elements in contact with this layer become PattynStokes elements
		matrixelements=ismember(md.elements,find(nodeonpattynstokes));
		commonelements=sum(matrixelements,2)~=0;
		commonelements(find(pattynflag))=0; %only one layer: the elements previously in macayeal
		stokesflag(find(commonelements))=0; %these elements are now macayealpattynelements
		pattynstokesflag=zeros(md.numberofelements,1);
		pattynstokesflag(find(commonelements))=1;
		nodeonstokes=zeros(md.numberofnodes,1);
		nodeonstokes(md.elements(find(stokesflag),:))=1;
		md.flowequation.borderstokes=nodeonstokes;

		%Create MacaAyealPattynApproximation where needed
		md.flowequation.element_equation(find(pattynstokesflag))=7;

		%Now recreate nodeonpattynstokes
		nodeonpattynstokes=zeros(md.numberofnodes,1);
		nodeonpattynstokes(md.elements(find(pattynstokesflag),:))=1;
	elseif any(stokesflag) & any(macayealflag),
		%Find node at the border
		nodeonmacayealstokes(find(nodeonmacayeal & nodeonstokes))=1;
		%Stokes elements in contact with this layer become MacAyealStokes elements
		matrixelements=ismember(md.elements,find(nodeonmacayealstokes));
		commonelements=sum(matrixelements,2)~=0;
		commonelements(find(macayealflag))=0; %only one layer: the elements previously in macayeal
		stokesflag(find(commonelements))=0; %these elements are now macayealmacayealelements
		macayealstokesflag=zeros(md.numberofelements,1);
		macayealstokesflag(find(commonelements))=1;
		nodeonstokes=zeros(md.numberofnodes,1);
		nodeonstokes(md.elements(find(stokesflag),:))=1;
		md.flowequation.borderstokes=nodeonstokes;

		%Create MacaAyeal Approximation where needed
		md.flowequation.element_equation(find(macayealstokesflag))=6;

		%Now recreate nodeonmacayealstokes
		nodeonmacayealstokes=zeros(md.numberofnodes,1);
		nodeonmacayealstokes(md.elements(find(macayealstokesflag),:))=1;
	elseif any(stokesflag) & any(hutterflag),
		error('type of coupling not supported yet');
	end
end

%Create vertices_type
md.flowequation.vertex_equation=zeros(md.numberofnodes,1);
pos=find(nodeonhutter);
md.flowequation.vertex_equation(pos)=1;
pos=find(nodeonmacayeal);
md.flowequation.vertex_equation(pos)=2;
pos=find(nodeonpattyn);
md.flowequation.vertex_equation(pos)=3;
pos=find(nodeonhutter);
md.flowequation.vertex_equation(pos)=1;
pos=find(nodeonpattyn & nodeonmacayeal);
md.flowequation.vertex_equation(pos)=3;
pos=find(nodeonmacayealpattyn);
md.flowequation.vertex_equation(pos)=5;
pos=find(nodeonstokes);
md.flowequation.vertex_equation(pos)=4;
if any(stokesflag),
	pos=find(~nodeonstokes);
	if(~any(pattynflag) & ~any(macayealflag)),
		md.flowequation.vertex_equation(pos)=0;
	end
end
pos=find(nodeonpattynstokes);
md.flowequation.vertex_equation(pos)=7;
pos=find(nodeonmacayealstokes);
md.flowequation.vertex_equation(pos)=6;

%figure out solution types
md.flowequation.ishutter=double(any(md.flowequation.element_equation==1));
md.flowequation.ismacayealpattyn=double(any(md.flowequation.element_equation==2 | md.flowequation.element_equation==3));
md.flowequation.isstokes=double(any(md.flowequation.element_equation==4));

end
