function md=BasinConstrain(md,domain);
%BASINCONSTRAIN - constrain basin
%
%   Constrain basin using a constraint domain outline, 
%   to dirichlet boundary conditions.
%   constraindomain is an Argus domain outline file enclosing 
%   the geographical area of interest.
%
%   Usage: 
%      md=BasinConstrain(md,constraindomain)
%
%   Example:
%      md=BasinConstrain(md,'DomainOutline.exp');
%      md=BasinConstrain(md,'~Iceshelves.exp');

%now, flag nodes and elements outside the domain outline.
if ischar(domain),
	if isempty(domain),
		elementondomain=zeros(md.numberofelements,1);
		vertexondomain=zeros(md.numberofnodes,1);
		invert=0;
	elseif strcmpi(domain,'all')
		elementondomain=ones(md.numberofelements,1);
		vertexondomain=ones(md.numberofnodes,1);
		invert=0;
	else
		%make sure that we actually don't want the elements outside the domain outline!
		if strcmpi(domain(1),'~'),
			domain=domain(2:end);
			invert=1;
		else
			invert=0;
		end
		%ok, flag elements and nodes
		[vertexondomain elementondomain]=ContourToMesh(md.elements(:,1:3),md.x,md.y,domain,'element and node',2);
	end
	if invert,
		vertexondomain=~vertexondomain;
		elementondomain=~elementondomain;
	end
else
	error('BasinConstrain error message: domain type not supported yet');
end

%list of elements and nodes not on domain
vertexnotondomain=find(~vertexondomain);
elementnotondomain=find(~elementondomain);

%all elements outside the constraint domain are equivalent to water. all nodes outside are spc'd.
md.diagnostic.spcvx(vertexnotondomain)=md.vx_obs(vertexnotondomain);
md.diagnostic.spcvy(vertexnotondomain)=md.vy_obs(vertexnotondomain);
md.mask.elementonwater(elementnotondomain)=1;

%now, make sure all elements on water have nodes that are spc'd, otherwise, we'll get a singular problem.
pos=find(~md.mask.elementonwater);
numpos=unique(md.elements(pos,:));
nodes=setdiff(1:1:md.numberofnodes,numpos);
md.diagnostic.spcvx(nodes)=md.vx_obs(nodes);
md.diagnostic.spcvy(nodes)=md.vy_obs(nodes);

%make sure icefronts that are completely spc'd are taken out:
free_segments=find((~isnan(md.diagnostic.spcvx(md.diagnostic.icefront(:,1:2))) + ~isnan(md.diagnostic.spcvy(md.diagnostic.icefront(:,1:2))))~=2);
md.diagnostic.icefront=md.diagnostic.icefront(free_segments,:);
