%MODEL class definition
%
%   Usage:
%      md = model(varargin)

classdef model
    properties (SetAccess=public) %Model fields
		 % {{{1
		 %Careful here: no other class should be used as default value this is a bug of matlab
		 cluster   = modelfield('default',0,'marshall',false);
		 verbose   = modelfield('default',0,'marshall',true,'preprocess','marshallverbose','format','Integer');
		 results   = modelfield('default',0,'marshall',false);
		 solver    = modelfield('default',0,'marshall',false);
		 debug     = modelfield('default',0,'marshall',false);
		 constants = modelfield('default',0,'marshall',true);
		 flaim     = modelfield('default',0,'marshall',false);
		 surfaceforcings = modelfield('default',0,'marshall',true);
		 basalforcings   = modelfield('default',0,'marshall',true);
		 friction  = modelfield('default',0,'marshall',true);
		 private   = modelfield('default',0,'marshall',false);
		 rifts     = modelfield('default',0,'marshall',true);
		 hydrology = modelfield('default',0,'marshall',true);
		 settings  = modelfield('default',0,'marshall',true);
		 radaroverlay = modelfield('default',0,'marshall',false);
		 thermal   = modelfield('default',0,'marshall',true);
		 miscellaneous = modelfield('default',0,'marshall',true);
		 timestepping = modelfield('default',0,'marshall',true);
		 groundingline = modelfield('default',0,'marshall',true);
		 materials = modelfield('default',0,'marshall',true);
		 mask = modelfield('default',0,'marshall',true);

		 %FIXME: all other fields should belong to other classes

		 %Mesh
		 dim                 = modelfield('default',0,'marshall',true,'format','Integer');
		 numberofelements    = modelfield('default',0,'marshall',true,'format','Integer');
		 numberofnodes       = modelfield('default',0,'marshall',true,'format','Integer');
		 elements            = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',2);
		 elements_type       = modelfield('default',NaN,'marshall',true,'preprocess','marshallapproximations','format','DoubleMat','mattype',2);
		 vertices_type       = modelfield('default',NaN,'marshall',true,'preprocess','marshallapproximations','format','DoubleMat','mattype',1);
		 x                   = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 y                   = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 z                   = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 nodeconnectivity    = modelfield('default',NaN,'marshall',false);
		 elementconnectivity = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',3);
		 edges               = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',3);

		 %Initial 2d mesh 
		 numberofelements2d = modelfield('default',0,'marshall',true,'format','Integer');
		 numberofnodes2d    = modelfield('default',0,'marshall',true,'format','Integer');
		 elements2d         = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',3);
		 x2d                = modelfield('default',NaN,'marshall',false);
		 y2d                = modelfield('default',NaN,'marshall',false);

		 %latlon of the coorinates
		 lat        = modelfield('default',NaN,'marshall',false);
		 long       = modelfield('default',NaN,'marshall',false);
		 hemisphere = modelfield('default',NaN,'marshall',false);

		 %Elements type
		 ishutter             = modelfield('default',0,'marshall',true,'format','Boolean');
		 ismacayealpattyn     = modelfield('default',0,'marshall',true,'format','Boolean');
		 isstokes             = modelfield('default',0,'marshall',true,'format','Boolean');

		 %Elements
		 elementonhutter      = modelfield('default',NaN,'marshall',false);
		 elementonmacayeal    = modelfield('default',NaN,'marshall',false);
		 elementonpattyn      = modelfield('default',NaN,'marshall',false);
		 elementonstokes      = modelfield('default',NaN,'marshall',false);

		 %Nodes
		 nodeonhutter         = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 nodeonmacayeal       = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 nodeonpattyn         = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 nodeonstokes         = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);

		 %Stokes
		 stokesreconditioning = modelfield('default',0,'marshall',true,'format','Double');
		 shelf_dampening      = modelfield('default',0,'marshall',true,'format','Integer');

		 %Penalties
		 penalties            = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',3);
		 penalty_offset       = modelfield('default',0,'marshall',true,'format','Double');
		 penalty_lock         = modelfield('default',0,'marshall',true,'format','Integer');
		 segments             = modelfield('default',NaN,'marshall',false);
		 segmentmarkers       = modelfield('default',NaN,'marshall',false);

		 %Projections
		 uppernodes           = modelfield('default',NaN,'marshall',false);
		 upperelements        = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',2);
		 lowerelements        = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',2);
		 lowernodes           = modelfield('default',NaN,'marshall',false);

		 %Extrusion
		 numlayers         = modelfield('default',0,'marshall',true,'format','Integer');
		 elementonbed      = modelfield('default',NaN,'marshall',true,'format','BooleanMat','mattype',2);
		 elementonsurface  = modelfield('default',NaN,'marshall',true,'format','BooleanMat','mattype',2);
		 nodeonbed         = modelfield('default',NaN,'marshall',true,'format','BooleanMat','mattype',1);
		 nodeonsurface     = modelfield('default',NaN,'marshall',true,'format','BooleanMat','mattype',1);
		 minh              = modelfield('default',0,'marshall',false);

		 %Extraction
		 extractednodes    = modelfield('default',NaN,'marshall',false);
		 extractedelements = modelfield('default',NaN,'marshall',false);

		 %Materials parameters

		 min_mechanical_constraints = modelfield('default',0,'marshall',true,'format','Integer');

		 %Geometrical parameters
		 surface           = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 thickness         = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 thickness_coeff   = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 bed               = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 bathymetry        = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);

		 %Boundary conditions
		 nodeonboundary = modelfield('default',NaN,'marshall',false);
		 pressureload   = modelfield('default',NaN,'marshall',true,'preprocess','marshallpressureload','format','DoubleMat','mattype',3);
		 spcvx          = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 spcvy          = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 spcvz          = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 spcthickness   = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 diagnostic_ref = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);

		 %Observations 
		 vx_obs                    = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 vy_obs                    = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 vel_obs                   = modelfield('default',NaN,'marshall',false);

		 dhdt                      = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 thickness_obs             = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);

		 %Statics parameters
		 eps_res                  = modelfield('default',0,'marshall',true,'format','Double');
		 eps_rel                  = modelfield('default',0,'marshall',true,'format','Double');
		 eps_abs                  = modelfield('default',0,'marshall',true,'format','Double');
		 max_nonlinear_iterations = modelfield('default',0,'marshall',true,'format','Integer');
		 max_steadystate_iterations = modelfield('default',0,'marshall',true,'format','Integer');
		 connectivity             = modelfield('default',0,'marshall',true,'format','Integer');
		 viscosity_overshoot      = modelfield('default',0,'marshall',true,'format','Double');

		 %Transient 
		 artificial_diffusivity = modelfield('default',0,'marshall',true,'format','Integer');
		 prognostic_DG          = modelfield('default',0,'marshall',true,'format','Integer');
		 hydrostatic_adjustment = modelfield('default',0,'marshall',true,'preprocess','StringToEnum','format','Integer');
		 isprognostic = modelfield('default',0,'marshall',true,'format','Boolean');
		 isdiagnostic = modelfield('default',0,'marshall',true,'format','Boolean');
		 isthermal    = modelfield('default',0,'marshall',true,'format','Boolean');
		 %Control
		 control_analysis = modelfield('default',0,'marshall',true,'format','Boolean');
		 control_type     = modelfield('default',NaN,'marshall',true,'preprocess','marshallcontroltype','format','DoubleMat','mattype',3);
		 weights          = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 nsteps           = modelfield('default',0,'marshall',true,'format','Integer');
		 maxiter          = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',3);
		 cm_responses     = modelfield('default',NaN,'marshall',true,'preprocess','marshallcmresponses','format','DoubleMat','mattype',3);
		 optscal          = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',3);
		 eps_cm           = modelfield('default',0,'marshall',true,'format','Double');
		 cm_min           = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',3);
		 cm_max           = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',3);
		 cm_jump          = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',3);
		 cm_gradient      = modelfield('default',0,'marshall',true,'format','Boolean');
		 num_control_type               = modelfield('default',0,'marshall',true,'format','Integer');
		 num_cm_responses               = modelfield('default',0,'marshall',true,'format','Integer');
		 %Output
		 requested_outputs               = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',3);
		 %Results fields
		 output_frequency               = modelfield('default',0,'marshall',true,'format','Integer');
		 results_on_vertices            = modelfield('default',0,'marshall',true,'format','Boolean');
		 vx                             = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 vy                             = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 vz                             = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 vel                            = modelfield('default',NaN,'marshall',false);
		 temperature                    = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1); %temperature solution vector
		 waterfraction                  = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 pressure                       = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);
		 %Hydrology
		 watercolumn                    = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',1);

		 %Parallelisation
		 outlog        = modelfield('default','','marshall',false);
		 errlog        = modelfield('default','','marshall',false);
		 waitonlock    = modelfield('default',0,'marshall',true,'format','Boolean');

		 %qmu
		 variables                       = modelfield('default',struct(),'marshall',false);
		 responses                       = modelfield('default',struct(),'marshall',false);
		 qmu_method                      = modelfield('default',struct(),'marshall',false);
		 qmu_params                      = modelfield('default',struct(),'marshall',false);
		 dakotaresults                   = modelfield('default',struct(),'marshall',false);
		 dakota_analysis                    = modelfield('default',0,'marshall',true,'format','Boolean');
		 part                            = modelfield('default',NaN,'marshall',true,'format','DoubleMat','mattype',2);
		 npart                           = modelfield('default',0,'marshall',true,'format','Integer');
		 numberofresponses               = modelfield('default',0,'marshall',true,'format','Integer');
		 variabledescriptors             = modelfield('default',{},'marshall',true,'format','StringArray');
		 responsedescriptors             = modelfield('default',{},'marshall',true,'format','StringArray');
		 qmu_mass_flux_profile_directory = modelfield('default',NaN,'marshall',false);
		 qmu_mass_flux_profiles          = modelfield('default',NaN,'marshall',false);
		 qmu_mass_flux_segments          = modelfield('default',{},'marshall',true,'format','MatArray');

		 %ad: automatic differentiation
		 ad_analysis = modelfield('default',false,'marshall',true,'format','Boolean');
		 ad_forward = modelfield('default',true,'marshall',true,'format','Boolean');
		 ad_reverse = modelfield('default',false,'marshall',true,'format','Boolean');

		 %partitioner:
		 adjacency = modelfield('default',NaN,'marshall',false);
		 vwgt      = modelfield('default',NaN,'marshall',false);
		 %}}}
	 end
	 methods (Static)
		 function md = loadobj(md) % {{{
			 % This function is directly called by matlab when a model object is
			 % loaded. If the input is a struct it is an old version of model and
			 % old fields must be recovered (make sure they are in the deprecated
			 % model properties)

			 if verLessThan('matlab','7.9'),
				 disp('Warning: your matlab version is old and there is a risk that load does not work correctly');
				 disp('         if the model is not loaded correctly, rename temporarily loadobj so that matlab does not use it');

				 % This is a Matlab bug: all the fields of md have their default value
				 % Example of error message:
				 % Warning: Error loading an object of class 'model':
				 % Undefined function or method 'exist' for input arguments of type 'cell'
				 %
				 % This has been fixed in MATLAB 7.9 (R2009b) and later versions
			 end

			 if isstruct(md)
				 disp('Recovering model object from a previous version');
				 md = structtomodel(model,md);
			 end
		 end% }}}
		 function temp = template() % {{{

			 %Get default fields
			 md=model(0);
			 modelprops=properties('model');

			 %build output
			 temp=struct();
			 for i=1:length(modelprops),
				 temp.(modelprops{i})=md.(modelprops{i});
			 end

		 end% }}}
	 end
	 methods
		 function md = model(varargin) % {{{1

			 switch nargin
				 case 0
					 md=setdefaultparameters(md);
				 case 1
					 in=varargin{1};
					 if (isa(in,'model'))
						 %If single argument of class model, we have a copy constructor. 
						 md = in;
					 elseif (isa(in,'numeric') & in==0)
						 %requesting template
						 %FIXME: for now recursive call
						 md2=model;
						 modelprops=properties('model');
						 for i=1:length(modelprops);
							 fieldname=modelprops{i};
							 if md.(fieldname).marshall & isobject(md2.(fieldname)) & isempty(md.(fieldname).preprocess),
								 eval(['md.' fieldname '=' fieldname '(0);']);
							 end
						 end
					 else
						 error('model constructor error message: unknown constructor for ''model'' class object');
					 end 

				 otherwise
					 error('model constructor error message: 0 of 1 argument only in input.');
				 end
		 end
		 %}}}
		 function disp(md) % {{{1
			 disp(sprintf('\n%s = \n',inputname(1)));

			 %name
			 disp(sprintf('   Name:'));
			 if isempty(md.miscellaneous.name),
				 disp(sprintf('      N/A'));
			 else
				 disp(sprintf('      %s',md.miscellaneous.name));
			 end

			 %notes: md.miscellaneous.notes is a cell array of strings
			 disp(sprintf('   Notes:'));
			 if isempty(md.miscellaneous.notes),
				 disp(sprintf('      N/A'));
			 else
				 if iscell(md.miscellaneous.notes),
					 strings=md.miscellaneous.notes;
					 for i=1:length(strings),
						 string=char(strings(i));
						 if i==1,
							 disp(sprintf('      %s',string));
						 else
							 disp(sprintf('      %s',string));
						 end
					 end
				 elseif ischar(md.miscellaneous.notes),
					 disp(sprintf('      %s',md.miscellaneous.notes));
				 else
					 disp('display error message: ''notes'' for model is neither a cell array of string, nor a string. Cannot display.');
				 end
			 end

			 %diverse
			 disp(sprintf('   Mesh:'));
			 disp(sprintf('%s%s%s','      Mesh: type ''',inputname(1),'.mesh'' to display'));
			 disp(sprintf('   Parameters:'));
			 disp(sprintf('%s%s%s','      Boundary conditions: type ''',inputname(1),'.bc'' to display'));
			 disp(sprintf('%s%s%s','      Observations: type ''',inputname(1),'.obs'' to display'));
			 disp(sprintf('%s%s%s','      Materials: type ''',inputname(1),'.mat'' to display'));
			 disp(sprintf('%s%s%s','      Parameters: type ''',inputname(1),'.par'' to display'));
			 disp(sprintf('   Solution parameters: %s'));
			 disp(sprintf('%s%s%s','      Qmu: type ''',inputname(1),'.qmu'' to display'));
			 disp(sprintf('%s%s%s','      Diagnostic solution parameters: type ''',inputname(1),'.diagnostic'' to display'));
			 disp(sprintf('%s%s%s','      Thermal solution parameters: type ''',inputname(1),'.thermal'' to display'));
			 disp(sprintf('%s%s%s','      Prognostic solution parameters: type ''',inputname(1),'.prognostic'' to display'));
			 disp(sprintf('%s%s%s','      Transient solution parameters: type ''',inputname(1),'.transient'' to display'));
			 disp(sprintf('%s%s%s','      Control solution parameters: type ''',inputname(1),'.control'' to display'));
			 disp(sprintf('   Parallel:'));
			 disp(sprintf('%s%s%s','      Parallel options: type ''',inputname(1),'.parallel'' to display'));
			 disp(sprintf('   Results:'));
			 disp(sprintf('%s%s%s','      Results: type ''',inputname(1),'.res'' to display'));
		 end
		 %}}}
		 function md = structtomodel(md,structmd) % {{{

			 if ~isstruct(structmd) error('input model is not a structure'); end

			 %loaded model is a struct, initialize output and recover all fields
			 md = structtoobj(model,structmd);

			 %Field name change
			 if isfield(structmd,'drag'), md.friction.coefficient=structmd.drag; end
			 if isfield(structmd,'p'), md.friction.p=structmd.p; end
			 if isfield(structmd,'q'), md.friction.q=structmd.p; end
			 if isfield(structmd,'melting'), md.basalforcings.melting_rate=structmd.melting; end
			 if isfield(structmd,'melting_rate'), md.basalforcings.melting_rate=structmd.melting_rate; end
			 if isfield(structmd,'accumulation'), md.surfaceforcings.mass_balance=structmd.accumulation; end
			 if isfield(structmd,'accumulation_rate'), md.surfaceforcings.mass_balance=structmd.accumulation_rate; end
			 if isfield(structmd,'numberofgrids'), md.numberofnodes=structmd.numberofgrids; end
			 if isfield(structmd,'numberofgrids2d'), md.numberofnodes2d=structmd.numberofgrids2d; end
			 if isfield(structmd,'gridonhutter'), md.nodeonhutter=structmd.gridonhutter; end
			 if isfield(structmd,'gridonmacayeal'), md.nodeonmacayeal=structmd.gridonmacayeal; end
			 if isfield(structmd,'gridonpattyn'), md.nodeonpattyn=structmd.gridonpattyn; end
			 if isfield(structmd,'gridonstokes'), md.nodeonstokes=structmd.gridonstokes; end
			 if isfield(structmd,'uppergrids'), md.uppernodes=structmd.uppergrids; end
			 if isfield(structmd,'lowergrids'), md.lowernodes=structmd.lowergrids; end
			 if isfield(structmd,'gridonbed'), md.nodeonbed=structmd.gridonbed; end
			 if isfield(structmd,'gridonsurface'), md.nodeonsurface=structmd.gridonsurface; end
			 if isfield(structmd,'extractedgrids'), md.extractednodes=structmd.extractedgrids; end
			 if isfield(structmd,'gridoniceshelf'), md.nodeoniceshelf=structmd.gridoniceshelf; end
			 if isfield(structmd,'gridonicesheet'), md.nodeonicesheet=structmd.gridonicesheet; end
			 if isfield(structmd,'gridonwater'), md.nodeonwater=structmd.gridonwater; end
			 if isfield(structmd,'gridonnuna'), md.nodeonnuna=structmd.gridonnuna; end
			 if isfield(structmd,'gridonboundary'), md.nodeonboundary=structmd.gridonboundary; end
			 if isfield(structmd,'petscoptions'), md.solver=structmd.petscoptions; end
			 if isfield(structmd,'g'), md.constants.g=structmd.g; end
			 if isfield(structmd,'yts'), md.constants.yts=structmd.yts; end
			 if isfield(structmd,'dakota_analysis'), md.dakota_analysis=structmd.dakota_analysis; end
			 if isfield(structmd,'surface_mass_balance'), md.surfaceforcings.mass_balance=structmd.surface_mass_balance; end
			 if isfield(structmd,'basal_melting_rate'), md.basalforcings.melting_rate=structmd.basal_melting_rate; end
			 if isfield(structmd,'basal_melting_rate_correction'), md.basalforcings.melting_rate_correction=structmd.basal_melting_rate_correction; end
			 if isfield(structmd,'geothermalflux'), md.basalforcings.geothermalflux=structmd.geothermalflux; end
			 if isfield(structmd,'drag_coefficient'), md.friction.coefficient=structmd.drag_coefficient; end
			 if isfield(structmd,'drag_p'), md.friction.p=structmd.drag_p; end
			 if isfield(structmd,'drag_q'), md.friction.q=structmd.drag_q; end
			 if isfield(structmd,'numrifts'), md.rifts.numrifts=structmd.numrifts; end
			 if isfield(structmd,'rifts'), md.rifts.riftstruct=structmd.rifts; end
			 if isfield(structmd,'riftproperties'), md.rifts.riftproperties=structmd.riftproperties; end
			 if isfield(structmd,'bamg'), md.private.bamg=structmd.bamg; end
			 if isfield(structmd,'lowmem'), md.settings.lowmem=structmd.lowmem; end
			 if isfield(structmd,'io_gather'), md.settings.io_gather=structmd.io_gather; end
			 if isfield(structmd,'spcwatercolumn'), md.hydrology.spcwatercolumn=structmd.spcwatercolumn; end
			 if isfield(structmd,'hydro_n'), md.hydrology.n=structmd.hydro_n; end
			 if isfield(structmd,'hydro_p'), md.hydrology.p=structmd.hydro_p; end
			 if isfield(structmd,'hydro_q'), md.hydrology.q=structmd.hydro_q; end
			 if isfield(structmd,'hydro_CR'), md.hydrology.CR=structmd.hydro_CR; end
			 if isfield(structmd,'hydro_kn'), md.hydrology.kn=structmd.hydro_kn; end
			 if isfield(structmd,'spctemperature'), md.thermal.spctemperature=structmd.spctemperature; end
			 if isfield(structmd,'min_thermal_constraints'), md.thermal.penalty_threshold=structmd.min_thermal_constraints; end
			 if isfield(structmd,'artificial_diffusivity'), md.thermal.stabilization=structmd.artificial_diffusivity; end
			 if isfield(structmd,'max_nonlinear_iterations'), md.thermal.maxiter=structmd.max_nonlinear_iterations; end
			 if isfield(structmd,'stabilize_constraints'), md.thermal.penalty_lock=structmd.stabilize_constraints; end
			 if isfield(structmd,'penalty_offset'), md.thermal.penalty_factor=structmd.penalty_offset; end
			 if isfield(structmd,'name'), md.miscellaneous.name=structmd.name; end
			 if isfield(structmd,'notes'), md.miscellaneous.notes=structmd.notes; end
			 if isfield(structmd,'dummy'), md.miscellaneous.dummy=structmd.dummy; end
			 if isfield(structmd,'dt'), md.timestepping.time_step=structmd.dt; end
			 if isfield(structmd,'ndt'), md.timestepping.final_time=structmd.ndt; end
			 if isfield(structmd,'time_adapt'), md.timestepping.time_adapt=structmd.time_adapt; end
			 if isfield(structmd,'cfl_coefficient'), md.timestepping.cfl_coefficient=structmd.cfl_coefficient; end
			 if isfield(structmd,'rheology_B'), md.materials.rheology_B=structmd.B; end
			 if isfield(structmd,'rheology_n'), md.materials.rheology_n=structmd.n; end
			 if isfield(structmd,'elementoniceshelf'), md.mask.elementonfloatingice=structmd.elementoniceshelf; end
			 if isfield(structmd,'elementonicesheet'), md.mask.elementongroundedice=structmd.elementonicesheet; end
			 if isfield(structmd,'elementonwater'), md.mask.elementonwater=structmd.elementonwater; end
			 if isfield(structmd,'nodeoniceshelf'), md.mask.vertexonfloatingice=structmd.nodeoniceshelf; end
			 if isfield(structmd,'nodeonicesheet'), md.mask.vertexongroundedice=structmd.nodeonicesheet; end
			 if isfield(structmd,'nodeonwater'), md.mask.vertexonwater=structmd.nodeonwater; end

			 %Field changes
			 if (isfield(structmd,'type') & ischar(structmd.type)), 
				 if strcmpi(structmd.type,'2d'), md.dim=2; end
				 if strcmpi(structmd.type,'3d'), md.dim=3; end
			 end
			 if isnumeric(md.verbose),
				 md.verbose=verbose;
			 end
			 if isfield(structmd,'spcvelocity'), 
				 md.spcvx=NaN*ones(md.numberofnodes,1);
				 md.spcvy=NaN*ones(md.numberofnodes,1);
				 md.spcvz=NaN*ones(md.numberofnodes,1);
				 pos=find(structmd.spcvelocity(:,1)); md.spcvx(pos)=structmd.spcvelocity(pos,4); 
				 pos=find(structmd.spcvelocity(:,2)); md.spcvy(pos)=structmd.spcvelocity(pos,5); 
				 pos=find(structmd.spcvelocity(:,3)); md.spcvz(pos)=structmd.spcvelocity(pos,6); 
			 end
			 if isfield(structmd,'spcvelocity'), 
				 md.spcvx=NaN*ones(md.numberofnodes,1);
				 md.spcvy=NaN*ones(md.numberofnodes,1);
				 md.spcvz=NaN*ones(md.numberofnodes,1);
				 pos=find(structmd.spcvelocity(:,1)); md.spcvx(pos)=structmd.spcvelocity(pos,4); 
				 pos=find(structmd.spcvelocity(:,2)); md.spcvy(pos)=structmd.spcvelocity(pos,5); 
				 pos=find(structmd.spcvelocity(:,3)); md.spcvz(pos)=structmd.spcvelocity(pos,6); 
			 end
			 if ismember(structmd.pressureload(end,end),[118 119 120]),
				 pos=find(structmd.pressureload(:,end)==120); md.pressureload(pos,end)=0;
				 pos=find(structmd.pressureload(:,end)==118); md.pressureload(pos,end)=1;
				 pos=find(structmd.pressureload(:,end)==119); md.pressureload(pos,end)=2;
			 end
			 if (structmd.elements_type(end,end)>100),
				 pos=find(structmd.elements_type==59); md.elements_type(pos,end)=0;
				 pos=find(structmd.elements_type==55); md.elements_type(pos,end)=1;
				 pos=find(structmd.elements_type==56); md.elements_type(pos,end)=2;
				 pos=find(structmd.elements_type==60); md.elements_type(pos,end)=3;
				 pos=find(structmd.elements_type==62); md.elements_type(pos,end)=4;
				 pos=find(structmd.elements_type==57); md.elements_type(pos,end)=5;
				 pos=find(structmd.elements_type==58); md.elements_type(pos,end)=6;
				 pos=find(structmd.elements_type==61); md.elements_type(pos,end)=7;
			 end
			 if isnumeric(structmd.rheology_law),
				 if (structmd.rheology_law==272), md.materials.rheology_law='None';      end
				 if (structmd.rheology_law==368), md.materials.rheology_law='Paterson';  end
				 if (structmd.rheology_law==369), md.materials.rheology_law='Arrhenius'; end
			 end
			 if isnumeric(structmd.groundingline_migration),
				 if (structmd.groundingline_migration==272), md.groundingline.migration='None';      end
				 if (structmd.groundingline_migration==273), md.groundingline.migration='AgressiveMigration';  end
				 if (structmd.groundingline_migration==274), md.groundingline.migration='SoftMigration'; end
			 end
			 if isnumeric(md.control_type),
				 if (structmd.control_type==143), md.control_type={'FrictionCoefficient'}; end
				 if (structmd.control_type==190), md.control_type={'RheologyBbar'}; end
				 if (structmd.control_type==147), md.control_type={'Dhdt'}; end
			 end
			 if ismember(structmd.cm_responses(end,end),[165:170 383 388 389]),
				 pos=find(structmd.cm_responses==166), md.control_type(pos)=101;
				 pos=find(structmd.cm_responses==167), md.control_type(pos)=102;
				 pos=find(structmd.cm_responses==168), md.control_type(pos)=103;
				 pos=find(structmd.cm_responses==169), md.control_type(pos)=104;
				 pos=find(structmd.cm_responses==170), md.control_type(pos)=105;
				 pos=find(structmd.cm_responses==165), md.control_type(pos)=201;
				 pos=find(structmd.cm_responses==389), md.control_type(pos)=501;
				 pos=find(structmd.cm_responses==388), md.control_type(pos)=502;
				 pos=find(structmd.cm_responses==382), md.control_type(pos)=503;
			 end

			 %New fields
			 if ~isfield(structmd,'upperelements');
				 md.upperelements=transpose(1:md.numberofelements)+md.numberofelements2d;
				 md.upperelements(end-md.numberofelements2d+1:end)=NaN;
			 end
			 if ~isfield(structmd,'lowerelements');
				 md.lowerelements=transpose(1:md.numberofelements)-md.numberofelements2d;
				 md.lowerelements(1:md.numberofelements2d)=NaN;
			 end
			 if ~isfield(structmd,'diagnostic_ref');
				 md.diagnostic_ref=NaN*ones(md.numberofnodes2d,6);
			 end
		 end% }}}
		 function md = setdefaultparameters(md) % {{{1
		 %SETDEFAULTPARAMETERS - plug default parameters onto model
		 %
		 %   Although the model parameterization should be done in
		 %   the parameter file, some parameters are initialized here
		 %   with a default value.
		 %   These default values can be changed if necessary.
		 %
		 %   Usage:
		 %      md=setdefaultparameters(md);

			 %first, use the defaults provided by the properties definition above. 
			 fieldnames=fields(md);
			 for i=1:length(fieldnames),
				 fieldname=fieldnames{i};
				 md.(fieldname)=md.(fieldname).default;
			 end

			 %initialize subclasses
			 md.cluster=none;
			 md.solver=solver;
			 md.solver=addoptions(md.solver,DiagnosticVertAnalysisEnum,mumpsoptions);
			 md.results=struct();
			 md.debug=debug;
			 md.constants=constants;
			 md.flaim=flaim;
			 md.surfaceforcings=surfaceforcings;
			 md.basalforcings=basalforcings;
			 md.friction=friction;
			 md.private=private;
			 md.rifts=rifts;
			 md.hydrology=hydrology;
			 md.settings=settings;
			 md.radaroverlay=radaroverlay;
			 md.thermal=thermal;
			 md.miscellaneous=miscellaneous;
			 md.timestepping=timestepping;
			 md.groundingline=groundingline;
			 md.materials=materials;
			 md.mask=mask;


			 %Physical parameters

			 %Solver parameters

			 %mechanical residue convergence criterion norm(K(uold)uold - F)/norm(F)
			 md.eps_res=10^-4; %from test233 in nightly runs

			 %relative convergence criterion ((vel(n-1)-vel(n))/vel(n))
			 md.eps_rel=0.01;

			 %absolute convergence criterion (max(vel(n-1)-vel(n)) (m/year)
			 md.eps_abs=10;

			 %maximum of non-linear iterations.
			 md.max_nonlinear_iterations=100;
			 
			 %maximum of steady state iterations
			 md.max_steadystate_iterations=100;

			 %the connectivity is the avergaded number of nodes linked to a
			 %given node through an edge. This connectivity is used to initially
			 %allocate memory to the stiffness matrix. A value of 16 seems to
			 %give a good memory/time ration. This value can be checked in
			 %trunk/test/Miscellaneous/runme.m
			 md.connectivity=25;

			 %coefficient to update the viscosity between each iteration of
			 %a diagnostic according to the following formula
			 %viscosity(n)=viscosity(n)+viscosity_overshoot(viscosity(n)-viscosity(n-1))
			 md.viscosity_overshoot=0;

			 %parameter used to print temporary results (convergence criterion,
			 %current step,...)
			 md.verbose=verbose('solution',true,'qmu',true,'control',true);

			 %Stokes
			 md.stokesreconditioning=10^13;
			 md.shelf_dampening=0;

			 %Penalties

			 %the penalty offset is the power used to compute the value
			 %of the penalty as follows
			 %kappa=max(stiffness matrix)*10^penalty_offset
			 md.penalty_offset=3;

			 %in some solutions, it might be needed to stop a run when only
			 %a few constraints remain unstable. For thermal computation, this
			 %parameter is often used.
			 md.penalty_lock=10;

			 %minimum thickness to avoid stiffness singularity, used in
			 %the parameter file
			 md.minh=1;

			 %in some cases, it might be needed to stop a run when only
			 %a few constraints remain unstable. For thermal computation, this
			 %parameter is often used.
			 md.min_mechanical_constraints=0;

			 %Hydrostatic adjustment
			 md.hydrostatic_adjustment='Absolute';
			 %the artificial diffusivity is used in prognostic to avoid
			 %numerical wiggles of the solution.
			 md.artificial_diffusivity=1;

			 %Discontinuous Galerkin is used in prognostic to avoid
			 %numerical wiggles of the solution and conserve mass
			 md.prognostic_DG=0;

			 %Solution activated for transient runs. By default we do a
			 %full analysis: Diagnostic, Prognostic and Thermal
			 md.isprognostic=1;
			 md.isdiagnostic=1;
			 md.isthermal=1;

			 %Control

			 %parameter to be inferred by control methods (only
			 %drag and B are supported yet)
			 md.control_type={'FrictionCoefficient'};

			 %number of steps in the control methods
			 md.nsteps=20;

			 %maximum number of iteration in the optimization algorithm for
			 %each step
			 md.maxiter=20*ones(md.nsteps,1);

			 %the inversed parameter is updated as follows:
			 %new_par=old_par + optscal(n)*C*gradient with C in [0 1];
			 %usually the optscal must be of the order of magnitude of the 
			 %inversed parameter (10^8 for B, 50 for drag) and can be decreased
			 %after the first iterations
			 md.optscal=50*ones(md.nsteps,1);

			 %several responses can be used:
			 md.cm_responses=101*ones(md.nsteps,1);

			 %cm_jump is used to speed up control method. When
			 %misfit(1)/misfit(0) < md.cm_jump, we go directly to
			 %the next step
			 md.cm_jump=.7*ones(md.nsteps,1); %30 per cent decrement.

			 %stop control solution at the gradient computation and return it? 
			 md.cm_gradient=0;

			 %eps_cm is a criteria to stop the control methods.
			 %if J[n]-J[n-1]/J[n] < criteria, the control run stops
			 %NaN if not applied
			 md.eps_cm=NaN; %not activated

			 %How often to save results, default is 1 so save every step
			 md.output_frequency=1;

			 %Parallelisation parameters

			 %cluster set as none for serial

			 %this option can be activated to load automatically the results
			 %onto the model after a parallel run by waiting for the lock file
			 %N minutes that is generated once the solution has converged
			 %0 to desactivate
			 md.waitonlock=Inf;

		 end
		 %}}}
		 function result = subsref(md,index) % {{{1

			 if length(index)==1,
				 index1=index(1);
				 if(strcmp(index1.subs,'mesh')), displaymesh(md);return; end
				 if(strcmp(index1.subs,'bc')), displaybc(md);return; end
				 if(strcmp(index1.subs,'mat')), displaymaterials(md);return; end
				 if(strcmp(index1.subs,'par')), displayparameters(md);return; end
				 if(strcmp(index1.subs,'res')), displayresults(md);return; end
				 if(strcmp(index1.subs,'obs')), displayobservations(md);return; end
				 if(strcmp(index1.subs,'qmu')), displayqmu(md);return; end
				 if(strcmp(index1.subs,'diagnostic')), displaydiagnostic(md);return; end
				 if(strcmp(index1.subs,'prognostic')), displayprognostic(md);return; end
				 if(strcmp(index1.subs,'transient')), displaytransient(md);return; end
				 if(strcmp(index1.subs,'control')), displaycontrol(md);return; end
				 if(strcmp(index1.subs,'parallel')), displayparallel(md);return; end
			 end

			 %return built in value
			 result=builtin('subsref',md,index);
		 end %}}}
	 end
 end
