/*!\file:  UnitConversion.cpp
 * \brief: convert units from the model to IU or from IU to the model.
 */ 

/*headers {{{1*/
#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../../Container/Container.h"
#include "../../objects/objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"

double UnitConversionScaleFactor(int type_enum);
/*}}}*/

void UnitConversion(double* values, int numvalues,int direction_enum, int type_enum){

	double scale;
	int i;

	/*Get scaling factor: */
	scale=UnitConversionScaleFactor(type_enum);

	/*Now, which direction are we going? once determined, use scale factor: */
	if(direction_enum==IuToExtEnum) for(i=0;i<numvalues;i++)values[i]=values[i]*scale; 
	else if(direction_enum==ExtToIuEnum) for(i=0;i<numvalues;i++)values[i]=values[i]/scale; 
	else _error_(" wrong direction for unit conversion, either IuToExtEnum or ExtToIuEnum. ");

}

double UnitConversion(double value, int direction_enum, int type_enum){

	UnitConversion(&value,1,direction_enum,type_enum);

	return value;
}


double UnitConversionScaleFactor(int type_enum){

	double yts=365.0*24.0*3600.0;
	
	double scale;
	switch(type_enum){
		case DtEnum:        scale=1.0/yts;break; //yr
		case NdtEnum:        scale=1.0/yts;break; //yr
		case TimeEnum:        scale=1.0/yts;break; //yr
		case VxEnum:          scale=yts;break; //m/yr
		case VxObsEnum:       scale=yts;break; //m/yr
		case VyEnum:          scale=yts;break; //m/yr
		case VyObsEnum:       scale=yts;break; //m/yr
		case VzEnum:          scale=yts;break; //m/yr
		case VzObsEnum:       scale=yts;break; //m/yr
		case VelEnum:         scale=yts;break; //m/yr
		case VelObsEnum:      scale=yts;break; //m/yr
		case DhdtEnum:        scale=yts;break; //m/yr
		case BasalMeltingRateEnum:        scale=yts;break; //m/yr
		case BasalMeltingRateCorrectionEnum: scale=yts;break; //m/yr
		case SurfaceforcingsAccumulationRateEnum: scale=yts;break; //m/yr
		case SurfaceforcingsAblationRateEnum:     scale=yts;break; //m/yr
		case SurfaceforcingsMassBalanceEnum:      scale=yts;break; //m/yr
		case MisfitEnum:      scale=pow(yts,2);break; //(m/yr)^2
		case MassFluxEnum:    scale=pow(10,-12)*yts;break; // (GigaTon/year)
		default: scale=1.0; break;
	}
	return scale;
}



