function femmodel=control_core(femmodel)
%CONTROL_CORE - compute the core inversion
%
%   Usage:
%      femmodel=control_core(femmodel);
%

	%recover parameters common to all solutions
	num_controls=femmodel.parameters.NumControlType;
	num_responses=femmodel.parameters.NumCmResponses;
	responses=femmodel.parameters.CmResponses;
	control_type=femmodel.parameters.ControlType;
	solution_type=femmodel.parameters.SolutionType;
	nsteps=femmodel.parameters.Nsteps;
	maxiter=femmodel.parameters.Maxiter;
	cm_jump=femmodel.parameters.CmJump;
	eps_cm=femmodel.parameters.EpsCm;
	cm_gradient=femmodel.parameters.CmGradient;
	dim=femmodel.parameters.Dim;
	isstokes=femmodel.parameters.Isstokes;

	%Initialise options with maxiter
	options.MaxIter=femmodel.parameters.Maxiter;

	%Initialize misfits with a vector of zeros
	J=zeros(nsteps,1);
	search_scalar=0;

	%Get core from solution type
	solutioncore=CorePointerFromSolutionEnum(solution_type);
	adjointcore=AdjointCorePointerFromSolutionEnum(solution_type);

	%Launch once a complete solution to set up all inputs
	issmprintf(VerboseControl,'   preparing initial solution');
	if isstokes,
		eval(['femmodel=' solutioncore '(femmodel);']);
	end

	for n=1:nsteps,

		issmprintf(VerboseControl(),['control method step ' num2str(n) '/' num2str(femmodel.parameters.Nsteps)]);
		femmodel.parameters.StepResponses=responses(n,:);

		%In case we are running a steady state control method, compute new temperature field using new parameter distribution: 
		if (solution_type==SteadystateSolutionEnum)
			femmodel=steadystate_core(femmodel);
		end

		issmprintf(VerboseControl,'   compute adjoint state');
		eval(['femmodel=' adjointcore '(femmodel);']);

		femmodel=gradient_core(femmodel,n,search_scalar);

		%Return gradient if asked
		if cm_gradient,
			femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,GradientEnum);
			return;
		end

		issmprintf(VerboseControl,'   optimizing along gradient direction');
		[search_scalar J(n)]=ControlOptimization('objectivefunctionC',0,1,options,femmodel,n,femmodel.parameters);

		issmprintf(VerboseControl,'   updating parameter using optimized search scalar');
		[femmodel.elements,femmodel.nodes,femmmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters]=InputControlUpdate(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,search_scalar,1);
		issmprintf(VerboseControl,['   value of misfit J after optimization #' num2str(n) ':' num2str(J(n))]);

		%Has convergence been reached?
		converged=controlconvergence(J,responses,eps_cm,n);
		if converged,
			break;
		end

	end

	%generate output
	issmprintf(VerboseControl,'   preparing final velocity solution');
	femmodel.parameters.ControlAnalysis=0;
	eval(['femmodel=' solutioncore '(femmodel);']);

	%Some results not computed by diagnostic or steadystate
	for i=1:num_controls,
		femmodel.elements=InputToResult(femmodel.elements,femmodel.nodes,femmodel.vertices,femmodel.loads,femmodel.materials,femmodel.parameters,control_type(i));
	end
	femmodel.results=AddExternalResult(femmodel.results,JEnum,J);
	%femmodel.results=AddExternalResult(femmodel.results,ControlTypeEnum,control_type);

end %end function
