function md=transient3d(md);
%TRANSIENT2D - 3d transient solution
%
%   Usage:
%      md=transient3d(md)
%
%   See also: TRANSIENT2D, TRANSIENT

%Build all models related to diagnostic
displaystring(md.debug,'%s',['reading diagnostic horiz model data']);
md.analysis_type='diagnostic'; md.sub_analysis_type='horiz'; models.dh=CreateFemModel(md);

displaystring(md.debug,'\n%s',['reading diagnostic vert model data']);
md.analysis_type='diagnostic'; md.sub_analysis_type='vert'; models.dv=CreateFemModel(md);

displaystring(md.debug,'\n%s',['reading diagnostic stokes model data']);
md.analysis_type='diagnostic'; md.sub_analysis_type='stokes'; models.ds=CreateFemModel(md);

displaystring(md.debug,'\n%s',['reading diagnostic hutter model data']);
md.analysis_type='diagnostic'; md.sub_analysis_type='hutter'; models.dhu=CreateFemModel(md);

displaystring(md.debug,'\n%s',['reading surface and bed slope computation model data']);
md.analysis_type='slope_compute'; md.sub_analysis_type=''; models.sl=CreateFemModel(md);

displaystring(md.debug,'%s',['reading prognostic model data']);
md.analysis_type='prognostic'; models.p=CreateFemModel(md);

%Build all models related to thermal
displaystring(md.debug,'%s',['reading thermal model data']);
md.analysis_type='thermal'; models.t=CreateFemModel(md);

displaystring(md.debug,'%s',['reading melting model data']);
md.analysis_type='melting'; models.m=CreateFemModel(md);


%initialize solution
solution=struct('step',[],'time',[],'u_g',[],'p_g',[],'h_g',[],'s_g',[],'b_g',[]);
solution.step=1;
solution.time=0;
solution.u_g=models.dh.parameters.u_g;
%solution.u_g=models.dh.parameters.u_g(dofsetgen([1,2],3,length(models.dh.parameters.u_g)));
solution.p_g=models.p.parameters.p_g;
solution.h_g=models.p.parameters.h_g;
solution.s_g=models.p.parameters.s_g;
solution.b_g=models.p.parameters.b_g;
solution.t_g=models.p.parameters.t_g;
solution.m_g=models.p.parameters.m_g;
solution.a_g=models.p.parameters.a_g;

%initialize inputs
displaystring(md.debug,'\n%s',['setup inputs...']);
inputs=inputlist;
inputs=add(inputs,'velocity',solution.u_g,'doublevec',3,models.p.parameters.numberofnodes);
inputs=add(inputs,'melting',solution.m_g,'doublevec',1,models.p.parameters.numberofnodes);
inputs=add(inputs,'accumulation',solution.a_g,'doublevec',1,models.p.parameters.numberofnodes);
inputs=add(inputs,'dt',models.p.parameters.dt,'double');

% figure out number of dof: just for information purposes.
md.dof=modelsize(models);

%first time step is given by model. 
dt=models.p.parameters.dt;
finaltime=models.p.parameters.ndt;
time=dt;
yts=models.p.parameters.yts;
n=1; %counter

while  time<finaltime+dt, %make sure we run up to finaltime.

	displaystring(md.debug,'\n%s%g%s%g%s%g\n','time [yr]: ',time/yts,'    iteration number: ',n,'/',floor(finaltime/dt));

	solution(n+1).step=n+1;
	solution(n+1).time=time;

	%update inputs
	inputs=add(inputs,'thickness',solution(n).h_g,'doublevec',1,models.p.parameters.numberofnodes);
	inputs=add(inputs,'surface',solution(n).s_g,'doublevec',1,models.p.parameters.numberofnodes);
	inputs=add(inputs,'bed',solution(n).b_g,'doublevec',1,models.p.parameters.numberofnodes);
	inputs=add(inputs,'velocity',solution(n).u_g,'doublevec',3,models.p.parameters.numberofnodes);
	inputs=add(inputs,'pressure',solution(n).p_g,'doublevec',1,models.p.parameters.numberofnodes);
	inputs=add(inputs,'temperature',solution(n).t_g,'doublevec',1,models.t.parameters.numberofnodes);

	%Deal with temperature first 
	displaystring(md.debug,'\n%s',['    computing temperatures...']);
	[solution(n+1).t_g models.t.loads melting_offset]=thermal_core_nonlinear(models.t,inputs,'thermal','transient');
	inputs=add(inputs,'temperature',solution(n+1).t_g,'doublevec',1,models.t.parameters.numberofnodes);
	
	displaystring(md.debug,'\n%s',['    computing melting...']);
	inputs=add(inputs,'melting_offset',melting_offset,'double');
	solution(n+1).m_g=diagnostic_core_linear(models.m,inputs,'melting','transient');

	%Compute depth averaged temperature
	temperature_average=FieldDepthAverage(models.t.elements,models.t.nodes,models.t.loads,models.t.materials,solution(n+1).t_g,'temperature');
	inputs=add(inputs,'temperature_average',temperature_average,'doublevec',1,models.t.parameters.numberofnodes);

	%Deal with velocities.
	rawresults=diagnostic_core(models,inputs);
	solution(n+1).u_g=rawresults.u_g; solution(n+1).p_g=rawresults.p_g;

	%compute new thickness
	displaystring(md.debug,'\n%s',['    computing new thickness...']);
	inputs=add(inputs,'velocity',solution(n+1).u_g,'doublevec',3,models.p.parameters.numberofnodes);
	rawresults=prognostic_core(models,inputs,'prognostic','');
	new_thickness=rawresults.h_g;

	%update surface and bed using the new thickness
	displaystring(md.debug,'\n%s',['    updating geometry...']);
	[new_thickness,new_bed,new_surface]=UpdateGeometry(models.p.elements,models.p.nodes,models.p.loads,models.p.materials,models.p.parameters,new_thickness,solution(n).b_g,solution(n).s_g);

	%Record bed surface and thickness in the solution
	solution(n+1).h_g=new_thickness;
	solution(n+1).b_g=new_bed;
	solution(n+1).s_g=new_surface;

	%figure out if time stepping is good
	%displaystring(md.debug,'   checking time stepping...'));
	%[back,dt,time]=TimeStepping(md,solution,dt,time);
	%if back,
	%	continue;
	%end

	%update node positions
	displaystring(md.debug,'\n%s',['    updating node positions...']);
	nodes_dh=UpdateNodePositions(models.dh.elements,models.dh.nodes,models.dh.loads,models.dh.materials,new_bed,new_thickness);
	nodes_dv=UpdateNodePositions(models.dv.elements,models.dv.nodes,models.dv.loads,models.dv.materials,new_bed,new_thickness);
	nodes_ds=UpdateNodePositions(models.ds.elements,models.ds.nodes,models.ds.loads,models.ds.materials,new_bed,new_thickness);
	nodes_sl=UpdateNodePositions(models.sl.elements,models.sl.nodes,models.sl.loads,models.sl.materials,new_bed,new_thickness);
	nodes_p=UpdateNodePositions(models.p.elements,models.p.nodes,models.p.loads,models.p.materials,new_bed,new_thickness);
	nodes_t=UpdateNodePositions(models.t.elements,models.t.nodes,models.t.loads,models.t.materials,new_bed,new_thickness);
	nodes_m=UpdateNodePositions(models.m.elements,models.m.nodes,models.m.loads,models.m.materials,new_bed,new_thickness);

	%update time and counter
	time=time+dt;
	n=n+1;
end

%Load results onto model
results=struct('step',[],'time',[],'vx',[],'vy',[],'vel',[],'pressure',[],'thickness',[],'surface',[],'bed',[]);
for i=1:length(solution),
	results(i).step=solution(i).step;
	results(i).time=solution(i).time/yts;
	results(i).vx=solution(i).u_g(1:3:end)*yts;
	results(i).vy=solution(i).u_g(2:3:end)*yts;
	results(i).vz=solution(i).u_g(3:3:end)*yts;
	results(i).vel=sqrt(solution(i).u_g(1:3:end).^2+solution(i).u_g(2:3:end).^2+solution(i).u_g(3:3:end).^2)*yts;
	results(i).pressure=solution(i).p_g;
	results(i).bed=solution(i).b_g;
	results(i).surface=solution(i).s_g;
	results(i).thickness=solution(i).h_g;
	results(i).temperature=solution(i).t_g;
	results(i).melting=solution(i).m_g;
end
if ~isstruct(md.results), md.results=struct(); end
md.results.transient=results;
