/*!\file:  issm.cpp
 * \brief: ISSM main parallel program
 */ 

#include "./issm.h"

int main(int argc,char  **argv){

	/*MPI: */
	extern int my_rank;
	extern int num_procs;

	/*I/O: */
	FILE *output_fid       = NULL;
	FILE *petscoptionsfid  = NULL;
	bool  qmu_analysis     = false;
	bool  control_analysis = false;
	bool  waitonlock       = false;

	/*FemModel: */
	FemModel *femmodel = NULL;

	/*configuration: */
	void (*solutioncore)(FemModel*)=NULL; //core solution function pointer
	int* analyses=NULL;
	int  numanalyses;
	int  solution_type;

	/*File names*/
	char *modelname      = NULL;
	char *lockfilename   = NULL;
	char *binfilename    = NULL;
	char *outbinfilename = NULL;
	char *petscfilename  = NULL;

	/*time*/
	double   start, finish;
	double   start_core, finish_core;
	double   start_init, finish_init;

	MODULEBOOT();

	#if !defined(_PARALLEL_) || (defined(_PARALLEL_) && !defined(_HAVE_PETSC_))
	_error_(" parallel executable was compiled without support of parallel libraries!");
	#endif

	/*Initialize Petsc and get start time*/
	int ierr=PetscInitialize(&argc,&argv,(char*)0,"");  
	if(ierr) _error_("Could not initialize Petsc");
	MPI_Barrier(MPI_COMM_WORLD); start=MPI_Wtime();

	/*Size and rank: */
	MPI_Comm_rank(MPI_COMM_WORLD,&my_rank);  
	MPI_Comm_size(MPI_COMM_WORLD,&num_procs); 

	/*First process inputs*/
	_printf_(true,"Launching solution sequence\n");
	if(argc<2)_error_("Usage error: no solution requested");
	solution_type=StringToEnumx(argv[1]);
	if(argc<3)_error_("Usage error: missing model name");
	modelname=argv[3];
	binfilename    = (char*)xmalloc((strlen(modelname)+strlen(".bin")   +1)*sizeof(char)); sprintf(binfilename,   "%s%s",modelname,".bin");
	outbinfilename = (char*)xmalloc((strlen(modelname)+strlen(".outbin")+1)*sizeof(char)); sprintf(outbinfilename,"%s%s",modelname,".outbin");
	petscfilename  = (char*)xmalloc((strlen(modelname)+strlen(".petsc") +1)*sizeof(char)); sprintf(petscfilename, "%s%s",modelname,".petsc");
	lockfilename   = (char*)xmalloc((strlen(modelname)+strlen(".lock")  +1)*sizeof(char)); sprintf(lockfilename,  "%s%s",modelname,".lock");

	MPI_Barrier(MPI_COMM_WORLD); start_init=MPI_Wtime();

	/*out of solution_type, figure out solution core function pointer, 
	 *and types of analyses needed: */
	SolutionConfiguration(&analyses,&numanalyses,&solutioncore,solution_type);

	/*Create femmodel, using input file: */
	femmodel=new FemModel(binfilename,outbinfilename,solution_type,analyses,numanalyses);

	/*Open output file once for all*/
	output_fid=pfopen(outbinfilename,"wb");
	femmodel->parameters->SetParam(output_fid,OutputFilePointerEnum);

	/*add petsc options to parameters: */
	petscoptionsfid=pfopen(petscfilename,"r");
	ParsePetscOptionsx(femmodel->parameters,petscoptionsfid);
	pfclose(petscoptionsfid,petscfilename);

	/*get parameters: */
	femmodel->parameters->FindParam(&qmu_analysis,QmuAnalysisEnum);
	femmodel->parameters->FindParam(&control_analysis,ControlAnalysisEnum);
	femmodel->parameters->FindParam(&waitonlock,WaitonlockEnum);

	MPI_Barrier(MPI_COMM_WORLD); finish_init=MPI_Wtime();

	/*if control is being run on top of a solution, change core: */
	if(control_analysis){
		#ifdef _HAVE_TAO_
		//solutioncore=&controltao_core;
		solutioncore=&control_core;
		#else
		solutioncore=&control_core;
		#endif
	}

	/*are we running the solution sequence, or a qmu wrapper around it? : */
	if(!qmu_analysis){
			
		_printf_(true,"call computational core:\n");
		MPI_Barrier(MPI_COMM_WORLD); start_core=MPI_Wtime( );
		solutioncore(femmodel);
		MPI_Barrier(MPI_COMM_WORLD); finish_core=MPI_Wtime( );

		_printf_(true,"write results to disk:\n");
		OutputResultsx(femmodel->elements, femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,&femmodel->results);
	}
	else{
		/*run qmu analysis: */
		_printf_(true,"calling qmu analysis on diagnostic core:\n");

		#ifdef _HAVE_DAKOTA_ 
		MPI_Barrier(MPI_COMM_WORLD); start_core=MPI_Wtime( );
		Qmux(femmodel,femmodel->parameters);
		MPI_Barrier(MPI_COMM_WORLD); finish_core=MPI_Wtime( );
	 	#else
		_error_(" Dakota not present, cannot do qmu!");
		#endif
	}

	/*Close output and petsc options file and write lock file if requested*/
	pfclose(output_fid,lockfilename);
	if (waitonlock>0){
		_printf_(true,"write lock file:\n");
		WriteLockFile(lockfilename);
	}

	/*Free ressources */
	xfree((void**)&analyses);
	xfree((void**)&lockfilename);
	xfree((void**)&binfilename);
	xfree((void**)&outbinfilename);
	xfree((void**)&petscfilename);
	delete femmodel;

	/*Get finish time and close*/
	MPI_Barrier(MPI_COMM_WORLD); finish = MPI_Wtime( );
	_printf_(true,"\n   %-34s %f seconds  \n","FemModel initialization elapsed time:",finish_init-start_init);
	_printf_(true,"   %-34s %f seconds  \n","Core solution elapsed time:",finish_core-start_core);
	_printf_(true,"\n   %s %i hrs %i min %i sec\n\n","Total elapsed time:",int((finish-start)/3600),int(int(finish-start)%3600/60),int(finish-start)%60);
	_printf_(true,"closing MPI and Petsc\n");
	PetscFinalize(); 
	
	/*end module: */
	MODULEEND();

	return 0; //unix success return;
}
