function md=extrude(md,varargin)
%EXTRUDE - vertically extrude a 2d mesh
%
%   vertically extrude a 2d mesh and create corresponding 3d mesh.
%   The vertical distribution can:
%    - follow a polynomial law
%    - follow two polynomial laws, one for the lower part and one for the upper part of the mesh
%    - be discribed by a list of coefficients (between 0 and 1)
%   
%
%   Usage:
%      md=extrude(md,numlayers,extrusionexponent);
%      md=extrude(md,numlayers,lowerexponent,upperexponent);
%      md=extrude(md,listofcoefficients);
%
%   Example:
%      md=extrude(md,8,3);
%      md=extrude(md,8,3,2);
%      md=extrude(md,[0 0.2 0.5 0.7 0.9 0.95 1]);
%
%   See also: MODELEXTRACT, COLLAPSE

%some checks on list of arguments
if ((nargin>4) | (nargin<2) | (nargout~=1)),
	help extrude;
	error('extrude error message');
end

%Extrude the mesh
if nargin==2, %list of coefficients
	list=varargin{1};
	if any(list<0) | any(list>1),
		error('extrusioncoefficients must be between 0 and 1');
	end
	extrusionlist=sort(unique([list(:);0;1]));
	numlayers=length(extrusionlist);
elseif nargin==3, %one polynomial law
	if varargin{2}<=0,
		help extrude;
		error('extrusionexponent must be >=0');
	end
	numlayers=varargin{1};
	extrusionlist=((0:1:numlayers-1)/(numlayers-1)).^varargin{2};
elseif nargin==4, %two polynomial laws
	numlayers=varargin{1};
	lowerexp=varargin{2};
	upperexp=varargin{3};

	if varargin{2}<=0 | varargin{3}<=0,
		help extrude;
		error('lower and upper extrusionexponents must be >=0');
	end

	lowerextrusionlist=[(0:2/(numlayers-1):1).^lowerexp]/2;
	upperextrusionlist=[(0:2/(numlayers-1):1).^upperexp]/2;
	extrusionlist=sort(unique([lowerextrusionlist 1-upperextrusionlist]));

end

if numlayers<2,
	disp('number of layers should be at least 2. returning initial model...');
	return
end

%Initialize with the 2d mesh
x3d=[]; 
y3d=[];
z3d=[];  %the lower node is on the bed
thickness3d=md.thickness; %thickness and bed for these nodes
bed3d=md.bed;

%Create the new layers
for i=1:numlayers,
	x3d=[x3d; md.x]; 
	y3d=[y3d; md.y];
	%nodes are distributed between bed and surface accordingly to the given exponent
	z3d=[z3d; bed3d+thickness3d*extrusionlist(i)]; 
end
number_nodes3d=size(x3d,1); %number of 3d nodes for the non extruded part of the mesh

%Extrude elements 
elements3d=[];
for i=1:numlayers-1,
	elements3d=[elements3d;[md.elements+(i-1)*md.numberofnodes md.elements+i*md.numberofnodes]]; %Create the elements of the 3d mesh for the non extruded part
end
number_el3d=size(elements3d,1); %number of 3d nodes for the non extruded part of the mesh

%Keep a trace of lower and upper nodes
lowernodes=NaN*ones(number_nodes3d,1);
uppernodes=NaN*ones(number_nodes3d,1);
lowernodes(md.numberofnodes+1:end)=1:(numlayers-1)*md.numberofnodes;
uppernodes(1:(numlayers-1)*md.numberofnodes)=md.numberofnodes+1:number_nodes3d;
md.lowernodes=lowernodes;
md.uppernodes=uppernodes;

%same for lower and upper elements
lowerelements=NaN*ones(number_el3d,1);
upperelements=NaN*ones(number_el3d,1);
lowerelements(md.numberofelements+1:end)=1:(numlayers-2)*md.numberofelements;
upperelements(1:(numlayers-2)*md.numberofelements)=md.numberofelements+1:(numlayers-1)*md.numberofelements;
md.lowerelements=lowerelements;
md.upperelements=upperelements;

%Save old mesh 
md.x2d=md.x;
md.y2d=md.y;
md.elements2d=md.elements;
md.numberofelements2d=md.numberofelements;
md.numberofnodes2d=md.numberofnodes;

%Update mesh type
md.dim=3;

%Build global 3d mesh 
md.elements=elements3d;
md.x=x3d;
md.y=y3d;
md.z=z3d;
md.numberofelements=number_el3d;
md.numberofnodes=number_nodes3d;
md.numlayers=numlayers;

%Ok, now deal with the other fields from the 2d mesh:

%drag_coefficient is limited to nodes that are on the bedrock.
md.drag_coefficient=project3d(md,'vector',md.drag_coefficient,'type','node','layer',1);

%p and q (same deal, except for element that are on the bedrock: )
md.drag_p=project3d(md,'vector',md.drag_p,'type','element');
md.drag_q=project3d(md,'vector',md.drag_q,'type','element');

%observations
md.vx_obs=project3d(md,'vector',md.vx_obs,'type','node');
md.vy_obs=project3d(md,'vector',md.vy_obs,'type','node');
md.vel_obs=project3d(md,'vector',md.vel_obs,'type','node');
md.surface_mass_balance=project3d(md,'vector',md.surface_mass_balance,'type','node');
md.surface_accumulation_rate=project3d(md,'vector',md.surface_accumulation_rate,'type','node');
md.surface_ablation_rate=project3d(md,'vector',md.surface_ablation_rate,'type','node');
md.dhdt=project3d(md,'vector',md.dhdt,'type','node');

%results
if ~isnan(md.vx),md.vx=project3d(md,'vector',md.vx,'type','node');end;
if ~isnan(md.vy),md.vy=project3d(md,'vector',md.vy,'type','node');end;
if ~isnan(md.vz),md.vz=project3d(md,'vector',md.vz,'type','node');end;
if ~isnan(md.vel),md.vel=project3d(md,'vector',md.vel,'type','node');end;
if ~isnan(md.temperature),md.temperature=project3d(md,'vector',md.temperature,'type','node');end;
if ~isnan(md.waterfraction),md.waterfraction=project3d(md,'vector',md.waterfraction,'type','node');end;

%bedinfo and surface info
md.elementonbed=project3d(md,'vector',ones(md.numberofelements2d,1),'type','element','layer',1);
md.elementonsurface=project3d(md,'vector',ones(md.numberofelements2d,1),'type','element','layer',md.numlayers-1);
md.nodeonbed=project3d(md,'vector',ones(md.numberofnodes2d,1),'type','node','layer',1);
md.nodeonsurface=project3d(md,'vector',ones(md.numberofnodes2d,1),'type','node','layer',md.numlayers);

%elementstype
if ~isnan(md.elements_type)
	oldelements_type=md.elements_type;
	md.elements_type=zeros(number_el3d,1);
	md.elements_type=project3d(md,'vector',oldelements_type,'type','element');
	md.nodeonhutter=project3d(md,'vector',md.nodeonhutter,'type','node');
	md.nodeonmacayeal=project3d(md,'vector',md.nodeonmacayeal,'type','node');
	md.nodeonpattyn=project3d(md,'vector',md.nodeonpattyn,'type','node');
	md.nodeonstokes=project3d(md,'vector',md.nodeonstokes,'type','node');
end

%verticestype
if ~isnan(md.vertices_type)
	md.vertices_type=zeros(number_nodes3d,1);
	md.vertices_type=project3d(md,'vector',oldvertices_type,'type','node');
end

%boundary conditions
md.spcvx=project3d(md,'vector',md.spcvx,'type','node');
md.spcvy=project3d(md,'vector',md.spcvy,'type','node');
md.spcvz=project3d(md,'vector',md.spcvz,'type','node');
md.spctemperature=project3d(md,'vector',md.spctemperature,'type','node','layer',md.numlayers,'padding',NaN);
md.spcthickness=project3d(md,'vector',md.spcthickness,'type','node');
md.diagnostic_ref=project3d(md,'vector',md.diagnostic_ref,'type','node');

%in 3d, pressureload: [node1 node2 node3 node4 element]
pressureload_layer1=[md.pressureload(:,1:2)  md.pressureload(:,2)+md.numberofnodes2d  md.pressureload(:,1)+md.numberofnodes2d  md.pressureload(:,3:4)]; %Add two columns on the first layer 
pressureload=[];
for i=1:numlayers-1,
	pressureload=[pressureload ;pressureload_layer1(:,1:4)+(i-1)*md.numberofnodes2d pressureload_layer1(:,5)+(i-1)*md.numberofelements2d pressureload_layer1(:,6)];
end
md.pressureload=pressureload;

%connectivity
md.elementconnectivity=repmat(md.elementconnectivity,numlayers-1,1);
md.elementconnectivity(find(md.elementconnectivity==0))=NaN;
for i=2:numlayers-1,
	md.elementconnectivity((i-1)*md.numberofelements2d+1:(i)*md.numberofelements2d,:)...
	=md.elementconnectivity((i-1)*md.numberofelements2d+1:(i)*md.numberofelements2d,:)+md.numberofelements2d;
end
md.elementconnectivity(find(isnan(md.elementconnectivity)))=0;

%materials
md.rheology_B=project3d(md,'vector',md.rheology_B,'type','node');
md.rheology_n=project3d(md,'vector',md.rheology_n,'type','element');

%parameters
md.surface=project3d(md,'vector',md.surface,'type','node');
md.thickness=project3d(md,'vector',md.thickness,'type','node');
md.thickness_coeff=project3d(md,'vector',md.thickness_coeff,'type','node');
md.bed=project3d(md,'vector',md.bed,'type','node');
md.bathymetry=project3d(md,'vector',md.bathymetry,'type','node');
md.nodeonboundary=project3d(md,'vector',md.nodeonboundary,'type','node');
md.elementoniceshelf=project3d(md,'vector',md.elementoniceshelf,'type','element');
md.nodeoniceshelf=project3d(md,'vector',md.nodeoniceshelf,'type','node');
md.elementonicesheet=project3d(md,'vector',md.elementonicesheet,'type','element');
md.nodeonicesheet=project3d(md,'vector',md.nodeonicesheet,'type','node');
md.elementonwater=project3d(md,'vector',md.elementonwater,'type','element');
md.nodeonwater=project3d(md,'vector',md.nodeonwater,'type','node');
if ~isnan(md.weights),md.weights=project3d(md,'vector',md.weights,'type','node');end;

%Put lithostatic pressure is there is an existing pressure
if ~isnan(md.pressure),
	md.pressure=md.g*md.rho_ice*(md.surface-md.z);
end

%special for thermal modeling:
md.basal_melting_rate=project3d(md,'vector',md.basal_melting_rate,'type','node','layer',1); 
if ~isnan(md.geothermalflux)
	md.geothermalflux=project3d(md,'vector',md.geothermalflux,'type','node','layer',1); %bedrock only gets geothermal flux
end

%increase connectivity if less than 25:
if md.connectivity<=25,
	md.connectivity=100;
end
