function marshall(md)
%MARSHALL - output JPL-package compatible binary file from @model md, for certain solution type.
%
%   The routine creates a JPL-package compatible binary file from @model md
%   This binary file will be used for parallel runs in JPL-package
%
%   Usage:
%      marshall(md)

disp(['marshalling file ' md.name '.bin']);

%open file for binary writing
fid=fopen([ md.name '.bin'],'wb');
if fid==-1,
	error(['marshall error message: could not open ' [md.name '.bin'],' file for binary writing']);
end

%automatically marshall model md, using template information available from an empty model class.
template=model.template();
fields=fieldnames(template);
for i=1:length(fields),
	field=fields{i};
	marshall=template.(field){2};
	if marshall,
		if strcmpi(field,'verbose'),
			WriteData(fid,field,VerboseToBinary(md.verbose),{'Integer'});
		elseif strcmpi(field,'rheology_law'),
			if strcmpi(md.rheology_law,'none'),
				WriteData(fid,field,NoneEnum,{'Integer'});
			elseif strcmpi(md.rheology_law,'paterson'),
				WriteData(fid,field,PatersonEnum,{'Integer'});
			elseif strcmpi(md.rheology_law,'arrhenius'),
				WriteData(fid,field,ArrheniusEnum,{'Integer'});
			else
				error('unkown value');
			end
		else
			WriteData(fid,field,md.(field),template.(field)(3:end));
		end
	end
end

%Quick fix: add some fields needed by parameters
WriteData(fid,'numberofvertices',md.numberofnodes,{'Integer'});
WriteData(fid,'numberofedges',size(md.edges,1),{'Integer'});

%close file
st=fclose(fid);
if st==-1,
	error(['marshall error message: could not close file ' [md.name '.bin']]);
end

function WriteData(fid,fieldname,fieldvalue,fieldtype) % {{{
%WRITEDATA - write model field in binary file
%
%   Usage:
%      WriteData(fid,fieldname,fieldvalue,fieldtype)
%
%   Example:
%     WriteData(fid,'thickness',md.thickness,{'DoubleMat',1})

%first recover data, enum of the data and type of data
data=fieldvalue;
enum=BuildEnum(fieldname);
data_type=fieldtype{1};

if issparse(data),
	data=full(data);
end

%Ok! write the enum (name) to identify this record uniquely
fwrite(fid,enum,'int'); 

%Now, write the data itself.
if     strcmpi(data_type,'Boolean'),% {{{
	if(numel(data)~=1), error(['field ' field ' cannot be marshalled as it has more than one element!']); end

	%first write length of record
	fwrite(fid,4+4,'int');  %1 bool (disguised as an int)+code

	%write data code: 
	fwrite(fid,TypeToCode(data_type),'int'); 

	%now write integer
	fwrite(fid,data,'int');  %send an int, not easy to send a bool
	% }}}
elseif strcmpi(data_type,'Integer'), % {{{
	if(numel(data)~=1), error(['field ' field ' cannot be marshalled as it has more than one element!']); end

	%first write length of record
	fwrite(fid,4+4,'int');  %1 integer + code

	%write data code: 
	fwrite(fid,TypeToCode(data_type),'int'); 

	%now write integer
	fwrite(fid,data,'int'); 
	% }}}
elseif strcmpi(data_type,'Double'), % {{{
	if(numel(data)~=1), error(['field ' field ' cannot be marshalled as it has more than one element!']); end

	%first write length of record
	fwrite(fid,8+4,'int');  %1 double+code

	%write data code: 
	fwrite(fid,TypeToCode(data_type),'int'); 

	%now write double
	fwrite(fid,data,'double'); 
	% }}}
elseif strcmpi(data_type,'String'), % {{{
	%first write length of record
	fwrite(fid,length(data)+4+4,'int');  %string + string size + code

	%write data code: 
	fwrite(fid,TypeToCode(data_type),'int'); 

	%now write string
	fwrite(fid,length(data),'int'); 
	fwrite(fid,data,'char'); 
	% }}}
elseif strcmpi(data_type,'BooleanMat'), % {{{

	%matrix type:
	mattype=fieldtype{2};

	%Get size
	s=size(data);
	%if matrix = NaN, then do not write anything
	if (s(1)==1 & s(2)==1 & isnan(data)),
		s(1)=0; s(2)=0;
	end

	%first write length of record
	fwrite(fid,4+4+8*s(1)*s(2)+4+4,'int');  %2 integers (32 bits) + the double matrix + code + matrix type

	%write data code and matrix type: 
	fwrite(fid,TypeToCode(data_type),'int'); 
	fwrite(fid,mattype,'int');

	%now write matrix
	fwrite(fid,s(1),'int'); 
	fwrite(fid,s(2),'int'); 
	if s(1)*s(2),
		fwrite(fid,data','double'); %get to the "c" convention, hence the transpose
	end
	% }}}
elseif strcmpi(data_type,'IntMat'), % {{{

	%matrix type:
	mattype=fieldtype{2};

	%Get size
	s=size(data);
	%if matrix = NaN, then do not write anything
	if (s(1)==1 & s(2)==1 & isnan(data)),
		s(1)=0; s(2)=0;
	end

	%first write length of record
	fwrite(fid,4+4+8*s(1)*s(2)+4+4,'int');  %2 integers (32 bits) + the double matrix + code + matrix type

	%write data code and matrix type: 
	fwrite(fid,TypeToCode(data_type),'int'); 
	fwrite(fid,mattype,'int');

	%now write matrix
	fwrite(fid,s(1),'int'); 
	fwrite(fid,s(2),'int'); 
	if s(1)*s(2),
		fwrite(fid,data','double'); %get to the "c" convention, hence the transpose
	end
	% }}}
elseif strcmpi(data_type,'DoubleMat'), % {{{

	%matrix type:
	mattype=fieldtype{2};

	%Get size
	s=size(data);
	%if matrix = NaN, then do not write anything
	if (s(1)==1 & s(2)==1 & isnan(data)),
		s(1)=0; s(2)=0;
	end

	%first write length of record
	fwrite(fid,4+4+8*s(1)*s(2)+4+4,'int');  %2 integers (32 bits) + the double matrix + code + matrix type

	%write data code and matrix type: 
	fwrite(fid,TypeToCode(data_type),'int'); 
	fwrite(fid,mattype,'int');

	%now write matrix
	fwrite(fid,s(1),'int'); 
	fwrite(fid,s(2),'int'); 
	if s(1)*s(2),
		fwrite(fid,data','double'); %get to the "c" convention, hence the transpose
	end
	% }}}
elseif strcmpi(data_type,'MatArray'), % {{{

	numrecords=numel(data);

	%first get length of record
	recordlength=4+4; %number of records + code
	for i=1:numrecords,
		matrix=data{i};
		s=size(matrix);
		recordlength=recordlength+4*2+... %row and col of matrix
			s(1)*s(2)*8; %matrix of doubles
	end

	%write length of record
	fwrite(fid,recordlength,'int'); 

	%write data code: 
	fwrite(fid,TypeToCode(data_type),'int'); 

	%write data, first number of records
	fwrite(fid,numrecords,'int'); 

	%write each matrix: 
	for i=1:numrecords,
		matrix=data{i};
		s=size(matrix);
		fwrite(fid,s(1),'int'); 
		fwrite(fid,s(2),'int'); 
		fwrite(fid,matrix','double');
	end
	% }}}
elseif strcmpi(data_type,'StringArray'), % {{{

	%first get length of string array: 
	num=numel(data);
	%now get length of record: 
	recordlength=4+4; %for length of array + code
	for i=1:num,
		string=data{i};
		recordlength=recordlength+4+length(string); %for each string
	end

	%write length of record
	fwrite(fid,recordlength,'int'); 

	%write data code: 
	fwrite(fid,TypeToCode(data_type),'int'); 

	%now write length of string array
	fwrite(fid,num,'int'); 

	%now write the strings
	for i=1:num,
		string=data{i};
		fwrite(fid,length(string),'int'); 
		fwrite(fid,string,'char'); 
	end
	% }}}
else 
	error(['WriteData error message: data type: ' num2str(data_type) ' not supported yet! (' EnumToString(enum) ')']);
end
% }}}
function enum=BuildEnum(string) % {{{
%BUILDENUM - build enum out of string
%
%   Usage:
%      enum=BuildEnum(string)

if findstr(string,'icesheet'),
	indices=findstr(string,'icesheet');
	for i=1:length(indices),
		string(indices(i):(indices(i)+7))='IceSheet';
	end
end

if findstr(string,'iceshelf'),
	indices=findstr(string,'iceshelf');
	for i=1:length(indices),
		string(indices(i):(indices(i)+7))='IceShelf';
	end
end

if strncmpi(string,'elementon',9),
	string(1:9)='ElementOn';
	string(10)=upper(string(10));
end

if strncmpi(string,'nodeon',6),
	string(1:6)='NodeOn';
	string(7)=upper(string(7));
end

if strncmpi(string,'numberof',8),
	string(1:8)='NumberOf';
	string(9)=upper(string(9));
end

if findstr(string,'_'),
	indices=findstr(string,'_');
	for i=1:length(indices),
		string(indices(i)+1)=upper(string(indices(i)+1));
	end
	string(indices)=[];
end

if findstr(string,'2d'),
	indices=findstr(string,'2d');
	for i=1:length(indices),
		string(indices(i)+1)=upper(string(indices(i)+1));
	end
end

if findstr(string,'3d'),
	indices=findstr(string,'3d');
	for i=1:length(indices),
		string(indices(i)+1)=upper(string(indices(i)+1));
	end
end

%take first letter of string and make it uppercase: 
string(1)=upper(string(1));

%Get Enum
enum=eval([string 'Enum']); 
% }}}
function code=TypeToCode(data_type) % {{{1
%This routine takes the data_type string, and hardcodes it into an integer, which 
%is passed along the record, in order to identify the nature of the dataset being 
%sent.
if     strcmpi(data_type,'Boolean'),
	code=1;
elseif strcmpi(data_type,'Integer'), 
	code=2;
elseif strcmpi(data_type,'Double'), 
	code=3;
elseif strcmpi(data_type,'String'), 
	code=4;
elseif strcmpi(data_type,'BooleanMat'),
	code=5;
elseif strcmpi(data_type,'IntMat'),
	code=6;
elseif strcmpi(data_type,'DoubleMat'),
	code=7;
elseif strcmpi(data_type,'MatArray'), 
	code=8;
elseif strcmpi(data_type,'StringArray'),
	code=9;
else 
	error('TypeToCode error message: data type not supported yet!');
end% }}}
