/* \file IoModel.h
 * \brief  Header file defining the IoModel structure that will help in processing the input data coming 
 * into ISSM, from Matlab, or through a binary file opened for reading.
 * \sa IoModel.cpp
 */

#ifndef _IOMODEL_H
#define _IOMODEL_H

#include "../include/include.h"
#include "../EnumDefinitions/EnumDefinitions.h"

class Elements;
class IoModel {

	private: 
		FILE* fid; //pointer to input file

	public:
		Parameters* parameters; //this dataset holds all double, int, bool and char* parameters read in from the input file.* 
		/*Data: {{{1*/

		int     nodecounter; //keep track of how many nodes are being created in each analysis type
		int     loadcounter; //keep track of how many loads are being created in each analysis type
		int     constraintcounter; //keep track of how many constraints are being created in each analysis type

		/*2d mesh: */
		double* x;
		double* y;
		double* z;
		double* elements;
		double* elements_type;
		double* vertices_type;
		double* elementconnectivity;

		/*3d mesh: */
		double* elements2d;
		double* upperelements;
		double* lowerelements;

		/*elements type: */
		double* nodeonhutter;
		double* nodeonmacayeal;
		double* nodeonpattyn;

		/*results: */
		double* vx;
		double* vy;
		double* vz;
		double* pressure;
		double* temperature;
		double* waterfraction;

		/*observations: */
		double*  thickness_obs;
		double*  vx_obs;
		double*  vy_obs;
		double*  vz_obs;

		/*geometry: */
		double* elementonbed;
		double* elementonsurface;
		double* nodeonbed;
		double* nodeonsurface;
		double* nodeonstokes;
		double* borderstokes;
		double* thickness;
		double* thickness_coeff;
		double* surface;
		double* bed;
		double* bathymetry;
		double* elementoniceshelf;
		double* elementonwater;
		double* nodeonicesheet;
		double* nodeoniceshelf;
		double* nodeonwater;

		/*friction: */
		double* drag_coefficient;
		double* drag_p;
		double* drag_q;

		/*boundary conditions: */
		double* pressureload;
		double* spcvx;
		double* spcvy;
		double* spcvz;
		double* spctemperature;
		double* spcthickness;
		double* spcwatercolumn;
		double* diagnostic_ref;
		double* geothermalflux;
		double* edges;

		/*materials: */
		double* rheology_B;
		double* rheology_n;

		/*numerical parameters: */
		double* cm_min;
		double* cm_max;

		/*control methods: */
		double*  control_type;

		/*solution parameters: */
		double* cm_responses;
		double* weights;
		double* cm_jump;
		double* maxiter;
		double* optscal;

		/*rifts: */
		double*  riftinfo;

		/*penalties: */
		double*  penalties;

		/*basal: */
		double*  basal_melting_rate;
		double*  watercolumn;
		double*  basal_melting_rate_correction;
		double*  surface_mass_balance;
		double*  surface_accumulation_rate;
		double*  surface_ablation_rate;

		double*  dhdt;

		/*exterior partitioning data, to be carried around: */
		bool*   my_elements;
		bool*   my_nodes;
		int*    my_vertices;
		int*    singlenodetoelementconnectivity;
		int*    numbernodetoelementconnectivity;

		 /*}}}*/
		/*Methods: {{{1*/
		~IoModel();
		IoModel();
		IoModel(FILE* iomodel_handle);
		void IoModelInit(void);
		/*}}}*/
		/*Input/Output:{{{1*/
		void  FetchData(bool*     pboolean,int data_enum);
		void  FetchData(int*      pinteger,int data_enum);
		void  FetchData(double*   pscalar,int data_enum);
		void  FetchData(char**    pstring,int data_enum);
		void  FetchData(int** pmatrix,int* pM,int* pN,int data_enum);
		void  FetchData(double**  pscalarmatrix,int* pM,int* pN,int data_enum);
		void  FetchData(char***   pstringarray,int* pnumstrings,int data_enum);
		void  FetchData(double*** pmatrixarray,int** pmdims,int** pndims, int* pnumrecords,int data_enum);
		void  FetchParameters(Parameters* parameters);
		FILE* SetFilePointerToData(int* pcode,int* pvector_type, int data_enum);
		void  FetchDataToInput(Elements* elements,int vector_enum,int default_vector_enum=NoneEnum,double default_value=0);
		/*}}}*/

};

#endif  /* _IOMODEL_H */
