/*!\file: CreateParametersQmu.cpp
 * \brief general driver for creating parameters dataset
 */ 

#include "../../../Container/Container.h"
#include "../../../toolkits/toolkits.h"
#include "../../../EnumDefinitions/EnumDefinitions.h"
#include "../../../objects/objects.h"
#include "../../../shared/shared.h"
#include "../../../io/io.h"
#include "../../../include/include.h"
#include "../../MeshPartitionx/MeshPartitionx.h"
#include "../ModelProcessorx.h"

void CreateParametersQmu(Parameters** pparameters,IoModel* iomodel,int solution_type,int analysis_type){

	/*variable declarations: {{{1*/
	int i,j,k;
	
	Parameters* parameters = NULL;
	int      second_count;
	
	int*     part=NULL;
	double*  dpart=NULL;

	char**  responsedescriptors=NULL;
	int     numresponsedescriptors;
	char**  variabledescriptors=NULL;
	int     numvariabledescriptors;
	char*   descriptor=NULL;
	double* dakota_parameter=NULL;

	//qmu files
	char* qmuinname=NULL;
	char* qmuerrname=NULL;
	char* qmuoutname=NULL;
	
	//descriptors:
	char tag[50];
	
	/*parameters for mass flux: */
	bool     qmu_mass_flux_present=false;
	int qmu_mass_flux_num_profiles=0;
	double** array=NULL;
	int*     mdims_array=NULL;
	int*     ndims_array=NULL;
	
	double*  matrix=NULL;
	double*  temp_matrix=NULL;
	int      M;
	int      temp_m,temp_n;
	int      m,n;
	int      count;
	bool     qmu_analysis=false;
	char*    name=NULL;
	int      numberofresponses;
	int      numberofvertices;

	/*}}}*/
	
	/*recover parameters : */
	parameters=*pparameters;

	/*recover parameters: */
	parameters->FindParam(&qmu_analysis,QmuAnalysisEnum);
	parameters->FindParam(&name,NameEnum);
	parameters->FindParam(&numberofresponses,NumberOfResponsesEnum);
	parameters->FindParam(&numberofvertices,NumberOfVerticesEnum);
	if(qmu_analysis){

		/*name of qmu input, error and output files:{{{1*/
		qmuinname=(char*)xmalloc((strlen(name)+strlen(".qmu.in")+1)*sizeof(char));
		sprintf(qmuinname,"%s%s",name,".qmu.in");
		parameters->AddObject(new   StringParam(QmuInNameEnum,qmuinname));

		qmuoutname=(char*)xmalloc((strlen(name)+strlen(".qmu.out")+1)*sizeof(char));
		sprintf(qmuoutname,"%s%s",name,".qmu.out");
		parameters->AddObject(new   StringParam(QmuOutNameEnum,qmuoutname));

		qmuerrname=(char*)xmalloc((strlen(name)+strlen(".qmu.err")+1)*sizeof(char));
		sprintf(qmuerrname,"%s%s",name,".qmu.err");
		parameters->AddObject(new   StringParam(QmuErrNameEnum,qmuerrname));
		/*}}}*/
		/*Fetch variable descriptors: {{{1*/
		iomodel->FetchData(&variabledescriptors,&numvariabledescriptors,VariabledescriptorsEnum);

		/*Ok, we have all the variable descriptors. Build a parameter with it: */
		parameters->AddObject(new StringArrayParam(VariabledescriptorsEnum,variabledescriptors,numvariabledescriptors));

		/*}}}*/
		/*Fetch response descriptors: {{{1*/
		iomodel->FetchData(&responsedescriptors,&numresponsedescriptors,ResponsedescriptorsEnum);

		/*Ok, we have all the response descriptors. Build a parameter with it: */
		parameters->AddObject(new StringArrayParam(ResponsedescriptorsEnum,responsedescriptors,numresponsedescriptors));
		parameters->AddObject(new    IntParam(QmuNumberOfResponsesEnum,numberofresponses));
		/*}}}*/
		/*Deal with partitioning: {{{1*/
		/*partition vertices in iomodel->qmu_npart parts, unless a partition is already present: */
		
		iomodel->FetchData(&dpart,NULL,NULL,PartEnum);

		if(!dpart){

			/*Partition elements and vertices and nodes: */
			ElementsAndVerticesPartitioning(&iomodel->my_elements,&iomodel->my_vertices,iomodel);

			dpart=(double*)xmalloc(numberofvertices*sizeof(double));
			for(i=0;i<numberofvertices;i++)dpart[i]=iomodel->my_vertices[i];
		}
		parameters->AddObject(new DoubleVecParam(QmuPartEnum,dpart,numberofvertices));
		/*}}}*/
		/*Deal with data needed because of qmu variables: {{{1*/
		
		for(i=0;i<numvariabledescriptors;i++){

			if (strncmp(variabledescriptors[i],"scaled_",7)==0){
				/*Ok, we are dealing with a variable that is distributed over nodes. Recover the name of the variable (ex: scaled_Thickness): */
				sscanf(variabledescriptors[i],"scaled_%s",tag);
				
				/*Recover data: */
				iomodel->FetchData(&dakota_parameter,NULL,NULL,StringToEnumx(tag));

				/*Convert units: */
				UnitConversion(dakota_parameter,numberofvertices,ExtToIuEnum,StringToEnumx(tag));

				/*Add to parameters: */
				parameters->AddObject(new DoubleVecParam(StringToEnumx(tag),dakota_parameter,numberofvertices));
				
				/*Free ressources:*/
				xfree((void**)&dakota_parameter);
			}
		}
		/*}}}*/
		/*Deal with data needed to compute qmu responses: {{{1*/
		for(i=0;i<numresponsedescriptors;i++){
			
			if(strncmp(responsedescriptors[i],"indexed_MassFlux",16)==0){
				qmu_mass_flux_present=true;
			}
		}
		
		
		if(qmu_mass_flux_present){
			
			/*Fetch the mass flux segments necessary to compute the mass fluxes.  Build a DoubleMatArrayParam object out of them: */ 
			iomodel->FetchData(&array,&mdims_array,&ndims_array,&qmu_mass_flux_num_profiles,QmuMassFluxSegmentsEnum);
			if(qmu_mass_flux_num_profiles==0)_error_(" qmu_mass_flux_num_profiles is 0, when MassFlux computations were requested!");

			/*Go through segments, and extract those that belong to this cpu: */
			for(i=0;i<qmu_mass_flux_num_profiles;i++){
				temp_matrix=array[i];
				temp_m=mdims_array[i];
				temp_n=ndims_array[i];

				m=0;
				for(j=0;j<temp_m;j++){
					if (  iomodel->my_elements[(int)(*(temp_matrix+5*j+4))-1])m++;
				}
				if(m){
					matrix=(double*)xcalloc(5*m,sizeof(double));
					count=0;
					for(j=0;j<temp_m;j++){
						if (iomodel->my_elements[(int)*(temp_matrix+5*j+4)-1]){
							for(k=0;k<5;k++)*(matrix+5*count+k)=*(temp_matrix+5*j+k);
							count++;
						}
					}
				}
				else{
					matrix=NULL;
				}

				/*Assign: */
				array[i]=matrix;
				mdims_array[i]=m;
				ndims_array[i]=5;

				/*Free temporary matrix: */
				xfree((void**)&temp_matrix);
			}

			/*Ok, we have an array of segments, different on every cpu. Create a DoubleMatArrayParam object with it: */
			parameters->AddObject(new DoubleMatArrayParam(QmuMassFluxSegmentsEnum,array,qmu_mass_flux_num_profiles,mdims_array,ndims_array));

			/*Free data: */
			for(i=0;i<qmu_mass_flux_num_profiles;i++){
				double* matrix=array[i];
				xfree((void**)&matrix);
			}
			xfree((void**)&mdims_array); 
			xfree((void**)&ndims_array);
			xfree((void**)&array);
		}
		/*}}}*/
		/*Free data: {{{1*/
		for(i=0;i<numresponsedescriptors;i++){
			descriptor=responsedescriptors[i];
			xfree((void**)&descriptor);
		}
		xfree((void**)&responsedescriptors);

		for(i=0;i<numvariabledescriptors;i++){
			descriptor=variabledescriptors[i];
			xfree((void**)&descriptor);
		}
		xfree((void**)&variabledescriptors);

		xfree((void**)&iomodel->elements);
		xfree((void**)&iomodel->elements2d);
		xfree((void**)&part);
		xfree((void**)&dpart);
		xfree((void**)&qmuinname);
		xfree((void**)&qmuerrname);
		xfree((void**)&qmuoutname);
		xfree((void**)&name);
		/*}}}*/
	} //if(qmu_analysis)

	/*Assign output pointer: */
	*pparameters=parameters;
}
