/*! \file IoModel.cpp
 * \brief  file containing the methods that will help in processing the input data coming 
 * into ISSM, from Matlab, or through a binary file opened for reading.
 */

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <string.h>
#include <stdio.h>
#include "./objects.h"
#include "../shared/shared.h"
#include "../io/io.h"
#include "../include/include.h"

/*FUNCTION IoModel::IoModel(){{{1*/
IoModel::IoModel(){
	this->IoModelInit();
}
/*}}}*/
/*FUNCTION IoModel::~IoModel(){{{1*/
IoModel::~IoModel(){
	
	int i;

	/*!Two cases here: 
	 * - serial mode: matlab's memory manager will take care of delete iomodel when returning from Imp. Do nothing here, so as not to confuse 
	 *                the memory manager.
     * - in parallel, anything the io layer does (FetchData) did needs to be erased explicitely in the iomodel.
	 */

	#ifdef _PARALLEL_
	xfree((void**)&this->x);
	xfree((void**)&this->y);
	xfree((void**)&this->z);
	xfree((void**)&this->elements);
	xfree((void**)&this->elementconnectivity);
	xfree((void**)&this->elements_type);
	xfree((void**)&this->vertices_type);
	xfree((void**)&this->nodeonhutter);
	xfree((void**)&this->nodeonmacayeal);
	if (this->dim==3){
		xfree((void**)&this->elements2d);
		xfree((void**)&this->upperelements);
		xfree((void**)&this->lowerelements);
		xfree((void**)&this->nodeonpattyn);
	}
	xfree((void**)&this->elementonbed);
	xfree((void**)&this->elementonsurface);
	xfree((void**)&this->nodeonbed);
	xfree((void**)&this->nodeonsurface);
	xfree((void**)&this->nodeonstokes);
	xfree((void**)&this->borderstokes);
	xfree((void**)&this->thickness_obs);
	xfree((void**)&this->thickness_coeff);
	xfree((void**)&this->thickness);
	xfree((void**)&this->surface);
	xfree((void**)&this->bed);
	xfree((void**)&this->bathymetry);
	xfree((void**)&this->vx_obs);
	xfree((void**)&this->vy_obs);
	xfree((void**)&this->vz_obs);
	xfree((void**)&this->vx);
	xfree((void**)&this->vy);
	xfree((void**)&this->vz);
	xfree((void**)&this->pressure);
	xfree((void**)&this->temperature);
	xfree((void**)&this->waterfraction);
	xfree((void**)&this->drag_coefficient);
	xfree((void**)&this->drag_p);
	xfree((void**)&this->drag_q);
	xfree((void**)&this->elementoniceshelf);
	xfree((void**)&this->elementonwater);
	xfree((void**)&this->nodeonicesheet);
	xfree((void**)&this->nodeoniceshelf);
	xfree((void**)&this->nodeonwater);
	xfree((void**)&this->pressureload);
	xfree((void**)&this->spcvx);
	xfree((void**)&this->spcvy);
	xfree((void**)&this->spcvz);
	xfree((void**)&this->spcthickness);
	xfree((void**)&this->spcwatercolumn);
	xfree((void**)&this->spctemperature);
	xfree((void**)&this->diagnostic_ref);
	xfree((void**)&this->edges);
	xfree((void**)&this->geothermalflux);
	xfree((void**)&this->basal_melting_rate);
	xfree((void**)&this->watercolumn);
	xfree((void**)&this->basal_melting_rate_correction);
	xfree((void**)&this->surface_accumulation_rate);
	xfree((void**)&this->surface_ablation_rate);
	xfree((void**)&this->surface_mass_balance);
	xfree((void**)&this->dhdt);
	xfree((void**)&this->rheology_B);
	xfree((void**)&this->rheology_n);
	xfree((void**)&this->control_type);
	xfree((void**)&this->cm_responses);
	xfree((void**)&this->weights);
	xfree((void**)&this->cm_jump);
	xfree((void**)&this->cm_min);
	xfree((void**)&this->cm_max);
	xfree((void**)&this->optscal);
	xfree((void**)&this->maxiter);


	/*!Delete structure fields: */
	xfree((void**)&this->inputfilename);
	xfree((void**)&this->outputfilename);
	xfree((void**)&this->repository);
	xfree((void**)&this->name);
	xfree((void**)&this->riftinfo);
	xfree((void**)&this->penalties);
	
	/*exterior data: */
	xfree((void**)&this->my_elements);
	xfree((void**)&this->my_vertices);
	xfree((void**)&this->my_nodes);
	xfree((void**)&this->singlenodetoelementconnectivity);
	xfree((void**)&this->numbernodetoelementconnectivity);
	#endif
}
/*}}}*/
/*FUNCTION IoModel::IoModel(FILE*  iomodel_handle){{{1*/
IoModel::IoModel(FILE* iomodel_handle){

	/*First, keep track of the fil handle: */
	this->fid=iomodel_handle;
	
	int i,j;
		
	/*First, initialize the structure: */
	this->IoModelInit();
	
	/*Get all the data that consists of scalars, integers and strings: */

	this->FetchData(&this->name,NameEnum); 
	this->FetchData(&this->inputfilename,InputfilenameEnum); 
	this->FetchData(&this->outputfilename,OutputfilenameEnum); 
	this->FetchData(&this->qmu_analysis,QmuAnalysisEnum); 
	this->FetchData(&this->control_analysis,ControlAnalysisEnum); 
	this->FetchData(&this->dim,DimEnum);
	/*!Get numberofelements and numberofvertices: */
	this->FetchData(&this->numberofvertices,NumberOfNodesEnum);
	this->FetchData(&this->numberofelements,NumberOfElementsEnum);
	/*!In case we are running 3d, we are going to need the collapsed and non-collapsed 2d meshes, from which the 3d mesh was extruded: */
	if (this->dim==3){
	
		/*!Deal with 2d mesh: */
		this->FetchData(&this->numberofelements2d,NumberOfElements2DEnum);
		this->FetchData(&this->numberofvertices2d,NumberOfNodes2DEnum);
		this->FetchData(&this->numlayers,NumlayersEnum);
	}


	/*elements type: */
	this->FetchData(&this->ishutter,IshutterEnum);
	this->FetchData(&this->ismacayealpattyn,IsmacayealpattynEnum);
	this->FetchData(&this->isstokes,IsstokesEnum);

	/*!Get drag_type, drag and p,q: */
	this->FetchData(&this->drag_type,DragTypeEnum);

	/*!Get materials: */
	this->FetchData(&this->rho_water,RhoWaterEnum);
	this->FetchData(&this->rho_ice,RhoIceEnum);
	this->FetchData(&this->g,GEnum);

	/*Get control parameters: */
	this->FetchData(&this->num_control_type,NumControlTypeEnum); 
	this->FetchData(&this->num_cm_responses,NumCmResponsesEnum); 

	/*!Get solution parameters: */
	this->FetchData(&this->yts,YtsEnum);
	this->FetchData(&this->meanvel,MeanvelEnum);
	this->FetchData(&this->epsvel,EpsvelEnum);
	this->FetchData(&this->verbose,VerboseBinaryEnum);
	this->FetchData(&this->output_frequency,OutputFrequencyEnum);
	this->FetchData(&this->prognostic_DG,PrognosticDGEnum);
	this->FetchData(&this->nsteps,NstepsEnum);
	this->FetchData(&this->eps_cm,EpsCmEnum);
	this->FetchData(&this->tolx,TolxEnum);
	this->FetchData(&this->cm_gradient,CmGradientEnum);
	this->FetchData(&this->eps_res,EpsResEnum);
	this->FetchData(&this->eps_rel,EpsRelEnum);
	this->FetchData(&this->eps_abs,EpsAbsEnum);
	this->FetchData(&this->max_nonlinear_iterations,MaxNonlinearIterationsEnum);
	this->FetchData(&this->max_steadystate_iterations,MaxSteadystateIterationsEnum);
	this->FetchData(&this->dt,DtEnum);
	this->FetchData(&this->ndt,NdtEnum);
	this->FetchData(&this->time_adapt,TimeAdaptEnum);
	this->FetchData(&this->cfl_coefficient,CflCoefficientEnum);
	this->FetchData(&this->hydrostatic_adjustment,HydrostaticAdjustmentEnum);
	this->FetchData(&this->penalty_offset,PenaltyOffsetEnum);
	this->FetchData(&this->penalty_melting,PenaltyMeltingEnum);
	this->FetchData(&this->penalty_lock,PenaltyLockEnum);
	this->FetchData(&this->sparsity,SparsityEnum);
	this->FetchData(&this->connectivity,ConnectivityEnum);
	this->FetchData(&this->lowmem,LowmemEnum);
	this->FetchData(&this->viscosity_overshoot,ViscosityOvershootEnum);
	this->FetchData(&this->artdiff,ArtificialDiffusivityEnum);
	this->FetchData(&this->prognostic_DG,PrognosticDGEnum);
	this->FetchData(&this->stokesreconditioning,StokesreconditioningEnum);
	this->FetchData(&this->shelf_dampening,ShelfDampeningEnum);
	this->FetchData(&this->waitonlock,WaitonlockEnum);
	this->FetchData(&this->gl_migration,GlMigrationEnum);
	this->FetchData(&this->isdiagnostic,IsdiagnosticEnum); 
	this->FetchData(&this->isprognostic,IsprognosticEnum); 
	this->FetchData(&this->isthermal,IsthermalEnum); 

	/*!Get thermal parameters: */
	this->FetchData(&this->beta,BetaEnum);
	this->FetchData(&this->meltingpoint,MeltingpointEnum);
	this->FetchData(&this->referencetemperature,ReferencetemperatureEnum);
	this->FetchData(&this->latentheat,LatentheatEnum);
	this->FetchData(&this->heatcapacity,HeatcapacityEnum);
	this->FetchData(&this->thermalconductivity,ThermalconductivityEnum);
	this->FetchData(&this->min_thermal_constraints,MinThermalConstraintsEnum);
	this->FetchData(&this->min_mechanical_constraints,MinMechanicalConstraintsEnum);
	this->FetchData(&this->stabilize_constraints,StabilizeConstraintsEnum);
	this->FetchData(&this->mixed_layer_capacity,MixedLayerCapacityEnum);
	this->FetchData(&this->thermal_exchange_velocity,ThermalExchangeVelocityEnum);
	this->FetchData(&this->basal_melting_rate_correction_apply,BasalMeltingRateCorrectionApplyEnum);
	this->FetchData(&this->gl_melting_rate,GlMeltingRateEnum);
	this->FetchData(&this->rheology_law,RheologyLawEnum);

	/*!Get hydrology parameters: */		 
	this->FetchData(&this->hydro_kn,HydroKnEnum);			 
	this->FetchData(&this->hydro_p,HydroPEnum);		 
	this->FetchData(&this->hydro_q,HydroQEnum);		 
	this->FetchData(&this->hydro_CR,HydroCREnum);		 
	this->FetchData(&this->hydro_n,HydroNEnum); 

	/*qmu: */
	if(this->qmu_analysis){
		this->FetchData(&this->numberofvariables,NumberOfVariablesEnum);
		this->FetchData(&this->numberofresponses,NumberOfResponsesEnum);
		this->FetchData(&this->qmu_npart,NpartEnum);
		this->FetchData(&this->qmu_save_femmodel,QmuSaveFemmodelEnum); 
	}

	/*i/o: */
	this->FetchData(&this->io_gather,IoGatherEnum);
}
/*}}}*/
/*FUNCTION IoModel::IoModelInit{{{1*/
void IoModel::IoModelInit(void){
	
	/*!initialize all pointers to 0: */
	this->name=NULL;
	this->inputfilename=NULL;
	this->outputfilename=NULL;
	this->repository=NULL;
	this->qmu_analysis=0;
	this->control_analysis=0;
	this->numberofvariables=0;
	this->numvariabledescriptors=0;
	this->numberofresponses=0;
	this->numresponsedescriptors=0;
	this->qmu_npart=0; 
	this->numberofelements=0;
	this->numberofvertices=0;
	this->x=NULL; 
	this->y=NULL;
	this->z=NULL;
	this->elements=NULL;
	this->elementconnectivity=NULL;
	this->elements_type=NULL;
	this->vertices_type=NULL;
	this->numberofvertices2d=0;
	this->elements2d=NULL;
	this->numlayers=0;
	this->upperelements=NULL;
	this->lowerelements=NULL;
	this->nodeonhutter=NULL;
	this->nodeonmacayeal=NULL;
	this->nodeonpattyn=NULL;
	this->io_gather=1;
	
	this->vx_obs=NULL;
	this->vy_obs=NULL;
	this->vz_obs=NULL;
	this->vx=NULL;
	this->vy=NULL;
	this->vz=NULL;
	this->pressure=NULL;
	this->temperature=NULL;
	this->waterfraction=NULL;
	this->gl_melting_rate=0;
	this->basal_melting_rate=NULL;
	this->watercolumn=NULL;
	this->basal_melting_rate_correction=NULL;
	this->basal_melting_rate_correction_apply=0;
	this->geothermalflux=NULL;
	this->elementonbed=NULL;
	this->elementonsurface=NULL;
	this->nodeonbed=NULL;
	this->nodeonsurface=NULL;
	this->nodeonstokes=NULL;
	this->borderstokes=NULL;
	this->thickness_obs=NULL;
	this->thickness_coeff=NULL;
	this->thickness=NULL;
	this->surface=NULL;
	this->bed=NULL;
	this->bathymetry=NULL;
	this->elementoniceshelf=NULL;
	this->elementonwater=NULL;
	this->nodeonicesheet=NULL;
	this->nodeoniceshelf=NULL;
	this->nodeonwater=NULL;

	this->drag_type=0;
	this->drag_coefficient=NULL;
	this->drag_p=NULL;
	this->drag_q=NULL;
	
	
	this->numberofpressureloads=0;
	this->pressureload=NULL;
	this-> spcvx=NULL;
	this-> spcvy=NULL;
	this-> spcvz=NULL;
	this-> spctemperature=NULL;
	this-> spcthickness=NULL;
	this-> spcwatercolumn=NULL;
	this-> diagnostic_ref=NULL;
	this->numberofedges=0;
	this->edges=NULL;
	
	/*!materials: */
	this->rho_water=0;
	this->rho_ice=0;
	this->g=0;
	this->rheology_n=NULL;
	this->rheology_B=NULL;
	this->rheology_law=0;

	/*!solution parameters: */
	this->control_type=NULL;
	this->cm_responses=NULL;
	this->weights=NULL;
	this->cm_jump=NULL;
	this->meanvel=0;
	this->epsvel=0;
	this->nsteps=0;
	this->eps_cm=0;
	this->tolx=0;
	this->maxiter=NULL;
	this->cm_min=NULL;
	this->cm_max=NULL;
	this->cm_gradient=0;
	this->verbose=0;
	this->output_frequency=0;
	this->eps_res=0;
	this->eps_rel=0;
	this->eps_abs=0;
	this->max_nonlinear_iterations=0;
	this->max_steadystate_iterations=0;
	this->dt=0;
	this->ndt=0;
	this->time_adapt=0;
	this->cfl_coefficient=0;
	this->hydrostatic_adjustment=0;
	this->gl_migration=0;
	this->penalty_offset=0;
	this->penalty_melting=0;
	this->penalty_lock=0;
	this->sparsity=0;
	this->connectivity=0;
	this->lowmem=0;
	this->optscal=NULL;
	this->yts=0;
	this->viscosity_overshoot=0;
	this->artdiff=0;
	this->prognostic_DG=0;
	this->stokesreconditioning=0;
	this->shelf_dampening=0;
	this->waitonlock=0;
	this->isdiagnostic=0;
	this->isprognostic=0;
	this->isthermal=0;

	/*!thermal parameters: */
	this->beta=0;
	this->meltingpoint=0;
	this->referencetemperature=0;
	this->latentheat=0;
	this->heatcapacity=0;
	this->thermalconductivity=0;
	this->min_thermal_constraints=0;
	this->min_mechanical_constraints=0;
	this->stabilize_constraints=0;
	this->mixed_layer_capacity=0;
	this->thermal_exchange_velocity=0;

	
	this->numrifts=0;
	this->riftinfo=NULL;

	/*!penalties: */
	this->numpenalties=0;
	this->penalties=NULL;

	/*!surface: */
	this->surface_mass_balance=NULL;
   this->surface_accumulation_rate=NULL;
	this->surface_ablation_rate=NULL;
	this->dhdt=NULL;

	/*elements type: */
	this->ishutter=0;
	this->ismacayealpattyn=0;
	this->isstokes=0;

	/*exterior data: */
	this->my_elements=NULL;
	this->my_vertices=NULL;
	this->my_nodes=NULL;
	this->singlenodetoelementconnectivity=NULL;
	this->numbernodetoelementconnectivity=NULL;
	this->nodecounter=0;
	this->loadcounter=0;
	this->constraintcounter=0;
}
/*}}}*/
/*FUNCTION IoModel::FetchData(bool*     pbool,int data_enum){{{1*/
void  IoModel::FetchData(bool* pboolean,int data_enum){

	extern int my_rank;
	extern int num_procs;
	

	/*output: */
	bool  boolean;
	int   code;
	
	/*Set file pointer to beginning of the data: */
	fid=this->SetFilePointerToData(&code,NULL,data_enum);
	
	if(code!=1)_error_("%s%s","IoModel::FetchData expecting a boolean for enum ",EnumToStringx(data_enum));
	
	/*We have to read a boolean from disk. */
	if(my_rank==0){  
		if(fread(&boolean,sizeof(bool),1,fid)!=1) _error_(" could not read boolean ");
	}

	MPI_Bcast(&boolean,1,MPI_BYTE,0,MPI_COMM_WORLD); 

	/*Assign output pointers: */
	*pboolean=boolean;

}
/*}}}*/
/*FUNCTION IoModel::FetchData(int*      pinteger,int data_enum){{{1*/
void  IoModel::FetchData(int* pinteger,int data_enum){

	extern int my_rank;
	extern int num_procs;
	

	/*output: */
	int   integer;
	int   code;
	
	/*Set file pointer to beginning of the data: */
	fid=this->SetFilePointerToData(&code,NULL,data_enum);
	
	if(code!=2)_error_("%s%s","IoModel::FetchData expecting an integer for enum ",EnumToStringx(data_enum));
	
	/*We have to read a integer from disk. First read the dimensions of the integer, then the integer: */
	if(my_rank==0){  
		if(fread(&integer,sizeof(int),1,fid)!=1) _error_(" could not read integer ");
	}

	MPI_Bcast(&integer,1,MPI_INT,0,MPI_COMM_WORLD); 

	/*Assign output pointers: */
	*pinteger=integer;

}
/*}}}*/
/*FUNCTION IoModel::FetchData(double*   pscalar,int data_enum){{{1*/
void  IoModel::FetchData(double* pscalar,int data_enum){


	extern int my_rank;
	extern int num_procs;
	

	/*output: */
	double   scalar;
	int      code;
	
	/*Set file pointer to beginning of the data: */
	fid=this->SetFilePointerToData(&code,NULL,data_enum);
	
	if(code!=3)_error_("%s%s","IoModel::FetchData expecting a double for enum ",EnumToStringx(data_enum));
	
	/*We have to read a scalar from disk. First read the dimensions of the scalar, then the scalar: */
	if(my_rank==0){
		if(fread(&scalar,sizeof(double),1,fid)!=1)_error_(" could not read scalar ");
	}
	MPI_Bcast(&scalar,1,MPI_DOUBLE,0,MPI_COMM_WORLD); 

	/*Assign output pointers: */
	*pscalar=scalar;
		 
}
/*}}}*/
/*FUNCTION IoModel::FetchData(char**    pstring,int data_enum){{{1*/
void  IoModel::FetchData(char** pstring,int data_enum){

	extern int my_rank;
	extern int num_procs;
	

	/*output: */
	char* string=NULL;
	int   string_size;
	int code=0;
	
	/*Set file pointer to beginning of the data: */
	fid=this->SetFilePointerToData(&code,NULL,data_enum);
	
	if(code!=4)_error_("%s%s","IoModel::FetchData expecting a string for enum ",EnumToStringx(data_enum));
	
	/*Now fetch: */
	
	/*We have to read a string from disk. First read the dimensions of the string, then the string: */
	if(my_rank==0){  
		if(fread(&string_size,sizeof(int),1,fid)!=1) _error_(" could not read length of string ");
	}

	MPI_Bcast(&string_size,1,MPI_INT,0,MPI_COMM_WORLD); 

	/*Now allocate string: */
	if(string_size){
		string=(char*)xmalloc((string_size+1)*sizeof(char));
		string[string_size]='\0';

		/*Read string on node 0, then broadcast: */
		if(my_rank==0){  
			if(fread(string,string_size*sizeof(char),1,fid)!=1)_error_("  could not read string ");
		}
		MPI_Bcast(string,string_size,MPI_CHAR,0,MPI_COMM_WORLD); 
	}
	else{
		string=(char*)xmalloc(sizeof(char));
		string[0]='\0';
	}


	/*Assign output pointers: */
	*pstring=string;
}
/*}}}*/
/*FUNCTION IoModel::FetchData(double**  pdoublematrix,int* pM,int* pN,int data_enum){{{1*/
void  IoModel::FetchData(double** pmatrix,int* pM,int* pN,int data_enum){

	extern int my_rank;
	extern int num_procs;

	/*output: */
	int M,N;
	double* matrix=NULL;
	int code=0;
	int vector_type=0;
	
	
	
	/*Set file pointer to beginning of the data: */
	fid=this->SetFilePointerToData(&code,&vector_type,data_enum);

	if((code!=5) && (code!=6) && (code!=7))_error_("%s%s","IoModel::FetchData expecting a double, integer or boolean matrix for enum ",EnumToStringx(data_enum));
	
	/*Now fetch: */

	/*We have to read a matrix from disk. First read the dimensions of the matrix, then the whole matrix: */
	/*numberofelements: */
	if(my_rank==0){  
		if(fread(&M,sizeof(int),1,fid)!=1) _error_("could not read number of rows for matrix ");
	}

	MPI_Bcast(&M,1,MPI_INT,0,MPI_COMM_WORLD); 

	if(my_rank==0){  
		if(fread(&N,sizeof(int),1,fid)!=1) _error_("could not read number of columns for matrix ");
	}
	MPI_Bcast(&N,1,MPI_INT,0,MPI_COMM_WORLD); 

	/*Now allocate matrix: */
	if(M*N){
		matrix=(double*)xmalloc(M*N*sizeof(double));

		/*Read matrix on node 0, then broadcast: */
		if(my_rank==0){  
			if(fread(matrix,M*N*sizeof(double),1,fid)!=1) _error_("could not read matrix ");
		}
		
		MPI_Bcast(matrix,M*N,MPI_DOUBLE,0,MPI_COMM_WORLD); 
	}

	/*Assign output pointers: */
	*pmatrix=matrix;
	if (pM)*pM=M;
	if (pN)*pN=N;

}
/*}}}*/
/*FUNCTION IoModel::FetchData(char***   pstrings,int* pnumstrings,int data_enum){{{1*/
void  IoModel::FetchData(char*** pstrings,int* pnumstrings,int data_enum){

	extern int my_rank;
	extern int num_procs;
	
	int i;

	/*output: */
	int   numstrings=0;
	char** strings=NULL;
	
	/*intermediary: */
	char* string=NULL;
	int   string_size;
	int   code;
	
	/*Set file pointer to beginning of the data: */
	fid=this->SetFilePointerToData(&code,NULL,data_enum);
	
	if(code!=9)_error_("%s%s","IoModel::FetchData expecting a string array for enum ",EnumToStringx(data_enum));
	
	/*We have to read a bunch of strings from disk. First read the number of strings, and allocate: */
	if(my_rank==0){  
		if(fread(&numstrings,sizeof(int),1,fid)!=1) _error_(" could not read length of string array");
	}
	MPI_Bcast(&numstrings,1,MPI_INT,0,MPI_COMM_WORLD); 

	/*Now allocate string array: */
	if(numstrings){
		strings=(char**)xmalloc(numstrings*sizeof(char*));
		for(i=0;i<numstrings;i++)strings[i]=NULL;

		/*Go through strings, and read: */
		for(i=0;i<numstrings;i++){
			
			if(my_rank==0){  
				if(fread(&string_size,sizeof(int),1,fid)!=1) _error_(" could not read length of string ");
			}
			MPI_Bcast(&string_size,1,MPI_INT,0,MPI_COMM_WORLD); 
			if(string_size){
				string=(char*)xmalloc((string_size+1)*sizeof(char));
				string[string_size]='\0';

				/*Read string on node 0, then broadcast: */
				if(my_rank==0){  
					if(fread(string,string_size*sizeof(char),1,fid)!=1)_error_("  could not read string ");
				}
				MPI_Bcast(string,string_size,MPI_CHAR,0,MPI_COMM_WORLD); 
			}
			else{
				string=(char*)xmalloc(sizeof(char));
				string[0]='\0';
			}

			strings[i]=string;
		}
	}

	/*Assign output pointers: */
	*pstrings=strings;
	*pnumstrings=numstrings;
}
/*}}}*/
/*FUNCTION IoModel::FetchData(double*** pmatrices,int** pmdims,int** pndims, int* pM,int data_enum){{{1*/
void  IoModel::FetchData(double*** pmatrices,int** pmdims,int** pndims, int* pnumrecords,int data_enum){

	int i;

	extern int my_rank;
	extern int num_procs;

	/*output: */
	double** matrices=NULL;
	int*     mdims=NULL;
	int*     ndims=NULL;
	int      numrecords=0;

	/*intermediary: */
	int M,N;
	double* matrix=NULL;
	
	
	int   code;
	
	/*Set file pointer to beginning of the data: */
	fid=this->SetFilePointerToData(&code,NULL,data_enum);
	
	if(code!=8)_error_("%s%s","IoModel::FetchData expecting a double mat  array for enum ",EnumToStringx(data_enum));
	
	/*Now fetch: */
	if(my_rank==0){  
		if(fread(&numrecords,sizeof(int),1,fid)!=1) _error_("could not read number of records in matrix array ");
	}
	MPI_Bcast(&numrecords,1,MPI_INT,0,MPI_COMM_WORLD); 

	if(numrecords){

		/*Allocate matrices :*/
		matrices=(double**)xmalloc(numrecords*sizeof(double*));
		mdims=(int*)xmalloc(numrecords*sizeof(int));
		ndims=(int*)xmalloc(numrecords*sizeof(int));

		for(i=0;i<numrecords;i++){
			matrices[i]=NULL;
			mdims[i]=0;
			ndims[i]=0;
		}

		/*Loop through records and fetch matrix: */
		for(i=0;i<numrecords;i++){

			if(my_rank==0){  
				if(fread(&M,sizeof(int),1,fid)!=1) _error_("%s%i%s","could not read number of rows in ",i,"th matrix of matrix array");
			}
			MPI_Bcast(&M,1,MPI_INT,0,MPI_COMM_WORLD); 

			if(my_rank==0){  
				if(fread(&N,sizeof(int),1,fid)!=1) _error_("%s%i%s","could not read number of columns in ",i,"th matrix of matrix array");
			}
			MPI_Bcast(&N,1,MPI_INT,0,MPI_COMM_WORLD); 

			/*Now allocate matrix: */
			if(M*N){
				matrix=(double*)xmalloc(M*N*sizeof(double));

				/*Read matrix on node 0, then broadcast: */
				if(my_rank==0){  
					if(fread(matrix,M*N*sizeof(double),1,fid)!=1) _error_("could not read matrix ");
				}

				MPI_Bcast(matrix,M*N,MPI_DOUBLE,0,MPI_COMM_WORLD); 
			}

			/*Assign: */
			matrices[i]=matrix;
			mdims[i]=M;
			ndims[i]=N;
		}
	}

	/*Assign output pointers: */
	*pmatrices=matrices;
	*pmdims=mdims;
	*pndims=ndims;
	*pnumrecords=numrecords;
}
/*}}}*/
/*FUNCTION IoModel::SetFilePointerToData(int* pcode,int* pvector_type, int data_enum){{{1*/
FILE* IoModel::SetFilePointerToData(int* pcode,int* pvector_type, int data_enum){

	extern int my_rank;
	extern int num_procs;
	
	
	int found=0;
	int record_enum;
	int record_length;
	int record_code; //1 to 7 number
	int vector_type; //nodal or elementary
 
	/*Go find in the binary file, the position of the data we want to fetch: */
	if(my_rank==0){
	
		/*First set FILE* position to the beginning of the file: */
		fseek(fid,0,SEEK_SET);

		/*Now march through file looking for the correct data identifier: */
		for(;;){
			/*Read enum for this size of first string name: */
			if(fread(&record_enum,sizeof(int),1,fid)==0){
				/*Ok, we have reached the end of the file. break: */
				found=0;
				break;
			}
			
			/*Is this the record sought for? : */
			if (data_enum==record_enum){
				/*Ok, we have found the correct string. Pass the record length, and read data type code: */
				fseek(fid,sizeof(int),SEEK_CUR);
				fread(&record_code,sizeof(int),1,fid);

				/*if record_code points to a vector, get its type (nodal or elementary): */
				if(5<=record_code && record_code<=7)fread(&vector_type,sizeof(int),1,fid);
				found=1;
				break;
			}
			else{
				/*This is not the correct string, read the record length, and use it to skip this record: */
				fread(&record_length,sizeof(int),1,fid);
				/*skip: */
				fseek(fid,record_length,SEEK_CUR);
			}
		}
	}
	MPI_Bcast(&found,1,MPI_INT,0,MPI_COMM_WORLD); 
	if(!found)_error_("%s %s ","could not find data with name",EnumToStringx(data_enum));

	/*Broadcast code and vector type: */
	MPI_Bcast(&record_code,1,MPI_INT,0,MPI_COMM_WORLD); 
	MPI_Bcast(&vector_type,1,MPI_INT,0,MPI_COMM_WORLD); 
	if(record_code==5) MPI_Bcast(&vector_type,1,MPI_INT,0,MPI_COMM_WORLD); 

	/*Assign output pointers:*/
	*pcode=record_code;
	if(pvector_type)*pvector_type=vector_type;

	return fid;
}
/*}}}*/
/*FUNCTION IoModel::Echo{{{1*/
void IoModel::Echo(int which_part,int rank) {

	//which_part  determines what gets echoed, otherwise, we'll get too much output.
	//1-> penalties

	int i,j;

	if(which_part==1 && my_rank==rank && this->dim==3){
		printf("IoModel penalties: \n");
		printf("   number of penalties: %i\n",this->numpenalties);
		printf("   nodes: \n");

		for(i=0;i<this->numpenalties;i++){
			for(j=0;j<this->numlayers;j++){
				printf("%i ",(int)*(this->penalties+this->numlayers*i+j));
			}
			printf("\n");
		}
	}
}
/*}}}*/
