/*!\file PetscVecParam.c
 * \brief: implementation of the PetscVecParam object
 */

/*header files: */
/*{{{1*/
#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <stdio.h>
#include <string.h>
#include "../objects.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../shared/shared.h"
#include "../../Container/Container.h"
#include "../../include/include.h"
/*}}}*/

/*PetscVecParam constructors and destructor*/
/*FUNCTION PetscVecParam::PetscVecParam(){{{1*/
PetscVecParam::PetscVecParam(){
	return;
}
/*}}}*/
/*FUNCTION PetscVecParam::PetscVecParam(int enum_type,IssmPetscVec value){{{1*/
PetscVecParam::PetscVecParam(int in_enum_type,Vec in_value){

	enum_type=in_enum_type;

	value=NULL;

	if(in_value){
		VecDuplicate(in_value,&value);
		VecCopy(in_value,value);
	}
}
/*}}}*/
/*FUNCTION PetscVecParam::~PetscVecParam(){{{1*/
PetscVecParam::~PetscVecParam(){
	VecFree(&value);
}
/*}}}*/

/*Object virtual functions definitions:*/
/*FUNCTION PetscVecParam::Echo {{{1*/
void PetscVecParam::Echo(void){

	printf("PetscVecParam:\n");
	printf("   enum: %i (%s)\n",this->enum_type,EnumToStringx(this->enum_type));

}
/*}}}*/
/*FUNCTION PetscVecParam::DeepEcho{{{1*/
void PetscVecParam::DeepEcho(void){

	int i;
	printf("PetscVecParam:\n");
	printf("   enum: %i (%s)\n",this->enum_type,EnumToStringx(this->enum_type));
	VecView(value,PETSC_VIEWER_STDOUT_WORLD);
}
/*}}}*/
/*FUNCTION PetscVecParam::Id{{{1*/
int    PetscVecParam::Id(void){ return -1; }
/*}}}*/
/*FUNCTION PetscVecParam::MyRank{{{1*/
int    PetscVecParam::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}*/
/*FUNCTION PetscVecParam::Marshall{{{1*/
void  PetscVecParam::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_value=0;
	int   M;
	double* serial_value=NULL;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum value of PetscVecParam: */
	enum_value=PetscVecParamEnum;
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_value,sizeof(enum_value));marshalled_dataset+=sizeof(enum_value);
	
	/*marshall PetscVecParam data: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	if(value){
		VecGetSize(value,&M);
		VecToMPISerial(&serial_value,value);
		memcpy(marshalled_dataset,&M,sizeof(M));marshalled_dataset+=sizeof(M);
		memcpy(marshalled_dataset,serial_value,M*sizeof(double));marshalled_dataset+=(M*sizeof(double));
	}
	else{
		M=0;
		memcpy(marshalled_dataset,&M,sizeof(M));marshalled_dataset+=sizeof(M);
	}
	/*Free ressources:*/
	xfree((void**)&serial_value);

	/*return:*/
	*pmarshalled_dataset=marshalled_dataset;
}
/*}}}*/
/*FUNCTION PetscVecParam::MarshallSize{{{1*/
int   PetscVecParam::MarshallSize(){

	int M=0;
	if(value)VecGetSize(value,&M);

	return sizeof(M)+M*sizeof(double)
		+sizeof(enum_type)+
		+sizeof(int); //sizeof(int) for enum value
}
/*}}}*/
/*FUNCTION PetscVecParam::Demarshall{{{1*/
void  PetscVecParam::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   i;
	int   M;
	double* serial_vec=NULL;
	int*    idxm=NULL;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/
	memcpy(&enum_type,marshalled_dataset,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*data: */
	
	memcpy(&M,marshalled_dataset,sizeof(M));marshalled_dataset+=sizeof(M);
	if(M){
		serial_vec=(double*)xmalloc(M*sizeof(double));
		memcpy(serial_vec,marshalled_dataset,M*sizeof(double));marshalled_dataset+=(M*sizeof(double));

		value=NewVec(M);
		idxm=(int*)xmalloc(M*sizeof(int));
		for(i=0;i<M;i++)idxm[i]=i;
		VecSetValues(value,M,idxm,serial_vec,INSERT_VALUES);

		VecAssemblyBegin(value);
		VecAssemblyEnd(value);

		
	}
	else{
		value=NULL;
	}

	/*Free ressources:*/
	xfree((void**)&serial_vec);
	xfree((void**)&idxm);

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
}
/*}}}*/
/*FUNCTION PetscVecParam::Enum{{{1*/
int PetscVecParam::Enum(void){

	return PetscVecParamEnum;

}
/*}}}*/
/*FUNCTION PetscVecParam::copy{{{1*/
Object* PetscVecParam::copy() {
	
	return new PetscVecParam(this->enum_type,this->value);

}
/*}}}*/

/*PetscVecParam virtual functions definitions: */
/*FUNCTION PetscVecParam::GetParameterValue{{{1*/
void  PetscVecParam::GetParameterValue(Vec* poutput){
	Vec output=NULL;

	if(value){
		VecDuplicate(value,&output);
		VecCopy(value,output);
	}
	*poutput=output;
}
/*}}}*/
/*FUNCTION PetscVecParam::GetParameterName{{{1*/
char* PetscVecParam::GetParameterName(void){
	return  EnumToStringx(this->enum_type);
}
/*}}}*/
/*FUNCTION PetscVecParam::SetMatlabField{{{1*/
#ifdef _SERIAL_
void  PetscVecParam::SetMatlabField(mxArray* dataref){

	mxArray* pfield=NULL;
	char* name=NULL;
	double* doublevec=NULL;
	int M;
	
	VecToMPISerial(&doublevec,value);
	VecGetSize(value,&M);
	name=this->GetParameterName();
	
	pfield=mxCreateDoubleMatrix(0,0,mxREAL);
	mxSetM(pfield,M);
	mxSetN(pfield,1);
	mxSetPr(pfield,doublevec);
	
	mxSetField( dataref, 0, name, pfield);
}
#endif
/*}}}*/
/*FUNCTION PetscVecParam::SetValue{{{1*/
void  PetscVecParam::SetValue(Vec vector){

	/*avoid leak: */
	VecFree(&value);
	
	/*copy: */
	VecDuplicate(vector,&value);
	VecCopy(vector,value);
}
/*}}}*/
