/*!\file:  NodesPartitioning.cpp
 * \brief: partition elements and nodes and vertices
 */ 

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <string.h>
#include "../../objects/objects.h"
#include "../../shared/shared.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../io/io.h"
#include "../../include/include.h"
#include "../MeshPartitionx/MeshPartitionx.h"
#include "../ModelProcessorx/ModelProcessorx.h"

void  DiscontinuousGalerkinNodesPartitioning(bool** pmy_nodes,bool* my_elements, int* my_vertices, IoModel* iomodel, FILE* iomodel_handle);
void  ContinuousGalerkinNodesPartitioning(bool** pmy_nodes,bool* my_elements, int* my_vertices, IoModel* iomodel, FILE* iomodel_handle);

void  NodesPartitioning(bool** pmy_nodes,bool* my_elements, int* my_vertices, IoModel* iomodel, FILE* iomodel_handle,bool continuous){
	
	/*First thing, this is a new partition for a new analysis_type, therefore, to avoid a leak, erase the nodes partition that might come through pmy_nodes: */
	xfree((void**)pmy_nodes);

	/*Now, depending on whether we are running galerkin discontinous or continuous elements, carry out a different partition of the nodes: */
	if(continuous==true)
		ContinuousGalerkinNodesPartitioning(pmy_nodes,my_elements, my_vertices, iomodel, iomodel_handle);
	else
		DiscontinuousGalerkinNodesPartitioning(pmy_nodes,my_elements, my_vertices, iomodel, iomodel_handle);
}

void  ContinuousGalerkinNodesPartitioning(bool** pmy_nodes,bool* my_elements, int* my_vertices, IoModel* iomodel, FILE* iomodel_handle){

	/*as many nodes as there are vertices */

	/*output: */
	bool* my_nodes=NULL;

	my_nodes=(bool*)xmalloc(iomodel->numberofvertices*sizeof(bool));
	memcpy(my_nodes,my_vertices,iomodel->numberofvertices*sizeof(bool));

	/*Assign output pointers:*/
	*pmy_nodes=my_nodes;
}


void  DiscontinuousGalerkinNodesPartitioning(bool** pmy_nodes,bool* my_elements, int* my_vertices, IoModel* iomodel, FILE* iomodel_handle){

	/*each element has it own nodes (as many as vertices) + additional nodes from neighbouring elements for each edge. This yields to a very different partition for 
	 * the nodes and the vertices. The vertices are similar to continuous galerkin, but the nodes partitioning involves edges, which mess up sorting of 
	 * ids. */
	
	int i,j;

	/*output: */
	bool*   my_nodes=NULL;

	int     i1,i2;
	int     cols;
	double  e1,e2;
	int     pos;

	/*Build discontinuous node partitioning
	 *  - there are three nodes per element (discontinous)
	 *  - for each element present of each partition, its three nodes will be in this partition
	 *  - the edges require the dofs of the 2 nodes of each elements sharing the edge.
	 *    if the 2 elements sharing the edge are on 2 different cpus, we must duplicate
	 *    the two nodes that are not on the cpus so that the edge can access the dofs of
	 *    all its 4 nodes
	 */

	/*Allocate*/
	my_nodes=(bool*)xcalloc(3*iomodel->numberofelements,sizeof(int));

	/*First: add all the nodes of all the elements belonging to this cpu*/
	if (iomodel->dim==2){
		for (i=0;i<iomodel->numberofelements;i++){
			if (my_elements[i]){
				my_nodes[3*i+0]=1;
				my_nodes[3*i+1]=1;
				my_nodes[3*i+2]=1;
			}
		}
	}
	else{
		_error_("not implemented yet");
	}

	/*Second: add all missing nodes*/

	/*Get edges and elements*/
	IoModelFetchData(&iomodel->edges,&iomodel->numberofedges,&cols,iomodel_handle,EdgesEnum);
	IoModelFetchData(&iomodel->elements,NULL,NULL,iomodel_handle,ElementsEnum);
	if (cols!=4) _error_("field edges should have 4 columns");

	/*!All elements have been partitioned above, only create elements for this CPU: */
	for (i=0;i<iomodel->numberofedges;i++){

		/*Get left and right elements*/
		e1=iomodel->edges[4*i+2]-1; //edges are [node1 node2 elem1 elem2]
		e2=iomodel->edges[4*i+3]-1; //edges are [node1 node2 elem1 elem2]

		/* 1) If the element e1 is in the current partition
		 * 2) and if the edge of the element is shared by another element (internal edge)
		 * 3) and if this element is not in the same partition:
		 * we must clone the nodes on this partition so that the loads (Numericalflux)
		 * will have access to their properties (dofs,...)*/
		if(my_elements[(int)e1] && !isnan(e2) && !my_elements[(int)e2]){ 

			/*1: Get vertices ids*/
			i1=(int)iomodel->edges[4*i+0];
			i2=(int)iomodel->edges[4*i+1];

			/*2: Get the column where these ids are located in the index*/
			pos=UNDEF;
			for(j=0;j<3;j++){
				if ((int)iomodel->elements[3*(int)e2+j]==i1) pos=j;
			}

			/*3: We have the id of the elements and the position of the vertices in the index
			 * we can now create the corresponding nodes:*/
			if (pos==0){
				my_nodes[(int)e2*3+0]=1;
				my_nodes[(int)e2*3+2]=1;
			}
			else if(pos==1){
				my_nodes[(int)e2*3+1]=1;
				my_nodes[(int)e2*3+0]=1;
			}
			else if (pos==2){
				my_nodes[(int)e2*3+2]=1;
				my_nodes[(int)e2*3+1]=1;
			}
			else{
				_error_("Problem in edges creation");
			}
		}
	}

	/*Free data: */
	xfree((void**)&iomodel->elements);
	xfree((void**)&iomodel->edges);

	/*Assign output pointers:*/
	*pmy_nodes=my_nodes;
}
