/*! \file IoModelFetchData.c
 *  \brief: wrapper to the I/O routines, for special processing of the IoModel structure.
 */

#ifdef HAVE_CONFIG_H
	#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../../shared/shared.h"
#include "../../include/include.h"
#include "./diskio.h"

/*FUNCTION IoModelFetchData(bool*     pbool,FILE* model_handle,int data_enum){{{1*/
void  IoModelFetchData(bool* pboolean,FILE* model_handle,int data_enum){

	extern int my_rank;
	extern int num_procs;
	FILE* fid=NULL;

	/*output: */
	bool  boolean;
	int   code;
	
	/*Set file pointer to beginning of the data: */
	fid=SetFilePointerToData(&code,NULL,model_handle,data_enum);
	
	if(code!=1)_error_("%s%s","IoModelFetchData expecting a boolean for enum ",EnumToStringx(data_enum));
	
	/*We have to read a boolean from disk. */
	if(my_rank==0){  
		if(fread(&boolean,sizeof(bool),1,fid)!=1) _error_(" could not read boolean ");
	}

	MPI_Bcast(&boolean,1,MPI_BYTE,0,MPI_COMM_WORLD); 

	/*Assign output pointers: */
	*pboolean=boolean;

}
/*}}}*/
/*FUNCTION IoModelFetchData(int*      pinteger,FILE* model_handle,int data_enum){{{1*/
void  IoModelFetchData(int* pinteger,FILE* model_handle,int data_enum){

	extern int my_rank;
	extern int num_procs;
	FILE* fid=NULL;

	/*output: */
	int   integer;
	int   code;
	
	/*Set file pointer to beginning of the data: */
	fid=SetFilePointerToData(&code,NULL,model_handle,data_enum);
	
	if(code!=2)_error_("%s%s","IoModelFetchData expecting an integer for enum ",EnumToStringx(data_enum));
	
	/*We have to read a integer from disk. First read the dimensions of the integer, then the integer: */
	if(my_rank==0){  
		if(fread(&integer,sizeof(int),1,fid)!=1) _error_(" could not read integer ");
	}

	MPI_Bcast(&integer,1,MPI_INT,0,MPI_COMM_WORLD); 

	/*Assign output pointers: */
	*pinteger=integer;

}
/*}}}*/
/*FUNCTION IoModelFetchData(double*   pscalar,FILE* model_handle,int data_enum){{{1*/
void  IoModelFetchData(double* pscalar,FILE* model_handle,int data_enum){


	extern int my_rank;
	extern int num_procs;
	FILE* fid=NULL;

	/*output: */
	double   scalar;
	int      code;
	
	/*Set file pointer to beginning of the data: */
	fid=SetFilePointerToData(&code,NULL,model_handle,data_enum);
	
	if(code!=3)_error_("%s%s","IoModelFetchData expecting a double for enum ",EnumToStringx(data_enum));
	
	/*We have to read a scalar from disk. First read the dimensions of the scalar, then the scalar: */
	if(my_rank==0){
		if(fread(&scalar,sizeof(double),1,fid)!=1)_error_(" could not read scalar ");
	}
	MPI_Bcast(&scalar,1,MPI_DOUBLE,0,MPI_COMM_WORLD); 

	/*Assign output pointers: */
	*pscalar=scalar;
		 
}
/*}}}*/
/*FUNCTION IoModelFetchData(char**    pstring,FILE* model_handle,int data_enum){{{1*/
void  IoModelFetchData(char** pstring,FILE* model_handle,int data_enum){

	extern int my_rank;
	extern int num_procs;
	FILE* fid=NULL;

	/*output: */
	char* string=NULL;
	int   string_size;
	int code=0;
	
	/*Set file pointer to beginning of the data: */
	fid=SetFilePointerToData(&code,NULL,model_handle,data_enum);
	
	if(code!=4)_error_("%s%s","IoModelFetchData expecting a string for enum ",EnumToStringx(data_enum));
	
	/*Now fetch: */
	
	/*We have to read a string from disk. First read the dimensions of the string, then the string: */
	if(my_rank==0){  
		if(fread(&string_size,sizeof(int),1,fid)!=1) _error_(" could not read length of string ");
	}

	MPI_Bcast(&string_size,1,MPI_INT,0,MPI_COMM_WORLD); 

	/*Now allocate string: */
	if(string_size){
		string=(char*)xmalloc((string_size+1)*sizeof(char));
		string[string_size]='\0';

		/*Read string on node 0, then broadcast: */
		if(my_rank==0){  
			if(fread(string,string_size*sizeof(char),1,fid)!=1)_error_("  could not read string ");
		}
		MPI_Bcast(string,string_size,MPI_CHAR,0,MPI_COMM_WORLD); 
	}
	else{
		string=(char*)xmalloc(sizeof(char));
		string[0]='\0';
	}


	/*Assign output pointers: */
	*pstring=string;
}
/*}}}*/
/*FUNCTION IoModelFetchData(double**  pdoublematrix,int* pM,int* pN,FILE* model_handle,int data_enum){{{1*/
void  IoModelFetchData(double** pmatrix,int* pM,int* pN,FILE* model_handle,int data_enum){

	extern int my_rank;
	extern int num_procs;

	/*output: */
	int M,N;
	double* matrix=NULL;
	int code=0;
	int vector_type=0;
	
	FILE* fid=NULL;
	
	/*Set file pointer to beginning of the data: */
	fid=SetFilePointerToData(&code,&vector_type,model_handle,data_enum);

	if((code!=5) && (code!=6) && (code!=7))_error_("%s%s","IoModelFetchData expecting a double, integer or boolean matrix for enum ",EnumToStringx(data_enum));
	
	/*Now fetch: */

	/*We have to read a matrix from disk. First read the dimensions of the matrix, then the whole matrix: */
	/*numberofelements: */
	if(my_rank==0){  
		if(fread(&M,sizeof(int),1,fid)!=1) _error_("could not read number of rows for matrix ");
	}

	MPI_Bcast(&M,1,MPI_INT,0,MPI_COMM_WORLD); 

	if(my_rank==0){  
		if(fread(&N,sizeof(int),1,fid)!=1) _error_("could not read number of columns for matrix ");
	}
	MPI_Bcast(&N,1,MPI_INT,0,MPI_COMM_WORLD); 

	/*Now allocate matrix: */
	if(M*N){
		matrix=(double*)xmalloc(M*N*sizeof(double));

		/*Read matrix on node 0, then broadcast: */
		if(my_rank==0){  
			if(fread(matrix,M*N*sizeof(double),1,fid)!=1) _error_("could not read matrix ");
		}
		
		MPI_Bcast(matrix,M*N,MPI_DOUBLE,0,MPI_COMM_WORLD); 
	}

	/*Assign output pointers: */
	*pmatrix=matrix;
	if (pM)*pM=M;
	if (pN)*pN=N;

}
/*}}}*/
/*FUNCTION IoModelFetchData(char***   pstrings,int* pnumstrings,FILE* model_handle,int data_enum){{{1*/
void  IoModelFetchData(char*** pstrings,int* pnumstrings,FILE* model_handle,int data_enum){

	extern int my_rank;
	extern int num_procs;
	FILE* fid=NULL;
	int i;

	/*output: */
	int   numstrings=0;
	char** strings=NULL;
	
	/*intermediary: */
	char* string=NULL;
	int   string_size;
	int   code;
	
	/*Set file pointer to beginning of the data: */
	fid=SetFilePointerToData(&code,NULL,model_handle,data_enum);
	
	if(code!=9)_error_("%s%s","IoModelFetchData expecting a string array for enum ",EnumToStringx(data_enum));
	
	/*We have to read a bunch of strings from disk. First read the number of strings, and allocate: */
	if(my_rank==0){  
		if(fread(&numstrings,sizeof(int),1,fid)!=1) _error_(" could not read length of string array");
	}
	MPI_Bcast(&numstrings,1,MPI_INT,0,MPI_COMM_WORLD); 

	/*Now allocate string array: */
	if(numstrings){
		strings=(char**)xmalloc(numstrings*sizeof(char*));
		for(i=0;i<numstrings;i++)strings[i]=NULL;

		/*Go through strings, and read: */
		for(i=0;i<numstrings;i++){
			
			if(my_rank==0){  
				if(fread(&string_size,sizeof(int),1,fid)!=1) _error_(" could not read length of string ");
			}
			MPI_Bcast(&string_size,1,MPI_INT,0,MPI_COMM_WORLD); 
			if(string_size){
				string=(char*)xmalloc((string_size+1)*sizeof(char));
				string[string_size]='\0';

				/*Read string on node 0, then broadcast: */
				if(my_rank==0){  
					if(fread(string,string_size*sizeof(char),1,fid)!=1)_error_("  could not read string ");
				}
				MPI_Bcast(string,string_size,MPI_CHAR,0,MPI_COMM_WORLD); 
			}
			else{
				string=(char*)xmalloc(sizeof(char));
				string[0]='\0';
			}

			strings[i]=string;
		}
	}

	/*Assign output pointers: */
	*pstrings=strings;
	*pnumstrings=numstrings;
}
/*}}}*/
/*FUNCTION IoModelFetchData(double*** pmatrices,int** pmdims,int** pndims, int* pM,FILE* model_handle,int data_enum){{{1*/
void  IoModelFetchData(double*** pmatrices,int** pmdims,int** pndims, int* pnumrecords,FILE* model_handle,int data_enum){

	int i;

	extern int my_rank;
	extern int num_procs;

	/*output: */
	double** matrices=NULL;
	int*     mdims=NULL;
	int*     ndims=NULL;
	int      numrecords=0;

	/*intermediary: */
	int M,N;
	double* matrix=NULL;
	
	FILE* fid=NULL;
	int   code;
	
	/*Set file pointer to beginning of the data: */
	fid=SetFilePointerToData(&code,NULL,model_handle,data_enum);
	
	if(code!=8)_error_("%s%s","IoModelFetchData expecting a double mat  array for enum ",EnumToStringx(data_enum));
	
	/*Now fetch: */
	if(my_rank==0){  
		if(fread(&numrecords,sizeof(int),1,fid)!=1) _error_("could not read number of records in matrix array ");
	}
	MPI_Bcast(&numrecords,1,MPI_INT,0,MPI_COMM_WORLD); 

	if(numrecords){

		/*Allocate matrices :*/
		matrices=(double**)xmalloc(numrecords*sizeof(double*));
		mdims=(int*)xmalloc(numrecords*sizeof(int));
		ndims=(int*)xmalloc(numrecords*sizeof(int));

		for(i=0;i<numrecords;i++){
			matrices[i]=NULL;
			mdims[i]=0;
			ndims[i]=0;
		}

		/*Loop through records and fetch matrix: */
		for(i=0;i<numrecords;i++){

			if(my_rank==0){  
				if(fread(&M,sizeof(int),1,fid)!=1) _error_("%s%i%s","could not read number of rows in ",i,"th matrix of matrix array");
			}
			MPI_Bcast(&M,1,MPI_INT,0,MPI_COMM_WORLD); 

			if(my_rank==0){  
				if(fread(&N,sizeof(int),1,fid)!=1) _error_("%s%i%s","could not read number of columns in ",i,"th matrix of matrix array");
			}
			MPI_Bcast(&N,1,MPI_INT,0,MPI_COMM_WORLD); 

			/*Now allocate matrix: */
			if(M*N){
				matrix=(double*)xmalloc(M*N*sizeof(double));

				/*Read matrix on node 0, then broadcast: */
				if(my_rank==0){  
					if(fread(matrix,M*N*sizeof(double),1,fid)!=1) _error_("could not read matrix ");
				}

				MPI_Bcast(matrix,M*N,MPI_DOUBLE,0,MPI_COMM_WORLD); 
			}

			/*Assign: */
			matrices[i]=matrix;
			mdims[i]=M;
			ndims[i]=N;
		}
	}

	/*Assign output pointers: */
	*pmatrices=matrices;
	*pmdims=mdims;
	*pndims=ndims;
	*pnumrecords=numrecords;
}
/*}}}*/

/*FUNCTION SetFilePointerToData(int* pcode,int* pvector_type, FILE* model_handle,int data_enum){{{1*/
FILE* SetFilePointerToData(int* pcode,int* pvector_type, FILE* model_handle,int data_enum){

	extern int my_rank;
	extern int num_procs;
	
	FILE* fid=NULL;
	int found=0;
	int record_enum;
	int record_length;
	int record_code; //1 to 7 number
	int vector_type; //nodal or elementary
 
	/*Go find in the binary file, the position of the data we want to fetch: */
	if(my_rank==0){
	
		/*First set FILE* position to the beginning of the file: */
		fid=(FILE*)model_handle;
		fseek(fid,0,SEEK_SET);

		/*Now march through file looking for the correct data identifier: */
		for(;;){
			/*Read enum for this size of first string name: */
			if(fread(&record_enum,sizeof(int),1,fid)==0){
				/*Ok, we have reached the end of the file. break: */
				found=0;
				break;
			}
			
			/*Is this the record sought for? : */
			if (data_enum==record_enum){
				/*Ok, we have found the correct string. Pass the record length, and read data type code: */
				fseek(fid,sizeof(int),SEEK_CUR);
				fread(&record_code,sizeof(int),1,fid);

				/*if record_code points to a vector, get its type (nodal or elementary): */
				if(5<=record_code && record_code<=7)fread(&vector_type,sizeof(int),1,fid);
				found=1;
				break;
			}
			else{
				/*This is not the correct string, read the record length, and use it to skip this record: */
				fread(&record_length,sizeof(int),1,fid);
				/*skip: */
				fseek(fid,record_length,SEEK_CUR);
			}
		}
	}
	MPI_Bcast(&found,1,MPI_INT,0,MPI_COMM_WORLD); 
	if(!found)_error_("%s %s ","could not find data with name",EnumToStringx(data_enum));

	/*Broadcast code and vector type: */
	MPI_Bcast(&record_code,1,MPI_INT,0,MPI_COMM_WORLD); 
	MPI_Bcast(&vector_type,1,MPI_INT,0,MPI_COMM_WORLD); 
	if(record_code==5) MPI_Bcast(&vector_type,1,MPI_INT,0,MPI_COMM_WORLD); 

	/*Assign output pointers:*/
	*pcode=record_code;
	if(pvector_type)*pvector_type=vector_type;

	return fid;
}
/*}}}*/
