/*!\file:  BrentSearch.cpp
 * \brief optimization algorithm
 */ 

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "./numerics.h"
#include "../../objects/objects.h"
#include "../../include/include.h"
#include "../../shared/shared.h"
#include <float.h>

void BrentSearch(double* psearch_scalar,double* pJ,OptPars* optpars,double (*f)(double,OptArgs*), OptArgs* optargs){

	/* This routine is optimizing a given function using Brent's method
	 * (Golden or parabolic procedure)*/

	/*Intermediary*/
	double si,gold,intervalgold,oldintervalgold;
	double parab_num,parab_den;
	double distance,cm_jump;
	double fxmax,fxmin,fxbest;
	double fx,fx1,fx2;
	double xmax,xmin,xbest;
	double x,x1,x2,xm;
	double tol1,tol2,seps,tolerance;
	int    maxiter,iter;
	bool   loop=true,goldenflag;

	/*Recover parameters:*/
	xmin=optpars->xmin;
	xmax=optpars->xmax;
	tolerance=optpars->tolerance;
	maxiter=optpars->maxiter;
	cm_jump=optpars->cm_jump;
	
	/*initialize counter and get response at the boundaries*/
	iter=0;
	fxmin = (*f)(xmin,optargs);
	if (isnan(fxmin)) _error_("Function evaluation returned NaN");
	_printf_(VerboseControl(),"\n        Iteration         x           f(x)       Tolerance         Procedure\n\n");
	_printf_(VerboseControl(),"        %s    %12.6g  %12.6g  %s","   N/A",xmin,fxmin,"         N/A         boundary\n");
	fxmax = (*f)(xmax,optargs);
	if (isnan(fxmax)) _error_("Function evaluation returned NaN");
	_printf_(VerboseControl(),"        %s    %12.6g  %12.6g  %s","   N/A",xmax,fxmax,"         N/A         boundary\n");

	/*test if jump option activated and xmin==0*/
	if (!isnan(cm_jump) && (xmin==0) && (fxmax/fxmin)<cm_jump){
		*psearch_scalar=xmax;
		*pJ=fxmax;
		return;
	}

	/*initialize optimization variables*/
	seps=sqrt(DBL_EPSILON);    //precision of a double
	distance=0.0;              //new_x=old_x + distance
	gold=0.5*(3.0-sqrt(5.0));  //gold = 1 - golden ratio
	intervalgold=0.0;          //distance used by Golden procedure

	/*1: initialize the values of the 4 x needed (x1,x2,x,xbest)*/
	x1=xmin+gold*(xmax-xmin);
	x2=x1;
	xbest=x1;
	x=xbest;

	/*2: call the function to be evaluated*/
	fxbest = (*f)(x,optargs);
	if(isnan(fxbest)) _error_("Function evaluation returned NaN");
	iter=iter+1;

	/*3: update the other variables*/
	fx1=fxbest;
	fx2=fxbest;
	/*xm is always in the middle of a and b*/
	xm=0.5*(xmin+xmax);                           
	/*update tolerances*/
	tol1=seps*sqrt(pow(xbest,2))+tolerance/3.0;
	tol2=2.0*tol1;

	/*4: print result*/
	_printf_(VerboseControl(),"         %5i    %12.6g  %12.6g  %12.6g  %s\n",iter,xbest,fxbest,pow(pow(xbest-xm,2),0.5),"       initial");
	if (!isnan(cm_jump) && (xmin==0) && ((fxbest/fxmin)<cm_jump)){
		_printf_(VerboseControl(),"      optimization terminated: current x satisfies criteria 'cm_jump'=%g\n",cm_jump);
		loop=false;
	}

	while(loop){

		goldenflag=true;

		// Is a parabolic fit possible ?
		if (sqrt(pow(intervalgold,2))>tol1){

			// Yes, so fit parabola
			goldenflag=false;
			parab_num=(xbest-x1)*(xbest-x1)*(fxbest-fx2)-(xbest-x2)*(xbest-x2)*(fxbest-fx1);;
			parab_den=2.0*(xbest-x1)*(fxbest-fx2)-2.0*(xbest-x2)*(fxbest-fx1);

			//reverse p if necessary
			if(parab_den>0.0){ 
				parab_num=-parab_num;
			}
			parab_den=sqrt(pow(parab_den,2));
			oldintervalgold=intervalgold;
			intervalgold=distance;

			// Is the parabola acceptable (we use seps here because in some configuration parab_num==parab_den*(xmax-xbest)
			// and the result is not repeatable anymore
			if (( sqrt(pow(parab_num,2)) < sqrt(pow(0.5*parab_den*oldintervalgold,2))) &&
						(parab_num>parab_den*(xmin-xbest)+seps) && 
						(parab_num<parab_den*(xmax-xbest)-seps)){

				// Yes, parabolic interpolation step
				distance=parab_num/parab_den;
				x=xbest+distance;

				// f must not be evaluated too close to min_x or max_x
				if (((x-xmin)<tol2) || ((xmax-x)<tol2)){
					if ((xm-xbest)<0.0) si=-1;
					else                si=1;
					//compute new distance
					distance=tol1*si;
				}
			}
			else{
				// Not acceptable, must do a golden section step
				goldenflag=true;
			}
		}

		//Golden procedure
		if(goldenflag){
			// compute the new distance d
			if(xbest>=xm){
				intervalgold=xmin-xbest;    
			}
			else{ 
				intervalgold=xmax-xbest;  
			}
			distance=gold*intervalgold;
		}

		// The function must not be evaluated too close to xbest
		if(distance<0) si=-1;
		else           si=1;
		if(sqrt(pow(distance,2))>tol1) x=xbest+si*sqrt(pow(distance,2));
		else                           x=xbest+si*tol1;

		//evaluate function on x
		fx = (*f)(x,optargs);
		if(isnan(fx)) _error_("Function evaluation returned NaN");
		iter=iter+1;

		// Update a, b, xm, x1, x2, tol1, tol2
		if (fx<=fxbest){
			if (x>=xbest) xmin=xbest;
			else          xmax=xbest;
			x1=x2;    fx1=fx2;
			x2=xbest; fx2=fxbest;
			xbest=x;  fxbest=fx;
		}
		else{ // fx > fxbest
			if (x<xbest) xmin=x;
			else         xmax=x;
			if ((fx<=fx2) || (x2==xbest)){
				x1=x2; fx1=fx2;
				x2=x;  fx2=fx;
			}
			else if ( (fx <= fx1) || (x1 == xbest) || (x1 == x2) ){
				x1=x;  fx1=fx;
			}
		}
		xm = 0.5*(xmin+xmax);
		tol1=seps*pow(pow(xbest,2),0.5)+tolerance/3.0;
		tol2=2.0*tol1;
		_printf_(VerboseControl(),"         %5i    %12.6g  %12.6g  %12.6g  %s\n",iter,x,fx,pow(pow(xbest-xm,2),0.5),goldenflag?"       golden":"       parabolic");

		/*Stop the optimization?*/
		if (sqrt(pow(xbest-xm,2)) < (tol2-0.5*(xmax-xmin))){
			_printf_(VerboseControl(),"      optimization terminated: current x satisfies criteria 'tolx'=%g\n",tolerance);
			loop=false;
		}
		else if (iter>=maxiter){
			_printf_(VerboseControl(),"      exiting: Maximum number of iterations has been exceeded  ('maxiter'=%i)\n",maxiter);
			loop=false;
		}
		else if (!isnan(cm_jump) && (xmin==0) && ((fxbest/fxmin)<cm_jump)){
			_printf_(VerboseControl(),"      optimization terminated: current x satisfies criteria 'cm_jump'=%g\n",cm_jump);
			loop=false;
		}
		else{
			//continue
			loop=true;
		}
	}//end while

	//Now, check that the value on the boundaries are not better than current fxbest
	if (fxbest>fxmin){
		xbest=optpars->xmin; fxbest=fxmin;
	}
	if (fxbest>fxmax){
		xbest=optpars->xmax; fxbest=fxmax;
	}

	/*Assign output pointers: */
	*psearch_scalar=xbest;
	*pJ=fxbest;
}
