%MODEL class definition
%
%   Usage:
%      md = model(varargin)

classdef model
    properties (SetAccess=public) %Model fields
		 % {{{1
		 %Careful here: no other class should be used as default value this is a bug of matlab
		 %in what follows, any field is defined according to the following convention: 
		 %field={defaultvalue,marshall?,data type,matrix type}

		 %Model general information
		 notes           = {'',false};
		 name            = {'',true,'String'};
		 runtimename     = {'',false}; %name used when running a parallel job
		 counter         = {0,false};
		 domainoutline   = {NaN,false};
		 riftoutline     = {NaN,false};
		 iceshelfoutline = {NaN,false};
		 icesheetoutline = {NaN,false};
		 wateroutline    = {NaN,false};
		 parameterfile   = {NaN,false};
		 runmefile       = {NaN,false};

		 %Mesh
		 bamg                = {struct(),false};
		 dim                 = {0,true,'Integer'};
		 numberofelements    = {0,true,'Integer'};
		 numberofnodes       = {0,true,'Integer'};
		 elements            = {NaN,true,'DoubleMat',2};
		 elements_type       = {NaN,true,'DoubleMat',2};
		 vertices_type       = {NaN,true,'DoubleMat',1};
		 x                   = {NaN,true,'DoubleMat',1};
		 y                   = {NaN,true,'DoubleMat',1};
		 z                   = {NaN,true,'DoubleMat',1};
		 bed_slopex          = {NaN,false};
		 bed_slopey          = {NaN,false};
		 surface_slopex      = {NaN,false};
		 surface_slopey      = {NaN,false};
		 nodeconnectivity    = {NaN,false};
		 elementconnectivity = {NaN,true,'DoubleMat',3};
		 edges               = {NaN,true,'DoubleMat',3};

		 %I/O
		 io_gather           = {NaN,true,'Integer'};

		 %Initial 2d mesh 
		 numberofelements2d = {0,true,'Integer'};
		 numberofnodes2d    = {0,true,'Integer'};
		 elements2d         = {NaN,true,'DoubleMat',3};
		 elements_type2d    = {NaN,false};
		 vertices_type2d    = {NaN,false};
		 x2d                = {NaN,false};
		 y2d                = {NaN,false};
		 z2d                = {NaN,false};

		 %latlon of the coorinates
		 lat        = {NaN,false};
		 long       = {NaN,false};
		 hemisphere = {NaN,false};

		 %Elements type
		 ishutter             = {0,true,'Integer'};
		 ismacayealpattyn     = {0,true,'Integer'};
		 isstokes             = {0,true,'Integer'};

		 %Elements
		 elementonhutter      = {NaN,false};
		 elementonmacayeal    = {NaN,false};
		 elementonpattyn      = {NaN,false};
		 elementonstokes      = {NaN,false};

		 %Nodes
		 nodeonhutter         = {NaN,true,'DoubleMat',1};
		 nodeonmacayeal       = {NaN,true,'DoubleMat',1};
		 nodeonpattyn         = {NaN,true,'DoubleMat',1};
		 nodeonstokes         = {NaN,true,'DoubleMat',1};
		 borderstokes         = {NaN,true,'DoubleMat',3};

		 %Stokes
		 stokesreconditioning = {0,true,'Double'};
		 shelf_dampening      = {0,true,'Integer'};

		 %Penalties
		 penalties            = {NaN,true,'DoubleMat',3};
		 penalty_offset       = {0,true,'Double'};
		 penalty_melting      = {0,true,'Double'};
		 penalty_lock         = {0,true,'Integer'};
		 segments             = {NaN,false};
		 segmentmarkers       = {NaN,false};
		 rifts                = {NaN,false};
		 riftinfo             = {NaN,true,'DoubleMat',3};
		 riftproperties       = {NaN,false};
		 numrifts             = {0,false};

		 %Projections
		 uppernodes           = {NaN,false};
		 upperelements        = {NaN,true,'DoubleMat',2};
		 lowerelements        = {NaN,true,'DoubleMat',2};
		 lowernodes           = {NaN,false};

		 %Extrusion
		 numlayers         = {0,true,'Integer'};
		 extrusionexponent = {0,false};
		 elementonbed      = {NaN,true,'BooleanMat',2};
		 elementonsurface  = {NaN,true,'BooleanMat',2};
		 nodeonbed         = {NaN,true,'BooleanMat',1};
		 nodeonsurface     = {NaN,true,'BooleanMat',1};
		 minh              = {0,false};
		 firn_layer        = {NaN,false};

		 %Extraction
		 extractednodes    = {NaN,false};
		 extractedelements = {NaN,false};

		 %Materials parameters
		 rho_ice                    = {0,true,'Double'};
		 rho_water                  = {0,true,'Double'};
		 rho_firn                   = {0,false};
		 heatcapacity               = {0,true,'Double'};
		 latentheat                 = {0,true,'Double'};
		 thermalconductivity        = {0,true,'Double'};
		 meltingpoint               = {0,true,'Double'};
		 referencetemperature       = {0,true,'Double'}; %for enthalpy
		 beta                       = {0,true,'Double'};
		 mixed_layer_capacity       = {0,true,'Double'};
		 thermal_exchange_velocity  = {0,true,'Double'};
		 min_thermal_constraints    = {0,true,'Integer'};
		 min_mechanical_constraints = {0,true,'Integer'};
		 stabilize_constraints      = {0,true,'Integer'};

		 %Physical parameters
		 g                = {0,true,'Double'};
		 yts              = {0,true,'Double'};
		 drag_type        = {0,true,'Integer'};
		 drag_coefficient = {NaN,true,'DoubleMat',1};
		 drag_p           = {NaN,true,'DoubleMat',2};
		 drag_q           = {NaN,true,'DoubleMat',2};
		 rheology_B       = {NaN,true,'DoubleMat',1};
		 rheology_n       = {NaN,true,'DoubleMat',2};
		 rheology_law     = {0,true,'Integer'};

		 %Geometrical parameters
		 elementoniceshelf = {NaN,true,'BooleanMat',2};
		 elementonicesheet = {NaN,true,'BooleanMat',2};
		 elementonwater    = {NaN,true,'BooleanMat',2};
		 nodeoniceshelf    = {NaN,true,'DoubleMat',1};
		 nodeonicesheet    = {NaN,true,'DoubleMat',1};
		 nodeonwater       = {NaN,true,'DoubleMat',1};
		 surface           = {NaN,true,'DoubleMat',1};
		 thickness         = {NaN,true,'DoubleMat',1};
		 thickness_coeff   = {NaN,true,'DoubleMat',1};
		 bed               = {NaN,true,'DoubleMat',1};
		 bathymetry        = {NaN,true,'DoubleMat',1};
		 mask              = {NaN,false};

		 %Boundary conditions
		 nodeonboundary = {NaN,false};
		 pressureload   = {NaN,true,'DoubleMat',3};
		 spcvx          = {NaN,true,'DoubleMat',1};
		 spcvy          = {NaN,true,'DoubleMat',1};
		 spcvz          = {NaN,true,'DoubleMat',1};
		 spctemperature = {NaN,true,'DoubleMat',1};
		 spcthickness   = {NaN,true,'DoubleMat',1};
		 spcwatercolumn = {NaN,true,'DoubleMat',1};
		 diagnostic_ref = {NaN,true,'DoubleMat',1};

		 %Observations 
		 vx_obs                    = {NaN,true,'DoubleMat',1};
		 vy_obs                    = {NaN,true,'DoubleMat',1};
		 vel_obs                   = {NaN,false};
		 vx_obs_raw                = {NaN,false};
		 vy_obs_raw                = {NaN,false};
		 vx_bal                    = {NaN,false};
		 vy_bal                    = {NaN,false};
		 vel_bal                   = {NaN,false};
		 vel_obs_raw               = {NaN,false};
		 surface_accumulation_rate = {NaN,true,'DoubleMat',1};
		 surface_ablation_rate     = {NaN,true,'DoubleMat',1};
		 surface_mass_balance      = {NaN,true,'DoubleMat',1};
		 dhdt                      = {NaN,true,'DoubleMat',1};
		 geothermalflux            = {NaN,true,'DoubleMat',1};
		 observed_temperature      = {NaN,false};
		 thickness_obs             = {NaN,true,'DoubleMat',1};

		 %Forcings
		 forcings = {struct(),false};

		 %Statics parameters
		 eps_res                  = {0,true,'Double'};
		 eps_rel                  = {0,true,'Double'};
		 eps_abs                  = {0,true,'Double'};
		 max_nonlinear_iterations = {0,true,'Double'};
		 max_steadystate_iterations = {0,true,'Double'};
		 sparsity                 = {0,true,'Double'};
		 connectivity             = {0,true,'Integer'};
		 lowmem                   = {0,true,'Integer'};
		 viscosity_overshoot      = {0,true,'Double'};

		 %Transient 
		 dt                     = {0,true,'Double'};
		 ndt                    = {0,true,'Double'};
		 time_adapt             = {0,true,'Integer'};
		 cfl_coefficient        = {0,true,'Double'};
		 adaptative_cfl         = {0,false};
		 artificial_diffusivity = {0,true,'Integer'};
		 prognostic_DG          = {0,true,'Integer'};
		 deltaH                 = {0,false};
		 DeltaH                 = {0,false};
		 deltaT                 = {0,false};
		 DeltaT                 = {0,false};
		 timeacc                = {0,false};
		 timedec                = {0,false};
		 hydrostatic_adjustment = {0,true,'Integer'};
		 isprognostic = {0,true,'Integer'};
		 isdiagnostic = {0,true,'Integer'};
		 isthermal    = {0,true,'Integer'};
		 %Control
		 control_analysis = {0,true,'Integer'};
		 control_type     = {0,true,'DoubleMat',3};
		 weights          = {[],true,'DoubleMat',1};
		 nsteps           = {0,true,'Integer'};
		 maxiter          = {[],true,'DoubleMat',3};
		 cm_responses     = {[],true,'DoubleMat',3};
		 tolx             = {0,true,'Double'};
		 optscal          = {[],true,'DoubleMat',3};
		 eps_cm           = {0,true,'Double'};
		 cm_min           = {NaN,true,'DoubleMat',3};
		 cm_max           = {NaN,true,'DoubleMat',3};
		 cm_jump          = {[],true,'DoubleMat',3};
		 cm_gradient      = {0,true,'Integer'};
		 epsvel                              = {0,true,'Double'};
		 meanvel                             = {0,true,'Double'};
		 num_control_type                    = {0,true,'Integer'};
		 num_cm_responses                    = {0,true,'Integer'};
		 %Output
		 requested_outputs                    = {[],true,'DoubleMat',3};
		 viscousheating                      = {NaN,false};
		 pressure_elem                       = {NaN,false};
		 stress                              = {NaN,false};
		 stress_surface                      = {NaN,false};
		 stress_bed                          = {NaN,false};
		 deviatoricstress                    = {NaN,false};
		 strainrate                          = {NaN,false};
		 %Debugging
		 verbose                             = {0,false};
		 verbose_binary                      = {0,true,'Integer'};
		 element_debug                       = {0,false};
		 element_debugid                     = {NaN,false};
		 mem_debug                           = {0,false};
		 gprof                               = {0,false};
		 memoryleaks                         = {NaN,false};
		 %Results fields
		 output_frequency                    = {0,true,'Integer'};
		 inputfilename                       = {'',true,'String'};
		 outputfilename                      = {'',true,'String'};
		 results                             = {struct(),false};
		 vx                                  = {NaN,true,'DoubleMat',1};
		 vy                                  = {NaN,true,'DoubleMat',1};
		 vz                                  = {NaN,true,'DoubleMat',1};
		 vel                                 = {NaN,false};
		 temperature                         = {NaN,true,'DoubleMat',1}; %temperature solution vector
		 waterfraction                       = {NaN,true,'DoubleMat',1};
		 gl_melting_rate                     = {NaN,true,'Double'};
		 basal_melting_rate                  = {NaN,true,'DoubleMat',1};
		 basal_melting_rate_correction       = {NaN,true,'DoubleMat',1};
		 basal_melting_rate_correction_apply = {0,true,'Integer'};
		 pressure                            = {NaN,true,'DoubleMat',1};
		 watercolumn                         = {NaN,true,'DoubleMat',1};
		 
		 %Parallelisation
		 cluster       = {NaN,false};
		 outlog        = {'',false};
		 errlog        = {'',false};
		 alloc_cleanup = {1,false};
		 waitonlock    = {0,true,'Double'};

		 %dummy
		 dummy = {NaN,false};

		 %mesh 
		 mesh_domainoutline = {'',false};
		 mesh_riftoutline   = {'',false};
		 mesh_resolution    = {NaN,false};

		 %PETSc and MATLAB solver string
		 petscoptions  = {NaN,false};

		 %Analysis
		 solution_type = {'',false};

		 %management of large models
		 repository    = {'',false};

		 %radar power images
		 sarpwr = {NaN,false};
		 sarxm  = {NaN,false};
		 sarym  = {NaN,false};

		 %qmu
		 variables                       = {struct(),false};
		 responses                       = {struct(),false};
		 qmu_method                      = {struct(),false};
		 qmu_params                      = {struct(),false};
		 dakotaresults                   = {struct(),false};
		 dakotain                        = {'',false};
		 dakotaout                       = {'',false};
		 dakotadat                       = {'',false};
		 qmu_analysis                    = {0,true,'Integer'};
		 part                            = {[],true,'DoubleMat',2};
		 npart                           = {0,true,'Integer'};
		 numberofvariables               = {0,true,'Integer'};
		 numberofresponses               = {0,true,'Integer'};
		 variabledescriptors             = {{},true,'StringArray'};
		 responsedescriptors             = {{},true,'StringArray'};
		 qmu_mass_flux_profile_directory = {NaN,false};
		 qmu_mass_flux_profiles          = {NaN,false};
		 qmu_mass_flux_segments          = {{},true,'MatArray'};
		 qmu_relax                       = {0,false};
		 qmu_save_femmodel               = {0,true,'Integer'};

		 %flaim
		 fm_tracks     = {'',false};
		 fm_flightreqs = {struct(),false};
		 fm_criterion  = {[],false};
		 fm_solution   = {'',false};
		 fm_quality    = {0,false};

		 %grounding line migration: 
		 gl_migration = {0,true,'Integer'};

		 %solution parameters
		 loadresults = {0,false};

		 %partitioner:
		 adjacency = {[],false};
		 vwgt      = {[],false};
		 %}}}
	 end
	 methods (Static)
		 function md = loadobj(md) % {{{
			 % This function is directly called by matlab when a model object is
			 % loaded. If the input is a struct it is an old version of model and
			 % old fields must be recovered (make sure they are in the deprecated
			 % model properties)

			 if verLessThan('matlab','7.9'),
				 disp('Warning: your matlab version is old and there is a risk that load does not work correctly');
				 disp('         if the model is not loaded correctly, rename temporarily loadobj so that matlab does not use it');

				 % This is a Matlab bug: all the fields of md have their default value
				 % md.name = {''    [1]    'String'}
				 % instead of recovering the saved fields
				 % 
				 % Example of error message:
				 % Warning: Error loading an object of class 'model':
				 % Undefined function or method 'exist' for input arguments of type 'cell'
				 %
				 % This has been fixed in MATLAB 7.9 (R2009b) and later versions
			 end

			 if isstruct(md)
				 disp('Recovering model object from a previous version');
				 md = structtomodel(model,md);
			 end
		 end% }}}
		 function temp = template() % {{{

			 %Get default fields
			 md=model(0);
			 modelprops=properties('model');

			 %build output
			 temp=struct();
			 for i=1:length(modelprops),
				 temp.(modelprops{i})=md.(modelprops{i});
			 end

		 end% }}}
	 end
	 methods
		 function md = model(varargin) % {{{1

			 switch nargin
				 case 0

					 %set default parameters
					 md=setdefaultparameters(md);

				 case 1
					 %If single argument of class model, we have a copy constructor. 
					 if (isa(varargin{1},'model'))
						 md = varargin{1};
					 elseif (isa(varargin{1},'char'))
						 %user gave us the name of the repository, this model is going to be large! 
						 md=model;
						 md.repository=varargin{1};
						 md=activaterepository(md);
					 elseif (isa(varargin{1},'numeric'))
						 if varargin{1}==0,
							 %just requesting model, without default parameters, do nothing.
						 else
							 error('model constructor error message: unknown constructor call');
						 end
					 else
						 error('model constructor error message: copy constructor called on a non ''model'' class object');
					 end 

				 otherwise
					 error('model constructor error message: 0 of 1 argument only in input.');
				 end
		 end
		 %}}}
		 function md = activaterepository(md) % {{{1
			 %ACTIVATEREPOSITORY - save the model fields separately
			 %
			 %   This function is required when a matlab session 
			 %   is running low on memory, or when a model is very large. 
			 %   By activating a repository (for example, 'repository.model'), 
			 %   the model fields will each separately be saved and accessed when 
			 %   needed. This will limit the memory use at the maximum; 
			 %
			 %   Usage:
			 %      md=activaterepository(md)


			 %for each field of the model, we create an equivalent variable, with the same name, which we 
			 %save in the repository. 

			 modelfields=fields(md);

			 %then save each of the big fields in their respective files
			 for i=1:length(modelfields),
				 field=char(modelfields(i));
				 eval([field '=md.' field ';']);

				 %anything NaN, will be saved 

				 if isnumeric(eval(field)),
					 if isnan(eval(field)),
						 if exist(['.' md.repository '.' field],'file');
							 error(['Repository ' md.repository ' already exists! Delete all files starting with .' md.repository ' in your directory, before activating a new repository']);
						 end
						 eval(['save .' md.repository '.' field ' ' field]);
					 end
				 end
			 end
		 end % }}}
		 function disp(md) % {{{1
			 disp(sprintf('\n%s = \n',inputname(1)));

			 %name
			 disp(sprintf('   Name:'));
			 if isempty(md.name),
				 disp(sprintf('      N/A'));
			 else
				 disp(sprintf('      %s',md.name));
			 end

			 %notes: md.notes is a cell array of strings
			 disp(sprintf('   Notes:'));
			 if isempty(md.notes),
				 disp(sprintf('      N/A'));
			 else
				 if iscell(md.notes),
					 strings=md.notes;
					 for i=1:length(strings),
						 string=char(strings(i));
						 if i==1,
							 disp(sprintf('      %s',string));
						 else
							 disp(sprintf('      %s',string));
						 end
					 end
				 elseif ischar(md.notes),
					 disp(sprintf('      %s',md.notes));
				 else
					 disp('display error message: ''notes'' for model is neither a cell array of string, nor a string. Cannot display.');
				 end
			 end

			 %repository
			 if ~isempty(md.repository),
				 disp(sprintf('   Repository: %s',md.repository));
			 end

			 %diverse
			 disp(sprintf('   Mesh: %s',md.repository));
			 disp(sprintf('%s%s%s','      Mesh: type ''',inputname(1),'.mesh'' to display'));
			 disp(sprintf('   Parameters: %s',md.repository));
			 disp(sprintf('%s%s%s','      Boundary conditions: type ''',inputname(1),'.bc'' to display'));
			 disp(sprintf('%s%s%s','      Observations: type ''',inputname(1),'.obs'' to display'));
			 disp(sprintf('%s%s%s','      Materials: type ''',inputname(1),'.mat'' to display'));
			 disp(sprintf('%s%s%s','      Parameters: type ''',inputname(1),'.par'' to display'));
			 disp(sprintf('%s%s%s','      ExpPar: type ''',inputname(1),'.exppar'' to display'));
			 disp(sprintf('   Solution parameters: %s',md.repository));
			 disp(sprintf('%s%s%s','      Qmu: type ''',inputname(1),'.qmu'' to display'));
			 disp(sprintf('%s%s%s','      FLAIM: type ''',inputname(1),'.flaim'' to display'));
			 disp(sprintf('%s%s%s','      Diagnostic solution parameters: type ''',inputname(1),'.diagnostic'' to display'));
			 disp(sprintf('%s%s%s','      Thermal solution parameters: type ''',inputname(1),'.thermal'' to display'));
			 disp(sprintf('%s%s%s','      Prognostic solution parameters: type ''',inputname(1),'.prognostic'' to display'));
			 disp(sprintf('%s%s%s','      Transient solution parameters: type ''',inputname(1),'.transient'' to display'));
			 disp(sprintf('%s%s%s','      Control solution parameters: type ''',inputname(1),'.control'' to display'));
			 disp(sprintf('   Parallel:'));
			 disp(sprintf('%s%s%s','      Parallel options: type ''',inputname(1),'.parallel'' to display'));
			 if(md.numrifts)disp(sprintf('      rifts: %i',md.numrifts));end
			 disp(sprintf('   Results:'));
			 disp(sprintf('%s%s%s','      Results: type ''',inputname(1),'.res'' to display'));
			 if(md.numrifts)disp(sprintf('      rifts: %i',md.numrifts));end
		 end
		 %}}}
		 function md = structtomodel(md,structmd) % {{{

			 if ~isstruct(structmd) error('input model is not a structure'); end

			 %loaded model is a struct, initialize output and recover all fields
			 md = structtoobj(model,structmd);

			 %Field name change
			 if isfield(structmd,'drag'), md.drag_coefficient=structmd.drag; end
			 if isfield(structmd,'p'), md.drag_p=structmd.p; end
			 if isfield(structmd,'q'), md.drag_q=structmd.p; end
			 if isfield(structmd,'B'), md.rheology_B=structmd.B; end
			 if isfield(structmd,'n'), md.rheology_n=structmd.n; end
			 if isfield(structmd,'melting'), md.basal_melting_rate=structmd.melting; end
			 if isfield(structmd,'melting_rate'), md.basal_melting_rate=structmd.melting_rate; end
			 if isfield(structmd,'accumulation'), md.surface_mass_balance=structmd.accumulation; end
			 if isfield(structmd,'accumulation_rate'), md.surface_mass_balance=structmd.accumulation_rate; end
			 if isfield(structmd,'numberofgrids'), md.numberofnodes=structmd.numberofgrids; end
			 if isfield(structmd,'numberofgrids2d'), md.numberofnodes2d=structmd.numberofgrids2d; end
			 if isfield(structmd,'gridonhutter'), md.nodeonhutter=structmd.gridonhutter; end
			 if isfield(structmd,'gridonmacayeal'), md.nodeonmacayeal=structmd.gridonmacayeal; end
			 if isfield(structmd,'gridonpattyn'), md.nodeonpattyn=structmd.gridonpattyn; end
			 if isfield(structmd,'gridonstokes'), md.nodeonstokes=structmd.gridonstokes; end
			 if isfield(structmd,'uppergrids'), md.uppernodes=structmd.uppergrids; end
			 if isfield(structmd,'lowergrids'), md.lowernodes=structmd.lowergrids; end
			 if isfield(structmd,'gridonbed'), md.nodeonbed=structmd.gridonbed; end
			 if isfield(structmd,'gridonsurface'), md.nodeonsurface=structmd.gridonsurface; end
			 if isfield(structmd,'extractedgrids'), md.extractednodes=structmd.extractedgrids; end
			 if isfield(structmd,'gridoniceshelf'), md.nodeoniceshelf=structmd.gridoniceshelf; end
			 if isfield(structmd,'gridonicesheet'), md.nodeonicesheet=structmd.gridonicesheet; end
			 if isfield(structmd,'gridonwater'), md.nodeonwater=structmd.gridonwater; end
			 if isfield(structmd,'gridonnuna'), md.nodeonnuna=structmd.gridonnuna; end
			 if isfield(structmd,'gridonboundary'), md.nodeonboundary=structmd.gridonboundary; end

			 %Field class change
			 if (size(structmd.pressureload,2)==3 | size(structmd.pressureload,2)==5),
				 md.pressureload=[structmd.pressureload...
					 WaterEnum*structmd.elementoniceshelf(structmd.pressureload(:,end))+AirEnum*structmd.elementonicesheet(structmd.pressureload(:,end))];
			 end
			 if (isfield(structmd,'type') & ischar(structmd.type)), 
				 if strcmpi(structmd.type,'2d'), md.dim=2; end
				 if strcmpi(structmd.type,'3d'), md.dim=3; end
			 end
			 if isnumeric(md.verbose),
				 md.verbose=verbose;
			 end
			 if isfield(structmd,'spcvelocity'), 
				 md.spcvx=NaN*ones(md.numberofnodes,1);
				 md.spcvy=NaN*ones(md.numberofnodes,1);
				 md.spcvz=NaN*ones(md.numberofnodes,1);
				 pos=find(structmd.spcvelocity(:,1)); md.spcvx(pos)=structmd.spcvelocity(pos,4); 
				 pos=find(structmd.spcvelocity(:,2)); md.spcvy(pos)=structmd.spcvelocity(pos,5); 
				 pos=find(structmd.spcvelocity(:,3)); md.spcvz(pos)=structmd.spcvelocity(pos,6); 
			 end
			 if isfield(structmd,'spcvelocity'), 
				 md.spcvx=NaN*ones(md.numberofnodes,1);
				 md.spcvy=NaN*ones(md.numberofnodes,1);
				 md.spcvz=NaN*ones(md.numberofnodes,1);
				 pos=find(structmd.spcvelocity(:,1)); md.spcvx(pos)=structmd.spcvelocity(pos,4); 
				 pos=find(structmd.spcvelocity(:,2)); md.spcvy(pos)=structmd.spcvelocity(pos,5); 
				 pos=find(structmd.spcvelocity(:,3)); md.spcvz(pos)=structmd.spcvelocity(pos,6); 
			 end

			 %New fields
			 if ~isfield(structmd,'upperelements');
				 md.upperelements=transpose(1:md.numberofelements)+md.numberofelements2d;
				 md.upperelements(end-md.numberofelements2d+1:end)=NaN;
			 end
			 if ~isfield(structmd,'lowerelements');
				 md.lowerelements=transpose(1:md.numberofelements)-md.numberofelements2d;
				 md.lowerelements(1:md.numberofelements2d)=NaN;
			 end
			 if ~isfield(structmd,'diagnostic_ref');
				 md.diagnostic_ref=NaN*ones(md.numberofnodes2d,6);
			 end
		 end% }}}
		 function md = setdefaultparameters(md) % {{{1
		 %SETDEFAULTPARAMETERS - plug default parameters onto model
		 %
		 %   Although the model parameterization should be done in
		 %   the parameter file, some parameters are initialized here
		 %   with a default value.
		 %   These default values can be changed if necessary.
		 %
		 %   Usage:
		 %      md=setdefaultparameters(md);

			 %first, use the defaults provided by the properties definition above. 
			 fieldnames=fields(md);
			 for i=1:length(fieldnames),
				 fieldname=fieldnames{i};
				 md.(fieldname)=md.(fieldname){1};
			 end

			 %Materials parameters

			 %ice density (kg/m^3)
			 md.rho_ice=917;

			 %water density (kg/m^3)
			 md.rho_water=1023;

			 %firn density (kg/m^3)
			 md.rho_firn=830;

			 %ice heat capacity cp (J/kg/K)
			 md.heatcapacity=2093;

			 %ice latent heat of fusion L (J/kg)
			 md.latentheat=3.34*10^5;

			 %basal melting rate correction: 
			 md.gl_melting_rate=0; 
			 md.basal_melting_rate_correction_apply=0; %do not apply

			 %ice thermal conductivity lamda (W/m/K)
			 md.thermalconductivity=2.4;

			 %the melting point of ice at 1 atmosphere of pressure in K
			 md.meltingpoint=273.15;

			 %the reference temperature for enthalpy model (cf Aschwanden)
			 md.referencetemperature=223.15;

			 %rate of change of melting point with pressure (K/Pa)
			 md.beta=9.8*10^-8;

			 %mixed layer (ice-water interface) heat capacity (J/kg/K)
			 md.mixed_layer_capacity=3974;

			 %thermal exchange velocity (ice-water interface) (m/s)
			 md.thermal_exchange_velocity=1.00*10^-4;

			 %Physical parameters

			 %acceleration due to gravity (m/s^2)
			 md.g=9.81;

			 %converstion from year to seconds
			 md.yts=365*24*3600;

			 %drag type (2=elastic)
			 md.drag_type=2;

			 %Solver parameters

			 %mechanical residue convergence criterion norm(K(uold)uold - F)/norm(F)
			 md.eps_res=10^-4; %from test233 in nightly runs

			 %relative convergence criterion ((vel(n-1)-vel(n))/vel(n))
			 md.eps_rel=0.01;

			 %absolute convergence criterion (max(vel(n-1)-vel(n)) (m/year)
			 md.eps_abs=10;

			 %maximum of non-linear iterations.
			 md.max_nonlinear_iterations=100;
			 
			 %maximum of steady state iterations
			 md.max_steadystate_iterations=100;

			 %sparsity
			 md.sparsity=0.001;

			 %the connectivity is the avergaded number of nodes linked to a
			 %given node through an edge. This connectivity is used to initially
			 %allocate memory to the stiffness matrix. A value of 16 seems to
			 %give a good memory/time ration. This value can be checked in
			 %trunk/test/Miscellaneous/runme.m
			 md.connectivity=25;

			 %lowmem??
			 md.lowmem=0;

			 %coefficient to update the viscosity between each iteration of
			 %a diagnostic according to the following formula
			 %viscosity(n)=viscosity(n)+viscosity_overshoot(viscosity(n)-viscosity(n-1))
			 md.viscosity_overshoot=0;

			 %parameter used to print temporary results (convergence criterion,
			 %current step,...)
			 md.verbose=verbose('solution',true,'qmu',true,'control',true);

			 %Stokes
			 md.stokesreconditioning=10^13;
			 md.shelf_dampening=0;

			 %Penalties

			 %the penalty offset is the power used to compute the value
			 %of the penalty as follows
			 %kappa=max(stiffness matrix)*10^penalty_offset
			 md.penalty_offset=3;

			 %penalty_melting is the penalty used to keep T<Tpmp
			 %kappa=penalty_melting
			 md.penalty_melting=10^7;

			 %in some solutions, it might be needed to stop a run when only
			 %a few constraints remain unstable. For thermal computation, this
			 %parameter is often used.
			 md.penalty_lock=10;

			 %minimum thickness to avoid stiffness singularity, used in
			 %the parameter file
			 md.minh=1;

			 %in some cases, it might be needed to stop a run when only
			 %a few constraints remain unstable. For thermal computation, this
			 %parameter is often used.
			 md.min_thermal_constraints=0;
			 md.min_mechanical_constraints=0;

			 %Transient parameters

			 %time between 2 time steps
			 md.dt=1/2;

			 %number of time steps = md.ndt / md.dt
			 md.ndt=10*md.dt;

			 %time adaptation? 
			 md.time_adapt=0;
			 md.cfl_coefficient=.5;

			 %Hydrostatic adjustment
			 md.hydrostatic_adjustment=AbsoluteEnum;
			 %the artificial diffusivity is used in prognostic to avoid
			 %numerical wiggles of the solution.
			 md.artificial_diffusivity=1;

			 %Discontinuous Galerkin is used in prognostic to avoid
			 %numerical wiggles of the solution and conserve mass
			 md.prognostic_DG=0;

			 %minimum thickness change between 2 time steps (m)
			 md.deltaH=10^-2;

			 %maximum thickness change between 2 time steps (m)
			 md.DeltaH=50;

			 %minimum temperature change between 2 time steps (K)
			 md.deltaT=10^-2;

			 %maximum temperature change between 2 time steps (K)
			 md.DeltaT=1;

			 %When the previous criteria are fulfilled, if the time steps
			 %can be accelerated as follows
			 %dt = timeacc * dt
			 md.timeacc=2;

			 %When the previous criteria are fulfilled, if the time steps
			 %can be reduced as follows
			 %dt = timedec * dt
			 md.timedec=.5;

			 %Solution activated for transient runs. By default we do a
			 %full analysis: Diagnostic, Prognostic and Thermal
			 md.isprognostic=1;
			 md.isdiagnostic=1;
			 md.isthermal=1;

			 %Control

			 %parameter to be inferred by control methods (only
			 %drag and B are supported yet)
			 md.control_type=DragCoefficientEnum();

			 %number of steps in the control methods
			 md.nsteps=20;

			 %maximum number of iteration in the optimization algorithm for
			 %each step
			 md.maxiter=20*ones(md.nsteps,1);

			 %tolerance used by the optimization algorithm
			 md.tolx=10^-4;

			 %the inversed parameter is updated as follows:
			 %new_par=old_par + optscal(n)*C*gradient with C in [0 1];
			 %usually the optscal must be of the order of magnitude of the 
			 %inversed parameter (10^8 for B, 50 for drag) and can be decreased
			 %after the first iterations
			 md.optscal=50*ones(md.nsteps,1);

			 %several responses can be used:
			 md.cm_responses=SurfaceAbsVelMisfitEnum*ones(md.nsteps,1);

			 %cm_jump is used to speed up control method. When
			 %misfit(1)/misfit(0) < md.cm_jump, we go directly to
			 %the next step
			 md.cm_jump=.7*ones(md.nsteps,1); %30 per cent decrement.

			 %stop control solution at the gradient computation and return it? 
			 md.cm_gradient=0;

			 %eps_cm is a criteria to stop the control methods.
			 %if J[n]-J[n-1]/J[n] < criteria, the control run stops
			 %NaN if not applied
			 md.eps_cm=NaN; %not activated

			 %minimum velocity to avoid the misfit to be singular
			 md.epsvel=eps;

			 %averaged velocity used to scale the logarithmic Misfit (000 m/an)
			 md.meanvel=1000/(365*24*3600);

			 %grounding line migration: 
			 md.gl_migration=NoneEnum;

			 %How often to save results, default is 1 so save every step
			 md.output_frequency=1;

			 %Parallelisation parameters

			 %cluster set as none for serial
			 md.cluster=none;

			 %this option can be activated to load automatically the results
			 %onto the model after a parallel run by waiting for the lock file
			 %N minutes that is generated once the solution has converged
			 %0 to desactivate
			 md.waitonlock=Inf;

			 %prerun allocation cleanup
			 md.alloc_cleanup=1;

			 %set petsc options for different analysis
			 md.petscoptions=petscoptions;
			 md.petscoptions=addoptions(md.petscoptions,DiagnosticVertAnalysisEnum,mumpsoptions);

			 %Rheology law: what is the temperature dependence of B with T
			 %available: None, Paterson and Arrhenius
			 md.rheology_law=PatersonEnum;

			 %i/o:
			 md.io_gather=1;
		 end
		 %}}}
		 function md = subsasgn(md,index,val) % {{{1

			 if ~isempty(md.repository),
				 field=index(1).subs;
				 if exist(['.' md.repository '.' field],'file'),
					 if length(index)==1,
						 %save val in corresponding repository
						 eval([field '=val;']);
						 eval(['save .' md.repository '.' field ' ' field]);
					 else
						 %load field from file
						 load(['.' md.repository '.' field],'-mat');
						 fieldval=eval(field);
						 eval([field '=builtin(''subsasgn'',fieldval,index(2),val);']);
						 eval(['save .' md.repository '.' field ' ' field]);
					 end
				 else
					 md=builtin('subsasgn',md,index,val);
				 end
			 else
				 md=builtin('subsasgn',md,index,val);
			 end
	 end %}}}
		 function result = subsref(md,index) % {{{1

			 if length(index)==1,
				 index1=index(1);
				 if(strcmp(index1.subs,'mesh')), displaymesh(md);return; end
				 if(strcmp(index1.subs,'bc')), displaybc(md);return; end
				 if(strcmp(index1.subs,'mat')), displaymaterials(md);return; end
				 if(strcmp(index1.subs,'par')), displayparameters(md);return; end
				 if(strcmp(index1.subs,'exppar')), displayexppar(md);return; end
				 if(strcmp(index1.subs,'res')), displayresults(md);return; end
				 if(strcmp(index1.subs,'obs')), displayobservations(md);return; end
				 if(strcmp(index1.subs,'qmu')), displayqmu(md);return; end
				 if(strcmp(index1.subs,'flaim')), displayflaim(md);return; end
				 if(strcmp(index1.subs,'diagnostic')), displaydiagnostic(md);return; end
				 if(strcmp(index1.subs,'prognostic')), displayprognostic(md);return; end
				 if(strcmp(index1.subs,'thermal')), displaythermal(md);return; end
				 if(strcmp(index1.subs,'transient')), displaytransient(md);return; end
				 if(strcmp(index1.subs,'control')), displaycontrol(md);return; end
				 if(strcmp(index1.subs,'parallel')), displayparallel(md);return; end

				 if ~isempty(md.repository),
					 %load variable needed, if necessary
					 if exist(['.' md.repository '.' index1.subs],'file'),
						 structure=load(['.' md.repository '.' index1.subs],'-mat');
						 eval(['result=structure.' index1.subs ';']);
					 else
						 %the file does not exist, just return built in value
						 result=builtin('subsref',md,index);
					 end
				 else
					 %return built in value
					 result=builtin('subsref',md,index);
				 end
			 else
				 if ~isempty(md.repository),
					 if exist(['.' md.repository '.' index(1).subs],'file'),
						 %load variable needed
						 variable=load(['.' md.repository '.' index(1).subs],'-mat');

						 %use temporary structure.
						 result=builtin('subsref',variable,index);
					 else
						 result=builtin('subsref',md,index);
					 end
				 else
					 result=builtin('subsref',md,index);
				 end
			 end
		 end %}}}
	 end
 end
