/*!\file: solver_stokescoupling_nonlinear.cpp
 * \brief: core of the coupling between stokes and macayealpattyn
 */ 

#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "../modules/modules.h"
#include "../solutions/solutions.h"
#include "./solvers.h"

void solver_stokescoupling_nonlinear(FemModel* femmodel,bool conserve_loads){

	/*intermediary: */
	Mat Kff_horiz = NULL, Kfs_horiz   = NULL;
	Vec ug_horiz  = NULL, uf_horiz  = NULL, old_uf_horiz = NULL;
	Vec pf_horiz  = NULL;
	Vec df_horiz  = NULL;
	Mat Kff_vert  = NULL, Kfs_vert    = NULL;
	Vec ug_vert   = NULL, uf_vert   = NULL;
	Vec pf_vert   = NULL;
	Vec df_vert   = NULL;
	Vec ys   = NULL;
	int converged;
	int constraints_converged;
	int num_unstable_constraints;
	int count;

	/*parameters:*/
	int min_mechanical_constraints;
	int max_nonlinear_iterations;
	int configuration_type;

	/*Recover parameters: */
	femmodel->parameters->FindParam(&min_mechanical_constraints,MinMechanicalConstraintsEnum);
	femmodel->parameters->FindParam(&max_nonlinear_iterations,MaxNonlinearIterationsEnum);
	UpdateConstraintsx(femmodel->nodes,femmodel->constraints,femmodel->parameters);
	
	count=1;
	converged=0;

	/*First get ug_horiz:*/
	femmodel->SetCurrentConfiguration(DiagnosticHorizAnalysisEnum);
	GetSolutionFromInputsx(&ug_horiz, femmodel->elements, femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters);
	Reducevectorgtofx(&uf_horiz, ug_horiz, femmodel->nodes,femmodel->parameters);

	for(;;){

		/*First diagnostic horiz:*/
		femmodel->SetCurrentConfiguration(DiagnosticHorizAnalysisEnum);
		femmodel->parameters->FindParam(&configuration_type,ConfigurationTypeEnum);
	
		//Update once again the solution to make sure that vx and vxold are similar (for next step in transient or steadystate)
		InputUpdateFromSolutionx(femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,ug_horiz);

		//save pointer to old velocity
		VecFree(&old_uf_horiz);old_uf_horiz=uf_horiz;

		/*solve: */
		SystemMatricesx(&Kff_horiz, &Kfs_horiz, &pf_horiz, &df_horiz, NULL,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);
		CreateNodalConstraintsx(&ys,femmodel->nodes,configuration_type);
		Reduceloadx(pf_horiz, Kfs_horiz, ys); MatFree(&Kfs_horiz);
		Solverx(&uf_horiz, Kff_horiz, pf_horiz, old_uf_horiz, df_horiz,femmodel->parameters);
		Mergesolutionfromftogx(&ug_horiz, uf_horiz,ys,femmodel->nodes,femmodel->parameters); VecFree(&ys);
		InputUpdateFromSolutionx( femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,ug_horiz);

		convergence(&converged,Kff_horiz,pf_horiz,uf_horiz,old_uf_horiz,femmodel->parameters); MatFree(&Kff_horiz);VecFree(&pf_horiz); VecFree(&df_horiz);

		/*Second compute vertical velocity: */
		femmodel->SetCurrentConfiguration(DiagnosticVertAnalysisEnum);
		femmodel->parameters->FindParam(&configuration_type,ConfigurationTypeEnum);
		
		/*solve: */
		SystemMatricesx(&Kff_vert, &Kfs_vert, &pf_vert,  &df_vert,NULL,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);
		CreateNodalConstraintsx(&ys,femmodel->nodes,configuration_type);
		Reduceloadx(pf_vert, Kfs_vert, ys); MatFree(&Kfs_vert);
		Solverx(&uf_vert, Kff_vert, pf_vert, NULL, df_vert,femmodel->parameters); MatFree(&Kff_vert); VecFree(&pf_vert); VecFree(&df_vert);
		Mergesolutionfromftogx(&ug_vert, uf_vert,ys,femmodel->nodes,femmodel->parameters);VecFree(&uf_vert); VecFree(&ys);
		InputUpdateFromSolutionx( femmodel->elements,femmodel->nodes, femmodel->vertices, femmodel->loads, femmodel->materials, femmodel->parameters,ug_vert);
		VecFree(&ug_vert); VecFree(&uf_vert);

		/*Increase count: */
		count++;
		if(converged==1)break;
		if(count>=max_nonlinear_iterations){
			_printf_(true,"   maximum number of iterations (%i) exceeded\n",max_nonlinear_iterations); 
			break;
		}
	}

	/*clean-up*/
	VecFree(&old_uf_horiz);
	VecFree(&uf_horiz);
	VecFree(&ug_horiz);
	VecFree(&ys);
}
