function ismodelselfconsistent(md),
%ISMODELSELFCONSISTENT - check that model forms a closed form solvable problem.
%
%   Usage:
%      ismodelselfconsistent(md),

%tolerance we use in litmus checks for the consistency of the model
tolerance=10^-9;
%check usage {{{1
if nargin~=1,
	help ismodelselfconsistent
	message('ismodelselfconsistent message message: wrong usage');
end
%}}}

%initialize consistency as true
modelconsistency(true);

% Common checks
%VERBOSE{{{1
if ~strcmp('verbose',class(md.verbose))
	disp('WARNING: md.verbose must now be an object of class ''verbose''');
	disp('         To update your model use the following command:');
	disp(' ');
	disp('         md.verbose=verbose;');
	disp(' ');
	message(['field verbose should be of class ''verbose'' ']);
end
%}}}
%COUNTER {{{1
if md.counter<3,
	message(['model ' md.name ' is not correctly configured. You forgot one step in the following sequence (mesh, geography, parameterize,setelementstype)!']);
end
%}}}
%NAME{{{1
if isempty(md.name),
	message(['model is not correctly configured: missing name!']);
end
%}}}
%ELEMENTS{{{1
fields={'elements'};
if (md.dim==2),
	checksize(md,fields,[md.numberofelements 3]);
else
	checksize(md,fields,[md.numberofelements 6]);
end
if any(~ismember(1:md.numberofnodes,sort(unique(md.elements(:)))));
	message('orphan nodes have been found. Check the mesh');
end
%}}}

%DG {{{1
if md.prognostic_DG==1;
	%CHECK THE COMPATIBILITY OF THE EDGES
	fields={'edges'};
	checksize(md,fields,[NaN 4]);
	fields={'edges(:,1:3)'};
	checknan(md,fields);
	checkgreaterstrict(md,fields,0);
end
%}}}
%PRESSURELOAD{{{1
if (md.dim==2),
	fields={'pressureload'};
	checksize(md,fields,[NaN 4]);
elseif md.dim==3,
	fields={'pressureload'};
	checksize(md,fields,[NaN 6]);
else
	message('dim should be either 2 3');
end
checkvalues(md,{'pressureload(:,end)'},[WaterEnum() AirEnum() IceEnum()]);
%}}}
%NO NAN {{{1
fields={'numberofelements','numberofnodes','x','y','z','drag_coefficient','drag_type','drag_p','drag_q',...
	'rho_ice','rho_water','rheology_B','elementoniceshelf','surface','thickness','bed','g','lowmem','sparsity','nsteps','maxiter',...
	'tolx','eps_res','max_nonlinear_iterations','rheology_n','nodeonbed','nodeonsurface','elementonbed','elementonsurface','deltaH','DeltaH','timeacc','timedec','elementconnectivity'};
checknan(md,fields);
%}}}}
%FIELDS >= 0 {{{1
fields={'numberofelements','numberofnodes','elements','drag_coefficient','drag_type','drag_p','drag_q',...
	'rho_ice','rho_water','rheology_B','elementoniceshelf','thickness','g','eps_res','max_nonlinear_iterations','eps_rel','eps_abs','nsteps','maxiter','tolx',...
	'sparsity','lowmem','rheology_n','nodeonbed','nodeonsurface','elementonbed','elementonsurface','deltaH','DeltaH','timeacc','timedec'};
checkgreater(md,fields,0);
%}}}
%FIELDS > 0 {{{1
fields={'numberofelements','numberofnodes','elements','drag_type','drag_p',...
	'rho_ice','rho_water','rheology_B','thickness','g','max_nonlinear_iterations','eps_res','eps_rel','eps_abs','maxiter','tolx',...
	'sparsity','deltaH','DeltaH','timeacc','timedec'};
checkgreaterstrict(md,fields,0);
%}}}
%SIZE NUMBEROFELEMENTS {{{1
fields={'drag_p','drag_q','elementoniceshelf','rheology_n','elementonbed'};
checksize(md,fields,[md.numberofelements 1]);
%}}}
%SIZE NUMBEROFNODES {{{1
fields={'x','y','z','rheology_B','drag_coefficient','basal_melting_rate','surface','thickness','bed','nodeonbed','nodeonsurface'};
checksize(md,fields,[md.numberofnodes 1]);
%}}}
%OTHER SIZES {{{1
fields={'diagnostic_ref'};
checksize(md,fields,[md.numberofnodes 6]);
if ~isempty(md.requested_outputs),
	if(size(md.requested_outputs,2)~=1),
		message(['model ' md.name ' requested outputs should be a column vector']);
	end
end
%}}}
%THICKNESS = SURFACE - BED {{{1
if any((md.thickness-md.surface+md.bed)>tolerance),
	message(['model not consistent: model ' md.name ' violates the equality thickness=surface-bed!']);
end
%GROUNDING LINE MIGRATION {{{1
checkvalues(md,{'gl_migration'},[NoneEnum() AgressiveMigrationEnum() SoftMigrationEnum()]);
if (md.gl_migration~=NoneEnum),
	if (md.dim==3 | strcmpi(md.cluster.name,'none')),
		message(['model ' md.name ' requesting grounding line migration, but grounding line module only implemented for 2d models and parallel runs!']);
	end
	if isnan(md.bathymetry),
		message(['model not consistent: model ' md.name ' requesting grounding line migration, but bathymetry is absent!']);
	end
	pos=find(md.nodeonicesheet); 
	if any(md.bed(pos)-md.bathymetry(pos)),
		message(['model not consistent: model ' md.name ' bathymetry not equal to bed on icesheet!']);
	end
	pos=find(md.nodeoniceshelf); 
	if any(md.bathymetry(pos)-md.bed(pos)>tolerance),
		message(['model not consistent: model ' md.name ' bathymetry superior to bed on iceshelf!']);
	end

end

%}}}
%RIFTS{{{1
if md.numrifts,
	if ~(md.dim==2),
		message(['model not consistent: models with rifts are only supported in 2d for now!']);
	end
	if ~isstruct(md.rifts),
		message(['model not consistent: md.rifts should be a structure!']);
	end
	if ~isempty(find(md.segmentmarkers>=2)),
		%We have segments with rift markers, but no rift structure!
		message(['model not consistent: model ' md.name ' should be processed for rifts (run meshprocessrifts)!']);
	end
	%Check that rifts are filled with proper material
	checkvalues(md,{'rifts.fill'},[WaterEnum() AirEnum() IceEnum() MelangeEnum()]);
else
	if ~isnans(md.rifts),
		message(['model not consistent: md.rifts shoud be NaN since md.numrifts is 0!']);
	end
end
%}}}
%FLAGS (0 or 1){{{1
if ~ismember(md.artificial_diffusivity,[0 1 2]),
	message('model not consistent: artificial_diffusivity should be a scalar (0 or 1 or 2)');
end
if ~ismember(md.prognostic_DG,[0 1]),
	message('model not consistent: prognostic_DG should be a scalar (1 or 0)');
end
if ~ismember(md.lowmem,[0 1]),
	message(['model not consistent: model ' md.name ' lowmem field should be 0 or 1']);
end
if ~ismember(md.time_adapt,[0 1]),
	message(['model not consistent: model ' md.name ' time_adapt field should be 0 or 1']);
end
if ~ismember(md.hydrostatic_adjustment,[AbsoluteEnum IncrementalEnum]),
	message(['model not consistent: model ' md.name ' hydrostatic_adjustment field should be AbsoluteEnum or IncrementalEnum']);
end
if ~ismember(md.rheology_law,[NoneEnum PatersonEnum ArrheniusEnum]),
	message(['model not consistent: model ' md.name ' rheology_law field should be NoneEnum, PatersonEnum or ArrheniusEnum']);
end
%}}}
%CONNECTIVITY {{{1
if (md.dim==2),
	if md.connectivity<9, 
		message('model not consistent: connectivity should be at least 9 for 2d models');
	end
end
if md.dim==3,
	if md.connectivity<24, 
		message('model not consistent: connectivity should be at least 24 for 3d models');
	end
end
%}}}
%PARALLEL{{{1
IsConsistent(md.cluster);
%}}}
%CONTROL{{{1
if md.control_analysis,

	%CONTROL TYPE
	num_controls=numel(md.control_type);
	num_costfunc=size(md.cm_responses,2);
	checkvalues(md,{'control_type'},[DhdtEnum DragCoefficientEnum RheologyBbarEnum VxEnum VyEnum]);

	%LENGTH CONTROL FIELDS
	fields={'maxiter','cm_jump'};
	checksize(md,fields,[md.nsteps 1]);
	fields={'cm_responses'};
	checksize(md,fields,[md.nsteps num_costfunc]);
	fields={'optscal'};
	checksize(md,fields,[md.nsteps num_controls]);
	fields={'cm_min','cm_max'};
	checksize(md,fields,[md.numberofnodes num_controls]);

	%RESPONSES
	checkvalues(md,{'cm_responses'},...
		[SurfaceAbsVelMisfitEnum SurfaceRelVelMisfitEnum SurfaceLogVelMisfitEnum SurfaceLogVxVyMisfitEnum SurfaceAverageVelMisfitEnum...
		ThicknessAbsMisfitEnum DragCoefficientAbsGradientEnum RheologyBbarAbsGradientEnum ThicknessAbsGradientEnum]);

	%WEIGHTS
	fields={'weights'};
	checksize(md,fields,[md.numberofnodes num_costfunc]);
	checkgreater(md,fields,0);

	%OBSERVED VELOCITIES
	if md.solution_type==BalancethicknessSolutionEnum
		fields={'thickness_obs'};
		checksize(md,fields,[md.numberofnodes 1]);
		checknan(md,fields);
	else
		fields={'vx_obs','vy_obs'};
		checksize(md,fields,[md.numberofnodes 1]);
		checknan(md,fields);
	end

	%DIRICHLET IF THICKNESS <= 0
	if any(md.thickness<=0),
		pos=find(md.thickness<=0);
		if any(isnan(md.spcthickness(pos))),
			message(['model not consistent: model ' md.name ' has some nodes with 0 thickness']);
		end
	end
end
%}}}
%QMU {{{1
if md.qmu_analysis,
	if md.qmu_params.evaluation_concurrency~=1,
		message(['model not consistent: concurrency should be set to 1 when running dakota in library mode']);
	end
	if ~isempty(md.part),
		if numel(md.part)~=md.numberofnodes,
			message(['model not consistent: user supplied partition for qmu analysis should have size md.numberofnodes x 1 ']);
		end
		if find(md.part)>=md.numberofnodes,
			message(['model not consistent: user supplied partition should be indexed from 0 (c-convention)']);
		end
		if min(md.part)~=0,
			message(['model not consistent: partition vector not indexed from 0 on']);
		end
		if max(md.part)>=md.numberofnodes,
			message(['model not consistent: partition vector cannot have maximum index larger than number of nodes']);
		end
		if ~isempty(find(md.part<0)),
			message(['model not consistent: partition vector cannot have values less than 0']);
		end
		if ~isempty(find(md.part>=md.npart)),
			message(['model not consistent: partition vector cannot have values more than md.npart-1']);
		end
		if max(md.part)>=md.npart,
			message(['model not consistent: for qmu analysis, partitioning vector cannot go over npart, number of partition areas']);
		end
	end
	if ~md.qmu_relax,
		%if md.eps_rel>1.1*10^-5,
		%	message(['model not consistent: for qmu analysis, eps_rel should be least than 10^-5, 10^-15 being a better value']);
		%end
	end
end

if strcmpi(md.solution_type,'qmu'),
	if ~strcmpi(md.cluster.name,'none'),
		if md.waitonlock==0,
			message(['model is not correctly configured: waitonlock should be activated when running qmu in parallel mode!']);
		end
	end
end
%}}}

%Solution specific check
%TRANSIENT {{{1
if (md.solution_type==TransientSolutionEnum),

	if md.dt<=0,
		message('model not consistent: field dt must be positive for a transient run')
	end
	if(md.cfl_coefficient>1 | md.cfl_coefficient<0),
		message(['model not consistent: model ' md.name ' cfl_coefficient field should between  0 and 1']);
	end
	if(md.cfl_coefficient>1 | md.cfl_coefficient<0),
		message(['model not consistent: model ' md.name ' cfl_coefficient field should between  0 and 1']);
	end
	if ~ismember(md.isdiagnostic,[0 1]),
		message('model not consistent: isdiagnostic should be a scalar (1 or 0)');
	end
	if ~ismember(md.isprognostic,[0 1]),
		message('model not consistent: isprognostic should be a scalar (1 or 0)');
	end
	if ~ismember(md.isthermal,[0 1]),
		message('model not consistent: isthermal should be a scalar (1 or 0)');
	end
end
%}}}
%STEADYSTATE{{{1
if md.solution_type==SteadystateSolutionEnum,

	%NDT
	if md.dt~=0,
		message(['model not consistent: for a steadystate computation, dt must be zero.']);
	end

	%eps: 
	if isnan(md.eps_rel),
		message(['model not consistent: for a steadystate computation, eps_rel (relative convergence criterion) must be defined!']);
	end
end
%}}}
%GROUNDINGLINEMIGRATION2D{{{1
if md.solution_type==GroundingLineMigration2DSolutionEnum,
	if strcmpi(md.cluster.name,'none'),
		message(['model not consistent: ' md.solution_type ' is only implemented in parallel mode !'])
	end

	if md.dt<=0,
		message('model not consistent: field dt must be positive for a transient run')
	end

	if (md.dim~=2),
		message(['model not consistent: for a ' md.solution_type ' computation, the grounding line module is only implemented in 2d !'])
	end

	if(md.cfl_coefficient>1 | md.cfl_coefficient<0),
		message(['model not consistent: model ' md.name ' cfl_coefficient field should between  0 and 1']);
	end
end
%}}}
%FLAIM {{{1
if (md.solution_type == FlaimSolutionEnum),
	if ~exist(md.fm_tracks,'file')
		message(['model not consistent: fm_tracks file ''' md.fm_tracks ''' must exist.']);
	end
	%   probably going to need some checks on fm_flightreqs here
	if (numel(md.fm_criterion) ~= md.numberofnodes) && (numel(md.fm_criterion) ~= md.numberofelements)
		message(['model not consistent: fm_criterion vector must have number of nodes (' int2str(md.numberofnodes) ') or elements (' int2str(md.numberofelements) ') values, not ' int2str(numel(md.fm_criterion)) ' values.']);
	end
end
%}}}

%Now check all analyses called for a given solution
%ANALYSESCHECKS {{{1
[analyses,numanalyses,solutioncore]=SolutionConfiguration(md.solution_type);
for i=1:numanalyses,
	checkforanalysis(md,analyses(i));
end
%}}}

	if modelconsistency==false, error(['model not consistent']); end
end

%analysis checks
%checkforanalysis {{{1
function checkforanalysis(md,analysis_enum)

	switch(analysis_enum),
		case DiagnosticHorizAnalysisEnum,
			% {{{2
			%SINGULAR
			if ~any((~isnan(md.spcvx)+~isnan(md.spcvy))==2),
				message(['model not consistent: model ' md.name ' is not well posed (singular). You need at least one node with fixed velocity!'])
			end

			%DIRICHLET IF THICKNESS <= 0
			%Check the size of spcthickness 
			fields={'spcthickness'};
			checksize(md,fields,[md.numberofnodes 1]);
			if any(md.thickness<=0),
				pos=find(md.thickness<=0);
				if any(isnan(md.spcthickness(pos))),
					message(['model not consistent: model ' md.name ' has some nodes with 0 thickness']);
				end
			end

			%ROTATED SPC
			%CHECK THAT EACH LINES CONTAINS ONLY NAN VALUES OR NO NAN VALUES
			if any(sum(isnan(md.diagnostic_ref),2)~=0 & sum(isnan(md.diagnostic_ref),2)~=6),
				message(['model not consistent: model ' md.name ' has problem with rotated spc. Each line of diagnostic_ref should contain either only NaN values or no NaN values']);
			end
			%CHECK THAT THE TWO VECTORS PROVIDED ARE ORTHOGONAL
			if any(sum(isnan(md.diagnostic_ref),2)==0),
				pos=find(sum(isnan(md.diagnostic_ref),2)==0);
				if any(dot(md.diagnostic_ref(pos,1:3),md.diagnostic_ref(pos,4:6))),
					dot(md.diagnostic_ref(pos,1:3),md.diagnostic_ref(pos,4:6))
					message(['model not consistent: model ' md.name ' has problem with rotated spc. Vectors in diagnostic_ref (colums 1 to 3 and 4 to 6) must be orthogonal']);
				end

			end
			%CHECK THAT ROTATION IS IN THE (X,Y) PLANE FOR 2D MODELS
			if any(md.nodeonmacayeal),
				pos=find(sum(isnan(md.diagnostic_ref),2)==0  & md.nodeonmacayeal);
				if any(md.diagnostic_ref(pos,3:5)~=0);
					message(['model not consistent: model ' md.name ' has problem with rotated spc. The rotation should be in the (x,y) plane for 2D diagnostic models (nodeonmacayeal)']);
				end
			end

			%INITIAL VELOCITY
			if ~isnan(md.vx) & ~isnan(md.vy),
				fields={'vx','vy'};
				checknan(md,fields);
				checksize(md,fields,[md.numberofnodes 1]);
			end

			%ELEMENTSTYPE
			%Check the size of elements_type
			fields={'elements_type'};
			checksize(md,fields,[md.numberofelements 1]);
			%Check the values of elements_type
			checkvalues(md,{'elements_type'},[MacAyealApproximationEnum() HutterApproximationEnum() PattynApproximationEnum()...
				MacAyealPattynApproximationEnum() MacAyealStokesApproximationEnum() PattynStokesApproximationEnum() StokesApproximationEnum() NoneApproximationEnum()]);
			if (md.dim==2),
				checkvalues(md,{'elements_type'},[MacAyealApproximationEnum() HutterApproximationEnum()]);
			end
			if (md.ismacayealpattyn==0 && md.ishutter==0 && md.isstokes==0),
				message(['model not consistent: no elements type set for this model. at least one of ismacayealpattyn, ishutter and isstokes need to be =1']);
			end
			
			%VERTICESTYPE
			%Check the size of verticess_type
			fields={'vertices_type'};
			checksize(md,fields,[md.numberofnodes 1]);
			%Check the values of vertices_type
			checkvalues(md,{'vertices_type'},[MacAyealApproximationEnum() HutterApproximationEnum() PattynApproximationEnum()...
				MacAyealPattynApproximationEnum() StokesApproximationEnum() MacAyealStokesApproximationEnum() PattynStokesApproximationEnum() NoneApproximationEnum()]);
			if (md.dim==2),
				checkvalues(md,{'vertices_type'},[MacAyealApproximationEnum() HutterApproximationEnum()]);
			end
			if (md.ismacayealpattyn==0 && md.ishutter==0 && md.isstokes==0),
				message(['model not consistent: no elements type set for this model. at least one of ismacayealpattyn, ishutter and isstokes need to be =1']);
			end
			%}}}
		case DiagnosticVertAnalysisEnum,
			% {{{2
			if md.dim==3,
				% No checks for now
			end
			%}}}
		case DiagnosticHutterAnalysisEnum,
			% {{{2
			%HUTTER ON ICESHELF WARNING
			if any(md.elements_type==HutterApproximationEnum & md.elementoniceshelf),
				disp(sprintf('\n !!! Warning: Hutter''s model is not consistent on ice shelves !!!\n'));
			end
			%}}}
		case BedSlopeAnalysisEnum,
			% {{{2
			% No checks for now
			%}}}
		case SurfaceSlopeAnalysisEnum,
			% {{{2
			% No checks for now
			%}}}
		case PrognosticAnalysisEnum,
			% {{{2
			%INITIAL VELOCITIES
			fields={'vx','vy'};
			checksize(md,fields,[md.numberofnodes 1]);
			checknan(md,fields);

			%FORCINGS
			fields={'surface_mass_balance'};
			checkforcing(md,fields);
			%}}}
		case HydrologyAnalysisEnum,
			% {{{2
			fields={'spcwatercolumn'};
			checksize(md,fields,[md.numberofnodes 2]);
			fields={'watercolumn'};
			checksize(md,fields,[md.numberofnodes 1]);
			%}}}
		case ThermalAnalysisEnum,
			% {{{2
			%EXTRUSION
			if (md.dim==2),
				if md.solution_type==TransientSolutionEnum,
					return;
				else
					message(['model not consistent: for a ' EnumToString(md.solution_type) ' computation, the model must be 3d, extrude it first!'])
				end
			end

			%CHECK THAT SPCTEMPERATURE IS AN APPROPRIATE FORCING
			fields={'spctemperature'};
			checkforcing(md,fields);

			%CHECK THAT WE ARE NOT FULLY CONSTRAINED
			if ~any(~isnan(md.spctemperature))
				message(['model not consistent: model ' md.name ' is totally constrained for temperature, no need to solve!']);
			end

			%VELOCITIES AND PRESSURE
			fields={'vx','vy','vz','pressure','geothermalflux'};
			checksize(md,fields,[md.numberofnodes 1]);
			checknan(md,fields);

			%THERMAL TRANSIENT
			if md.dt~=0,

				%DT and NDT
				fields={'dt','ndt'};
				checkgreaterstrict(md,fields,0);

				%INITIAL TEMPERATURE, MELTING AND ACCUMULATION
				fields={'temperature','basal_melting_rate'};
				checksize(md,fields,[md.numberofnodes 1]);
				checknan(md,fields);

				%INITIAL TEMPERATURE
				fields={'temperature','observed_temperature'};
				checkgreater(md,fields,0)

				%CHECK SPCTEMPERATURE that are not NaN are >0.
				if find(any(md.spctemperature(find(~isnan(md.spctemperature)))<=0)),
					message(['model not consistent: model ' md.name ' is constrained with negative or nil temperatures!']);
				end

			end
			%}}}
		case EnthalpyAnalysisEnum,
			% {{{2
			%EXTRUSION
			if (md.dim==2),
				if md.solution_type==TransientSolutionEnum,
					return;
				else
					message(['model not consistent: for a ' EnumToString(md.solution_type) ' computation, the model must be 3d, extrude it first!'])
				end
			end

			%VELOCITIES AND PRESSURE
			fields={'vx','vy','vz','pressure','geothermalflux'};
			checksize(md,fields,[md.numberofnodes 1]);
			checknan(md,fields);

			%THERMAL TRANSIENT
			if md.dt~=0,

				%DT and NDT
				fields={'dt','ndt'};
				checkgreaterstrict(md,fields,0);

				%INITIAL TEMPERATURE, MELTING, ACCUMULATION AND WATERFRACTION
				fields={'temperature','basal_melting_rate','waterfraction'};
				checksize(md,fields,[md.numberofnodes 1]);
				checknan(md,fields);

				%INITIAL TEMPERATURE
				fields={'temperature','observed_temperature'};
				checkgreater(md,fields,0)
			end
			%}}}
		case MeltingAnalysisEnum,
			% {{{2
			% No checks for now
			%}}}
		case BalancethicknessAnalysisEnum,
			% {{{2
			%VELOCITIES MELTING AND ACCUMULATION
			fields={'vx','vy','basal_melting_rate','dhdt'};
			checksize(md,fields,[md.numberofnodes 1]);
			checknan(md,fields);

			%Triangle with zero velocity
			if any(sum(abs(md.vx(md.elements)),2)==0 & sum(abs(md.vy(md.elements)),2)==0)
				message('model not consistent: at least one triangle has all its vertices with a zero velocity');
			end
			%}}}
		case BalancevelocitiesAnalysisEnum,
			% {{{2
			%VELOCITIES MELTING AND ACCUMULATION
			fields={'vx','vy','basal_melting_rate'};
			checksize(md,fields,[md.numberofnodes 1]);
			checknan(md,fields);

			%SPC
			if any((~isnan(md.spcvx(find(md.nodeonboundary))))~=1),
				message(['model not consistent: model ' md.name ' should have all the nodes on boundary constrained in field spcvx']);
			end

			if any((~isnan(md.spcvy(find(md.nodeonboundary))))~=1),
				message(['model not consistent: model ' md.name ' should have all the nodes on boundary constrained in field spcvy']);
			end
			%}}}
		case FlaimAnalysisEnum,
			% {{{2
			% No checks for now
			%}}}
		otherwise
			disp(['WARNING: no check implemented for analysis ' EnumToString(analysis_enum) '!']);
			message('stop')
	end

end
%}}}

%checks additional functions
%checklength {{{1
function checklength(md,fields,fieldlength)
	%CHECKSIZE - check length of a field
	for i=1:length(fields),
		eval(['field=md.' fields{i} ';']);
		if length(field)~=fieldlength,
			message(['model not consistent: field ' fields{i} ' length should be ' num2str(fieldlength)]);
		end
	end
end
%}}}
%checksize {{{1
function checksize(md,fields,fieldsize)
	%CHECKSIZE - check size of a field
	for i=1:length(fields),
		eval(['field=md.' fields{i} ';']);
		if isnan(fieldsize(1)),
			if (size(field,2)~=fieldsize(2)),
				message(['model not consistent: field ' fields{i} ' should have ' num2str(fieldsize(2)) ' columns']);
			end
		elseif isnan(fieldsize(2)),
			if (size(field,1)~=fieldsize(1)),
				message(['model not consistent: field ' fields{i} ' should have ' num2str(fieldsize(1)) ' rows']);
			end
		else
			if ((size(field)~=fieldsize(1)) |  (size(field,2)~=fieldsize(2)))
				%WARNING: new version
				if strcmp(fields{i},'cm_min'),
					disp('!!! ');
					disp('!!! WARNING: cm_min must now be of size [md.numberofnodes x 1]. Update your parameter file as follows:');
					disp('!!! ');
					disp('!!! md.cm_min=md.cm_min*ones(md.numberofnodes,1);');
					disp('!!! ');
				end
				%WARNING: new version
				if strcmp(fields{i},'cm_max'),
					disp('!!! ');
					disp('!!! WARNING: cm_max must now be of size [md.numberofnodes x 1]. Update your parameter file as follows:');
					disp('!!! ');
					disp('!!! md.cm_max=md.cm_max*ones(md.numberofnodes,1);');
					disp('!!! ');
				end
				message(['model not consistent: field ' fields{i} ' size should be ' num2str(fieldsize(1)) ' x ' num2str(fieldsize(2))]);
			end
		end
	end
end
%}}}
%checknan {{{1
function checknan(md,fields)
	%CHECKNAN - check nan values of a field
	for i=1:length(fields),
		eval(['field=reshape(md.' fields{i} ',[prod(size(md.' fields{i} ')) 1]);']);
		if any(isnan(field)),
			message(['model not consistent: NaN values found in field ' fields{i}]);
		end
	end
end
%}}}
%checkreal{{{1
function checkreal(md,fields)
	%CHECKREAL - check real values of a field
	for i=1:length(fields),
		eval(['field=reshape(md.' fields{i} ',[prod(size(md.' fields{i} ')) 1]);']);
		if any(~isreal(field)),
			message(['model not consistent: complex values found in field ' fields{i}]);
		end
	end
end
%}}}
%checkgreaterstrict{{{1
function checkgreaterstrict(md,fields,lowerbound)
	%CHECKGREATERSTRICT - check values of a field
	for i=1:length(fields),
		eval(['field=reshape(md.' fields{i} ',[prod(size(md.' fields{i} ')) 1]);']);
		if any(field<=lowerbound),
			message(['model not consistent: field ' fields{i} ' should have values stricly above ' num2str(lowerbound)]);
		end
	end
end
%}}}
%checkgreater{{{1
function checkgreater(md,fields,lowerbound)
	%CHECKGREATER - check values of a field
	for i=1:length(fields),
		eval(['field=reshape(md.' fields{i} ',[prod(size(md.' fields{i} ')) 1]);']);
		if any(field<lowerbound),
			message(['model not consistent: field ' fields{i} ' should have values above ' num2str(lowerbound)]);
		end
	end
end
%}}}
%checklessstrict{{{1
function checklessstrict(md,fields,upperbound)
	%CHECKLESSSTRICT - check values of a field
	for i=1:length(fields),
		eval(['field=reshape(md.' fields{i} ',[prod(size(md.' fields{i} ')) 1]);']);
		if any(field>=upperbound),
			message(['model not consistent: field ' fields{i} ' should have values stricly below ' num2str(upperbound)]);
		end
	end
end
%}}}
%checkless{{{1
function checkless(md,fields,upperbound)
	%CHECKLESS - check values of a field
	for i=1:length(fields),
		eval(['field=reshape(md.' fields{i} ',[prod(size(md.' fields{i} ')) 1]);']);
		if any(field>upperbound),
			message(['model not consistent: field ' fields{i} ' should have values below ' num2str(upperbound)]);
		end
	end
end
%}}}
%checkvalues {{{1
function checkvalues(md,fields,values)
	%CHECKVALUE - check that a field has specified values
	for i=1:length(fields),
		eval(['field=reshape(md.' fields{i} ',[prod(size(md.' fields{i} ')) 1]);']);
		if any(~ismember(field,values)),
			message(['model not consistent: field ' fields{i} ' should have values in ' num2str(values)]);
		end
	end
end
%}}}
%checkforcing {{{1
function checkforcing(md,fieldname)

	eval(['field=md.' fieldname ';']);
	if size(field,1)==md.numberofnodes,
		if ~size(field,2)==1,
			 message(['model not consistent: model ' md.name ': ' fieldname ' should have only one column as there are md.numberofnodes lines']);
		end
	elseif size(field,1)==md.numberofnodes+1
		if any(field(end,:)~=sort(field(end,:))),
			message(['model not consistent: model ' md.name ': ' fieldname ' columns should be chronological']);
		end
		if any(field(end,1:end-1)==field(end,2:end)),
			message(['model not consistent: model ' md.name ': ' fieldname ' columns must not contain duplicate timesteps']);
		end
	else
		 message(['model not consistent: model ' md.name ': ' fieldname ' should have md.numberofnodes or md.numberofnodes+1 lines']);
	end
end
%}}}

%error messages
%modelconsistency{{{1
function flag=modelconsistency(flag_in)

	persistent consistency;

	if nargin==1 & nargout==0,
		%OK model is inconsistent, set flag as false
		consistency=flag_in;
	elseif nargin==0 & nargout==1,
		if isempty(consistency),
			%modelinconsistent has never been called, model is consistent
			consistency=true;
		end
	else
		message('Bad usage');
	end

	flag=consistency;
end
%}}}
%message{{{1
function message(string)

	disp(string);
	modelconsistency(false);
end
%}}}
