function md=SetMarineIceSheetBC(md,varargin)
%SETICEMARINESHEETBC - Create the boundary conditions for diagnostic and thermal models for a  Marine Ice Sheet with Ice Front
%
%   Neumann BC are used on the ice front (an ARGUS contour around the ice front
%   can be given in input, or it will be deduced as oniceshelf & onboundary)
%   Dirichlet BC are used elsewhere for diagnostic
%
%   Usage:
%      md=SetMarineIceSheetBC(md,icefrontfile)
%      md=SetMarineIceSheetBC(md)
%
%   Example:
%      md=SetMarineIceSheetBC(md,'Front.exp')
%      md=SetMarineIceSheetBC(md)
%
%   See also: SETICESHELFBC, SETMARINEICESHEETBC

%node on Dirichlet (boundary and ~icefront)
if nargin==2,
	%User provided Front.exp, use it
	icefrontfile=varargin{1};
	if ~exist(icefrontfile)
		error(['SetMarineIceSheetBC error message: ice front file ' icefrontfile ' not found']);
	end
	nodeinsideicefront=ContourToMesh(md.elements,md.x,md.y,icefrontfile,'node',2);
	nodeonicefront=double(md.nodeonboundary & nodeinsideicefront);
else
	%Guess where the ice front is
	nodeoniceshelf=zeros(md.numberofnodes,1);
	nodeoniceshelf(md.elements(find(md.elementoniceshelf),:))=1;
	nodeonicefront=double(md.nodeonboundary & nodeoniceshelf);
end
pos=find(md.nodeonboundary & ~nodeonicefront);
if isempty(pos),
	warning('SetMarineIceSheetBC warning: ice front all around the glacier, no dirichlet found. Dirichlet must be added manually')
end
md.spcvx=NaN*ones(md.numberofnodes,1);
md.spcvy=NaN*ones(md.numberofnodes,1);
md.spcvz=NaN*ones(md.numberofnodes,1);
md.spcvx(pos)=0;
md.spcvy(pos)=0;
md.spcvz(pos)=0;
md.diagnostic_ref=NaN*ones(md.numberofnodes,6);

%Dirichlet Values
if (length(md.vx_obs)==md.numberofnodes & length(md.vy_obs)==md.numberofnodes)
	disp('      boundary conditions for diagnostic model: spc set as observed velocities');
	md.spcvx(pos)=md.vx_obs(pos);
	md.spcvy(pos)=md.vy_obs(pos);
else
	disp('      boundary conditions for diagnostic model: spc set as zero');
end

md.spcwatercolumn=zeros(md.numberofnodes,2);
pos=find(md.nodeonboundary); 
md.spcwatercolumn(pos,1)=1;

%segment on Neumann (Ice Front)
pos=find(nodeonicefront(md.segments(:,1)) | nodeonicefront(md.segments(:,2)));
if (md.dim==2)
	pressureload=md.segments(pos,:);
elseif md.dim==3
	pressureload_layer1=[md.segments(pos,1:2)  md.segments(pos,2)+md.numberofnodes2d  md.segments(pos,1)+md.numberofnodes2d  md.segments(pos,3)];
	pressureload=[];
	for i=1:md.numlayers-1,
		pressureload=[pressureload ;pressureload_layer1(:,1:4)+(i-1)*md.numberofnodes2d pressureload_layer1(:,5)+(i-1)*md.numberofelements2d ];
	end
end

%Add water or air enum depending on the element
pressureload=[pressureload WaterEnum*md.elementoniceshelf(pressureload(:,end))+AirEnum*md.elementonicesheet(pressureload(:,end))];

%plug onto model
md.pressureload=pressureload;


%Create zeros basal_melting_rate, surface_ablation_rate, surface_accumulation_rate
% and surface_mass_balance if not specified
if isnan(md.surface_accumulation_rate),
	md.surface_accumulation_rate=zeros(md.numberofnodes,1);
	md.forcings.surface_accumulation_rate=zeros(md.numberofnodes+1,1);
	disp('      no surface_accumulation_rate specified: values set as zero');
end
if isnan(md.surface_ablation_rate),
	md.surface_ablation_rate=zeros(md.numberofnodes,1);
	md.forcings.surface_ablation_rate=zeros(md.numberofnodes+1,1);
	disp('      no surface_ablation_rate specified: values set as zero');
end
if isnan(md.surface_mass_balance),
	md.surface_mass_balance=zeros(md.numberofnodes,1);
	md.forcings.surface_mass_balance=zeros(md.numberofnodes+1,1);
	disp('      no surface_mass_balance specified: values set as zero');
end
if isnan(md.basal_melting_rate),
	md.basal_melting_rate=zeros(md.numberofnodes,1);
	disp('      no basal_melting_rate specified: values set as zero');
end
if isnan(md.dhdt),
	md.dhdt=zeros(md.numberofnodes,1);
	disp('      no dhdt specified: values set as zero');
end

md.spcthickness=NaN*ones(md.numberofnodes,1);

if (length(md.observed_temperature)==md.numberofnodes),
	md.spctemperature=NaN*ones(md.numberofnodes,1);
	pos=find(md.nodeonsurface); md.spctemperature(pos)=md.observed_temperature(pos); %impose observed temperature on surface
	if (length(md.geothermalflux)~=md.numberofnodes),
		md.geothermalflux=zeros(md.numberofnodes,1);
		md.geothermalflux(find(md.nodeonicesheet))=50*10^-3; %50mW/m2
	end
else
	disp('      no thermal boundary conditions created: no observed temperature found');
end
