function projected_vector=project3d(md3d,varargin);
%PROJECT3D - vertically project a vector from 2d mesh
%
%   vertically project a vector from 2d mesh (split in noncoll and coll areas) into a 3d mesh.
%   This vector can be a node vector of size (md3d.numberofnodes2d,N/A) or an 
%   element vector of size (md3d.numberofelements2d,N/A). 
%   arguments: 
%      'vector': 2d vector
%      'type': 'element' or 'node'. 
%   options: 
%      'layer' a layer number where vector should keep its values. If not specified, all layers adopt the 
%             value of the 2d vector.
%      'padding': default to 0 (value adopted by other 3d layers not being projected0
%
%   Egs:
%      extruded_vector=project3d(md3d,'vector',vector2d,'type','node','layer',1,'padding',NaN);
%      extruded_vector=project3d(md3d,'vector',vector2d,'type','element','padding',0);
%      extruded_vector=project3d(md3d,'vector',vector2d,'type','node');


if nargin==0,
	Project3dUsage;
	error();
end

options=pairoptions(varargin{:});

%retrieve parameters from options.
vector2d=getfieldvalue(options,'vector'); %mandatory
type=getfieldvalue(options,'type'); %mandatory
layer=getfieldvalue(options,'layer',0); %optional (do all layers otherwise)
paddingvalue=getfieldvalue(options,'padding',0); %0 by default

if strcmpi(type,'node'),

	projected_vector=paddingvalue*ones(md3d.numberofnodes,size(vector2d,2));
	
	if layer==0,
		for i=1:md3d.numlayers,
			projected_vector(((i-1)*md3d.numberofnodes2d+1):(i*md3d.numberofnodes2d),:)=vector2d;
		end
	else
		projected_vector(((layer-1)*md3d.numberofnodes2d+1):(layer*md3d.numberofnodes2d),:)=vector2d;
	end
elseif strcmpi(type,'element'),

	projected_vector=paddingvalue*ones(md3d.numberofelements,size(vector2d,2));

	if layer==0,
		for i=1:(md3d.numlayers-1),
			projected_vector( ((i-1)*md3d.numberofelements2d+1):(i*md3d.numberofelements2d),:)=vector2d;
		end

	else
		projected_vector( ((layer-1)*md3d.numberofelements2d+1):(layer*md3d.numberofelements2d),:)=vector2d;
	end
else
	error('project3d error message: unknown projection type');
end

function Project3dUsage;

disp('function projected_vector=project3d(md3d,varargin);');
disp('PROJECT3D - vertically project a vector from 2d mesh...');
disp('');
disp('   vertically project a vector from 2d mesh (split in noncoll and coll areas) into a 3d mesh.');
disp('   This vector can be a node vector of size (md3d.numberofnodes2d,N/A) or an ');
disp('   element vector of size (md3d.numberofelements2d,N/A). ');
disp('   arguments: ');
disp('      ''vector'': 2d vector');
disp('      ''type'': ''element'' or ''node''. ');
disp('   options: ');
disp('      ''layer'' a layer number where vector should keep its values. If not specified, all layers adopt the ');
disp('             value of the 2d vector.');
disp('      ''padding'': default to 0 (value adopted by other 3d layers not being projected0');
disp('');
disp('   Egs:');
disp('      extruded_vector=project3d(md3d,''vector'',vector2d,''type'',''node'',''layer'',1,''padding'',NaN);');
disp('      extruded_vector=project3d(md3d,''vector'',vector2d,''type'',''element'',''padding'',0);');
disp('      extruded_vector=project3d(md3d,''vector'',vector2d,''type'',''node'');');
