/*!\file Spct.c
 * \brief: implementation of the Spct object
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "stdio.h"
#include <string.h>
#include "../../include/include.h"
#include "../../shared/shared.h"
#include "../../EnumDefinitions/EnumDefinitions.h"
#include "../../Container/Container.h"
#include "../objects.h"

/*Spct constructors and destructor*/
/*FUNCTION Spct::Spct(){{{1*/
Spct::Spct(){
	sid=-1;
	nodeid=-1;
	dof=-1;
	values=NULL;
	times=NULL;
	nsteps=-1;
	analysis_type=-1;
	return;
}
/*}}}1*/
/*FUNCTION Spct::Spct(int spc_sid,int spc_nodeid,...){{{1*/
Spct::Spct(int spc_sid,int spc_nodeid, int spc_dof,int spc_nsteps, double* spc_times, double* spc_values,int spc_analysis_type){

	sid=spc_sid;
	nodeid=spc_nodeid;
	dof=spc_dof;
	nsteps=spc_nsteps;
	if(spc_nsteps){
		values=(double*)xmalloc(spc_nsteps*sizeof(double));
		times=(double*)xmalloc(spc_nsteps*sizeof(double));
		memcpy(values,spc_values,nsteps*sizeof(double));
		memcpy(times,spc_times,nsteps*sizeof(double));
	}
	analysis_type=spc_analysis_type;
	return;
}
/*}}}1*/
/*FUNCTION Spct::~Spct{{{1*/
Spct::~Spct(){
	xfree((void**)&times);
	xfree((void**)&values);
	return;
}
/*}}}1*/
		
/*Object virtual functions definitions:*/
/*FUNCTION Spct::Echo {{{1*/
void Spct::Echo(void){

	int i;
	printf("Spct:\n");
	printf("   sid: %i\n",sid);
	printf("   nodeid: %i\n",nodeid);
	printf("   dof: %i\n",dof);
	printf("   nsteps: %i\n",nsteps);
	printf("   analysis_type: %s\n",EnumToStringx(analysis_type));
	printf("   steps|times|values\n");
	for(i=0;i<nsteps;i++){
		printf("%i-%g:%g\n",i,times[i],values[i]);
	}
	return;
}
/*}}}1*/
/*FUNCTION Spct::DeepEcho {{{1*/
void Spct::DeepEcho(void){
	this->Echo();
}		
/*}}}1*/
/*FUNCTION Spct::Id {{{1*/
int    Spct::Id(void){ return sid; }
/*}}}1*/
/*FUNCTION Spct::MyRank {{{1*/
int    Spct::MyRank(void){ 
	extern int my_rank;
	return my_rank; 
}
/*}}}1*/
/*FUNCTION Spct::Marshall {{{1*/
void  Spct::Marshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;
	int   enum_type=0;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*get enum type of Spct: */
	enum_type=SpctEnum;
	
	/*marshall enum: */
	memcpy(marshalled_dataset,&enum_type,sizeof(enum_type));marshalled_dataset+=sizeof(enum_type);
	
	/*marshall Spct data: */
	memcpy(marshalled_dataset,&sid,sizeof(sid));marshalled_dataset+=sizeof(sid);
	memcpy(marshalled_dataset,&nodeid,sizeof(nodeid));marshalled_dataset+=sizeof(nodeid);
	memcpy(marshalled_dataset,&dof,sizeof(dof));marshalled_dataset+=sizeof(dof);
	memcpy(marshalled_dataset,&nsteps,sizeof(nsteps));marshalled_dataset+=sizeof(nsteps);
	memcpy(marshalled_dataset,&analysis_type,sizeof(analysis_type));marshalled_dataset+=sizeof(analysis_type);
	if(nsteps){
		memcpy(marshalled_dataset,values,nsteps*sizeof(double));marshalled_dataset+=nsteps*sizeof(double);
		memcpy(marshalled_dataset,times,nsteps*sizeof(double));marshalled_dataset+=nsteps*sizeof(double);
	}

	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}1*/
/*FUNCTION Spct::MarshallSize {{{1*/
int   Spct::MarshallSize(){

	return sizeof(sid)
		+sizeof(nodeid)
		+sizeof(dof)
		+sizeof(nsteps)
		+nsteps*2*sizeof(double)
		+sizeof(analysis_type)
		+sizeof(int); //sizeof(int) for enum type
}
/*}}}1*/
/*FUNCTION Spct::Demarshall {{{1*/
void  Spct::Demarshall(char** pmarshalled_dataset){

	char* marshalled_dataset=NULL;

	/*recover marshalled_dataset: */
	marshalled_dataset=*pmarshalled_dataset;

	/*this time, no need to get enum type, the pointer directly points to the beginning of the 
	 *object data (thanks to DataSet::Demarshall):*/

	memcpy(&sid,marshalled_dataset,sizeof(sid));marshalled_dataset+=sizeof(sid);
	memcpy(&nodeid,marshalled_dataset,sizeof(nodeid));marshalled_dataset+=sizeof(nodeid);
	memcpy(&dof,marshalled_dataset,sizeof(dof));marshalled_dataset+=sizeof(dof);
	memcpy(&nsteps,marshalled_dataset,sizeof(nsteps));marshalled_dataset+=sizeof(nsteps);
	memcpy(&analysis_type,marshalled_dataset,sizeof(analysis_type));marshalled_dataset+=sizeof(analysis_type);
	if(nsteps){
		values=(double*)xmalloc(nsteps*sizeof(double));
		times=(double*)xmalloc(nsteps*sizeof(double));
		memcpy(values,marshalled_dataset,nsteps*sizeof(double));marshalled_dataset+=nsteps*sizeof(double);
		memcpy(times,marshalled_dataset,nsteps*sizeof(double));marshalled_dataset+=nsteps*sizeof(double);
	}

	/*return: */
	*pmarshalled_dataset=marshalled_dataset;
	return;
}
/*}}}1*/
/*FUNCTION Spct::Enum {{{1*/
int Spct::Enum(void){

	return SpctEnum;

}
/*}}}1*/
/*FUNCTION Spct::copy {{{1*/
Object* Spct::copy() {
	return new Spct(sid,nodeid,dof,nsteps,times,values,analysis_type);
}
/*}}}1*/

/*Constraint virtual functions definitions:*/
/*FUNCTION Spct::InAnalysis{{{1*/
bool Spct::InAnalysis(int in_analysis_type){
	
	if (in_analysis_type==this->analysis_type) return true;
	else return false;
}
/*}}}*/
/*FUNCTION Spct::ConstrainNode{{{1*/
void Spct::ConstrainNode(Nodes* nodes,Parameters* parameters){

	Node* node=NULL;
	double time=0;
	int    i;
	double alpha=-1;
	double value;
	bool   found=false;

	/*Chase through nodes and find the node to which this Spct applys: */
	node=(Node*)nodes->GetObjectById(NULL,nodeid);
	
	if(node){ //in case the spc is dealing with a node on another cpu

		/*Retrieve time in parameters: */
		parameters->FindParam(&time,TimeEnum);

		/*Now, go fetch value for this time: */
		if (time<=times[0]){
			value=values[0];
			found=true;
		}
		else if (time>=times[nsteps-1]){
			value=values[nsteps-1];
			found=true;
		}
		else{
			for(i=0;i<nsteps-1;i++){
				if (times[i]<=time && time<times[i+1]){
					alpha=(time-times[i])/(times[i+1]-times[i]);
					value=(1-alpha)*values[i]+alpha*values[i+1];
					found=true;
					break;
				}
			}
		}

		if(!found)_error_("could not find time segment for constraint");

		/*Apply or relax constraint: */
		if(isnan(value)){
			node->RelaxConstraint(dof);
		}
		else node->ApplyConstraint(dof,value);
	}
}
/*}}}*/

/*Spct functions*/
/*FUNCTION Spct::GetDof {{{1*/
int Spct::GetDof(){
	return dof;
}
/*}}}1*/
/*FUNCTION Spct::GetNodeId {{{1*/
int   Spct::GetNodeId(){
	
	return nodeid;
}
/*}}}1*/
/*FUNCTION Spct::GetValue {{{1*/
double Spct::GetValue(){
	return values[0];
}
/*}}}1*/

