/*!\file:  BrentSearch.cpp
 * \brief optimization algorithm
 */ 

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "./numerics.h"
#include "../../objects/objects.h"
#include "../../include/macros.h"
#include "../../shared/shared.h"
#include <float.h>

void BrentSearch(double* psearch_scalar,double* pJ,OptPars* optpars,double (*f)(double,OptArgs*), OptArgs* optargs){

	/* This routine is optimizing a given function using Brent's method
	 * (Golden or parabolic procedure)*/

	/*optimization variable: */
	double si;
	double gold;
	double intervalgold;
	double oldintervalgold;
	double parab_num,parab_den;
	double distance;

	/*function values: */
	double fxmax,fxmin,fxbest,fval;
	double fx,fx1,fx2;

	/*x : */
	double xmax,xmin,xbest;
	double x,x1,x2,xm,xval;

	/*tolerances: */
	double tol1,tol2,seps,tolerance;
	int    maxiter;

	/*counters: */
	int iter,goldenflag,loop;

	/*Recover parameters:*/
	xmin=optpars->xmin;
	xmax=optpars->xmax;
	tolerance=optpars->tolerance;
	maxiter=optpars->maxiter;
	
	//initialize counter and boundaries
	iter=0;

	//get the value of the function at the first boundary
	fxmin = (*f)(xmin,optargs);

	//display result
	_printf_("\n        Iteration       x           f(x)       Tolerance         Procedure\n\n");
	_printf_("        %s   %12.6g  %12.6g  %s","   N/A",xmin,fxmin,"         N/A         boundary\n");

	//get the value of the function at the first boundary xmax and display result
	fxmax = (*f)(xmax,optargs);
	_printf_("        %s   %12.6g  %12.6g  %s","   N/A",xmax,fxmax,"         N/A         boundary\n");

	//initialize the other variables
	seps=sqrt(DBL_EPSILON); //precision of a double
	distance=0.0;              //new_x=old_x + distance
	gold=0.5*(3.0-sqrt(5.0));  //gold = 1 - golden ratio
	intervalgold=0.0;          //distance used by Golden procedure

	//Compute initial point
	
	//1: initialize the value of the 4 x needed (x1,x2,x,xbest)
	x1=xmin+gold*(xmax-xmin);
	x2=x1;
	xbest=x1;
	x=xbest;

	//2: call the function to be evaluated
	fxbest = (*f)(x,optargs);
	iter=iter+1;

	//3: update the other variables
	fx1=fxbest;
	fx2=fxbest;
	//xm is always in the middle of a and b
	xm=0.5*(xmin+xmax);                           
	//update tolerances
	tol1=seps*sqrt(pow(xbest,2))+tolerance/3.0;
	tol2=2.0*tol1;

	//4: print result
	_printf_("         %5i    %12.6g  %12.6g  %12.6g  %s\n",iter,xbest,fxbest,pow(pow(xbest-xm,2),0.5),"       initial");

	//Main Loop
	loop=1;
	while(loop){

		goldenflag=1;

		// Is a parabolic fit possible ?
		if (sqrt(pow(intervalgold,2))>tol1){

			// Yes, so fit parabola
			goldenflag=0;
			parab_num=(xbest-x1)*(xbest-x1)*(fxbest-fx2)-(xbest-x2)*(xbest-x2)*(fxbest-fx1);;
			parab_den=2.0*(xbest-x1)*(fxbest-fx2)-2.0*(xbest-x2)*(fxbest-fx1);

			//reverse p if necessary
			if(parab_den>0.0){ 
				parab_num=-parab_num;
			}
			parab_den=sqrt(pow(parab_den,2));
			oldintervalgold=intervalgold;
			intervalgold=distance;

			// Is the parabola acceptable
			if (( sqrt(pow(parab_num,2)) < sqrt(pow(0.5*parab_den*oldintervalgold,2))) &&
						(parab_num>parab_den*(xmin-xbest)) &&
						(parab_num<parab_den*(xmax-xbest))){

				// Yes, parabolic interpolation step
				distance=parab_num/parab_den;
				x=xbest+distance;

				// f must not be evaluated too close to min_x or max_x
				if (((x-xmin)<tol2) || ((xmax-x)<tol2)){

					if ((xm-xbest)<0.0){
						si=-1;
					}
					else{
						si=1;
					}

					//compute new distance
					distance=tol1*si;
				}
			}
			else{

				// Not acceptable, must do a golden section step
				goldenflag=1;
			}
		}

		//Golden procedure
		if(goldenflag){

			// compute the new distance d
			if(xbest>=xm){
				intervalgold=xmin-xbest;    
			}
			else{ 
				intervalgold=xmax-xbest;  
			}
			distance=gold*intervalgold;
		}

		// The function must not be evaluated too close to xbest
		if(distance<0){
			si=-1;
		}
		else{
			si=1;
		}
		if (sqrt(pow(distance,2))>tol1){
			x=xbest+si*sqrt(pow(distance,2));
		}
		else{
			x=xbest+si*tol1;
		}

		//evaluate function on x
		fx = (*f)(x,optargs);
		iter=iter+1;

		// Update a, b, xm, x1, x2, tol1, tol2
		if (fx<=fxbest){
			if (x>=xbest){
				xmin=xbest;
			}
			else{
				xmax=xbest;
			}
			x1=x2;    fx1=fx2;
			x2=xbest; fx2=fxbest;
			xbest=x;  fxbest=fx;
		}

		else{ // fx > fxbest
			if (x < xbest){
				xmin=x;
			}
			else{
				xmax=x;
			}
			if ((fx<=fx2) || (x2==xbest)){
				x1=x2; fx1=fx2;
				x2=x;  fx2=fx;
			}
			else if ( (fx <= fx1) || (x1 == xbest) || (x1 == x2) ){
				x1=x;  fx1=fx;
			}
		}
		xm = 0.5*(xmin+xmax);
		tol1=seps*pow(pow(xbest,2),0.5)+tolerance/3.0;
		tol2=2.0*tol1;

		//print result
		if (goldenflag){
			_printf_("         %5i    %12.6g  %12.6g  %12.6g  %s\n",iter,x,fx,pow(pow(xbest-xm,2),0.5),"       golden");
		}
		else{
			_printf_("         %5i    %12.6g  %12.6g  %12.6g  %s\n",iter,x,fx,pow(pow(xbest-xm,2),0.5),"       parabolic");
		}

		//Stop the optimization?
		if (sqrt(pow(xbest-xm,2)) < (tol2-0.5*(xmax-xmin))){
			_printf_("\nOptimization terminated:\nthe current x satisfies the termination criteria using 'tolx' of %g \n", tolerance);
			loop=0;
		}
		else if (iter>=maxiter){
			_printf_("\nExiting: Maximum number of iterations has been exceeded  - increase 'maxiter'\n");
			loop=0;
		}
		else{
			//continue
			loop=1;
		}
	}//end while

	//Now, check that the value on the boundaries are not better than current fxbest
	if (fxbest>fxmin){
		xval=xmin;
		fval=fxmin;
	}
	else if (fxbest>fxmax){
		xval=xmax;
		fval=fxmax;
	}
	else{
		xval=xbest;
		fval=fxbest;
	}

	/*Assign output pointers: */
	*psearch_scalar=xval;
	*pJ=fval;
	
}
