/*! \file IoModel.cpp
 * \brief  file containing the methods that will help in processing the input data coming 
 * into ISSM, from Matlab, or through a binary file opened for reading.
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include <string.h>
#include "stdio.h"
#include "./objects.h"
#include "../shared/shared.h"
#include "../io/io.h"
#include "../include/include.h"

/*FUNCTION IoModel::IoModel(){{{1*/
IoModel::IoModel(){
	this->IoModelInit();
}
/*}}}*/
/*FUNCTION IoModel::~IoModel(){{{1*/
IoModel::~IoModel(){
	
	int i;

	/*!Two cases here: 
	 * - serial mode: matlab's memory manager will take care of delete iomodel when returning from Imp. Do nothing here, so as not to confuse 
	 *                the memory manager.
     * - in parallel, anything the io layer does (FetchData) did needs to be erased explicitely in the iomodel.
	 */

	#ifdef _PARALLEL_
	xfree((void**)&this->x);
	xfree((void**)&this->y);
	xfree((void**)&this->z);
	xfree((void**)&this->elements);
	xfree((void**)&this->elementconnectivity);
	xfree((void**)&this->elements_type);
	xfree((void**)&this->vertices_type);
	xfree((void**)&this->nodeonhutter);
	xfree((void**)&this->nodeonmacayeal);
	if (this->dim==3){
		xfree((void**)&this->elements2d);
		xfree((void**)&this->upperelements);
		xfree((void**)&this->lowerelements);
		xfree((void**)&this->nodeonpattyn);
	}
	xfree((void**)&this->elementonbed);
	xfree((void**)&this->elementonsurface);
	xfree((void**)&this->nodeonbed);
	xfree((void**)&this->nodeonsurface);
	xfree((void**)&this->nodeonstokes);
	xfree((void**)&this->borderstokes);
	xfree((void**)&this->thickness_obs);
	xfree((void**)&this->thickness_coeff);
	xfree((void**)&this->thickness);
	xfree((void**)&this->surface);
	xfree((void**)&this->bed);
	xfree((void**)&this->bathymetry);
	xfree((void**)&this->vx_obs);
	xfree((void**)&this->vy_obs);
	xfree((void**)&this->vz_obs);
	xfree((void**)&this->vx);
	xfree((void**)&this->vy);
	xfree((void**)&this->vz);
	xfree((void**)&this->pressure);
	xfree((void**)&this->temperature);
	xfree((void**)&this->waterfraction);
	xfree((void**)&this->drag_coefficient);
	xfree((void**)&this->drag_p);
	xfree((void**)&this->drag_q);
	xfree((void**)&this->elementoniceshelf);
	xfree((void**)&this->elementonwater);
	xfree((void**)&this->nodeonicesheet);
	xfree((void**)&this->nodeoniceshelf);
	xfree((void**)&this->nodeonwater);
	xfree((void**)&this->pressureload);
	xfree((void**)&this->spcvx);
	xfree((void**)&this->spcvy);
	xfree((void**)&this->spcvz);
	xfree((void**)&this->spcthickness);
	xfree((void**)&this->spcwatercolumn);
	xfree((void**)&this->spctemperature);
	xfree((void**)&this->diagnostic_ref);
	xfree((void**)&this->edges);
	xfree((void**)&this->geothermalflux);
	xfree((void**)&this->basal_melting_rate);
	xfree((void**)&this->watercolumn);
	xfree((void**)&this->basal_melting_rate_correction);
	xfree((void**)&this->surface_accumulation_rate);
	xfree((void**)&this->surface_ablation_rate);
	xfree((void**)&this->surface_mass_balance);
	xfree((void**)&this->dhdt);
	xfree((void**)&this->rheology_B);
	xfree((void**)&this->rheology_n);
	xfree((void**)&this->control_type);
	xfree((void**)&this->cm_responses);
	xfree((void**)&this->weights);
	xfree((void**)&this->cm_jump);
	xfree((void**)&this->cm_min);
	xfree((void**)&this->cm_max);
	xfree((void**)&this->optscal);
	xfree((void**)&this->maxiter);


	/*!Delete structure fields: */
	xfree((void**)&this->inputfilename);
	xfree((void**)&this->outputfilename);
	xfree((void**)&this->repository);
	xfree((void**)&this->name);
	xfree((void**)&this->riftinfo);
	xfree((void**)&this->penalties);
	
	/*exterior data: */
	xfree((void**)&this->my_elements);
	xfree((void**)&this->my_vertices);
	xfree((void**)&this->my_nodes);
	xfree((void**)&this->singlenodetoelementconnectivity);
	xfree((void**)&this->numbernodetoelementconnectivity);
	#endif
}
/*}}}*/
/*FUNCTION IoModel::IoModel(ConstDataHandle iomodel_handle){{{1*/
IoModel::IoModel(FILE* iomodel_handle){
	
	int i,j;
		
	/*First, initialize the structure: */
	this->IoModelInit();
	
	/*Get all the data that consists of scalars, integers and strings: */

	IoModelFetchData(&this->name,iomodel_handle,NameEnum); 
	IoModelFetchData(&this->inputfilename,iomodel_handle,InputfilenameEnum); 
	IoModelFetchData(&this->outputfilename,iomodel_handle,OutputfilenameEnum); 
	IoModelFetchData(&this->qmu_analysis,iomodel_handle,QmuAnalysisEnum); 
	IoModelFetchData(&this->control_analysis,iomodel_handle,ControlAnalysisEnum); 
	IoModelFetchData(&this->dim,iomodel_handle,DimEnum);
	/*!Get numberofelements and numberofvertices: */
	IoModelFetchData(&this->numberofvertices,iomodel_handle,NumberOfNodesEnum);
	IoModelFetchData(&this->numberofelements,iomodel_handle,NumberOfElementsEnum);
	/*!In case we are running 3d, we are going to need the collapsed and non-collapsed 2d meshes, from which the 3d mesh was extruded: */
	if (this->dim==3){
	
		/*!Deal with 2d mesh: */
		IoModelFetchData(&this->numberofelements2d,iomodel_handle,NumberOfElements2DEnum);
		IoModelFetchData(&this->numberofvertices2d,iomodel_handle,NumberOfNodes2DEnum);
		IoModelFetchData(&this->numlayers,iomodel_handle,NumlayersEnum);
	}


	/*elements type: */
	IoModelFetchData(&this->ishutter,iomodel_handle,IshutterEnum);
	IoModelFetchData(&this->ismacayealpattyn,iomodel_handle,IsmacayealpattynEnum);
	IoModelFetchData(&this->isstokes,iomodel_handle,IsstokesEnum);

	/*!Get drag_type, drag and p,q: */
	IoModelFetchData(&this->drag_type,iomodel_handle,DragTypeEnum);

	/*!Get materials: */
	IoModelFetchData(&this->rho_water,iomodel_handle,RhoWaterEnum);
	IoModelFetchData(&this->rho_ice,iomodel_handle,RhoIceEnum);
	IoModelFetchData(&this->g,iomodel_handle,GEnum);

	/*Get control parameters: */
	IoModelFetchData(&this->num_control_type,iomodel_handle,NumControlTypeEnum); 
	IoModelFetchData(&this->num_cm_responses,iomodel_handle,NumCmResponsesEnum); 

	/*!Get solution parameters: */
	IoModelFetchData(&this->yts,iomodel_handle,YtsEnum);
	IoModelFetchData(&this->meanvel,iomodel_handle,MeanvelEnum);
	IoModelFetchData(&this->epsvel,iomodel_handle,EpsvelEnum);
	IoModelFetchData(&this->verbose,iomodel_handle,VerboseBinaryEnum);
	IoModelFetchData(&this->output_frequency,iomodel_handle,OutputFrequencyEnum);
	IoModelFetchData(&this->prognostic_DG,iomodel_handle,PrognosticDGEnum);
	IoModelFetchData(&this->nsteps,iomodel_handle,NstepsEnum);
	IoModelFetchData(&this->eps_cm,iomodel_handle,EpsCmEnum);
	IoModelFetchData(&this->tolx,iomodel_handle,TolxEnum);
	IoModelFetchData(&this->cm_gradient,iomodel_handle,CmGradientEnum);
	IoModelFetchData(&this->eps_res,iomodel_handle,EpsResEnum);
	IoModelFetchData(&this->eps_rel,iomodel_handle,EpsRelEnum);
	IoModelFetchData(&this->eps_abs,iomodel_handle,EpsAbsEnum);
	IoModelFetchData(&this->max_nonlinear_iterations,iomodel_handle,MaxNonlinearIterationsEnum);
	IoModelFetchData(&this->dt,iomodel_handle,DtEnum);
	IoModelFetchData(&this->ndt,iomodel_handle,NdtEnum);
	IoModelFetchData(&this->time_adapt,iomodel_handle,TimeAdaptEnum);
	IoModelFetchData(&this->cfl_coefficient,iomodel_handle,CflCoefficientEnum);
	IoModelFetchData(&this->hydrostatic_adjustment,iomodel_handle,HydrostaticAdjustmentEnum);
	IoModelFetchData(&this->penalty_offset,iomodel_handle,PenaltyOffsetEnum);
	IoModelFetchData(&this->penalty_melting,iomodel_handle,PenaltyMeltingEnum);
	IoModelFetchData(&this->penalty_lock,iomodel_handle,PenaltyLockEnum);
	IoModelFetchData(&this->sparsity,iomodel_handle,SparsityEnum);
	IoModelFetchData(&this->connectivity,iomodel_handle,ConnectivityEnum);
	IoModelFetchData(&this->lowmem,iomodel_handle,LowmemEnum);
	IoModelFetchData(&this->viscosity_overshoot,iomodel_handle,ViscosityOvershootEnum);
	IoModelFetchData(&this->artdiff,iomodel_handle,ArtificialDiffusivityEnum);
	IoModelFetchData(&this->prognostic_DG,iomodel_handle,PrognosticDGEnum);
	IoModelFetchData(&this->stokesreconditioning,iomodel_handle,StokesreconditioningEnum);
	IoModelFetchData(&this->shelf_dampening,iomodel_handle,ShelfDampeningEnum);
	IoModelFetchData(&this->waitonlock,iomodel_handle,WaitonlockEnum);
	IoModelFetchData(&this->gl_migration,iomodel_handle,GlMigrationEnum);
	IoModelFetchData(&this->isdiagnostic,iomodel_handle,IsdiagnosticEnum); 
	IoModelFetchData(&this->isprognostic,iomodel_handle,IsprognosticEnum); 
	IoModelFetchData(&this->isthermal,iomodel_handle,IsthermalEnum); 

	/*!Get thermal parameters: */
	IoModelFetchData(&this->beta,iomodel_handle,BetaEnum);
	IoModelFetchData(&this->meltingpoint,iomodel_handle,MeltingpointEnum);
	IoModelFetchData(&this->referencetemperature,iomodel_handle,ReferencetemperatureEnum);
	IoModelFetchData(&this->latentheat,iomodel_handle,LatentheatEnum);
	IoModelFetchData(&this->heatcapacity,iomodel_handle,HeatcapacityEnum);
	IoModelFetchData(&this->thermalconductivity,iomodel_handle,ThermalconductivityEnum);
	IoModelFetchData(&this->min_thermal_constraints,iomodel_handle,MinThermalConstraintsEnum);
	IoModelFetchData(&this->min_mechanical_constraints,iomodel_handle,MinMechanicalConstraintsEnum);
	IoModelFetchData(&this->stabilize_constraints,iomodel_handle,StabilizeConstraintsEnum);
	IoModelFetchData(&this->mixed_layer_capacity,iomodel_handle,MixedLayerCapacityEnum);
	IoModelFetchData(&this->thermal_exchange_velocity,iomodel_handle,ThermalExchangeVelocityEnum);
	IoModelFetchData(&this->basal_melting_rate_correction_apply,iomodel_handle,BasalMeltingRateCorrectionApplyEnum);
	IoModelFetchData(&this->gl_melting_rate,iomodel_handle,GlMeltingRateEnum);
	IoModelFetchData(&this->rheology_law,iomodel_handle,RheologyLawEnum);
	
	/*!Get hydrology parameters: */
	IoModelFetchData(&this->hydro_kn,iomodel_handle,HydroKnEnum);
	IoModelFetchData(&this->hydro_p,iomodel_handle,HydroPEnum);
	IoModelFetchData(&this->hydro_q,iomodel_handle,HydroQEnum);
	IoModelFetchData(&this->hydro_CR,iomodel_handle,HydroCREnum);
	IoModelFetchData(&this->hydro_n,iomodel_handle,HydroNEnum);

	/*qmu: */
	if(this->qmu_analysis){
		IoModelFetchData(&this->numberofvariables,iomodel_handle,NumberOfVariablesEnum);
		IoModelFetchData(&this->numberofresponses,iomodel_handle,NumberOfResponsesEnum);
		IoModelFetchData(&this->qmu_npart,iomodel_handle,NpartEnum);
		IoModelFetchData(&this->qmu_save_femmodel,iomodel_handle,QmuSaveFemmodelEnum); 
	}

	/*i/o: */
	IoModelFetchData(&this->io_gather,iomodel_handle,IoGatherEnum);
}
/*}}}*/
/*FUNCTION IoModel::IoModelInit{{{1*/
void IoModel::IoModelInit(void){
	
	/*!initialize all pointers to 0: */
	this->name=NULL;
	this->inputfilename=NULL;
	this->outputfilename=NULL;
	this->repository=NULL;
	this->qmu_analysis=0;
	this->control_analysis=0;
	this->numberofvariables=0;
	this->numvariabledescriptors=0;
	this->numberofresponses=0;
	this->numresponsedescriptors=0;
	this->qmu_npart=0; 
	this->numberofelements=0;
	this->numberofvertices=0;
	this->x=NULL; 
	this->y=NULL;
	this->z=NULL;
	this->elements=NULL;
	this->elementconnectivity=NULL;
	this->elements_type=NULL;
	this->vertices_type=NULL;
	this->numberofvertices2d=0;
	this->elements2d=NULL;
	this->numlayers=0;
	this->upperelements=NULL;
	this->lowerelements=NULL;
	this->nodeonhutter=NULL;
	this->nodeonmacayeal=NULL;
	this->nodeonpattyn=NULL;
	this->io_gather=1;
	
	this->vx_obs=NULL;
	this->vy_obs=NULL;
	this->vz_obs=NULL;
	this->vx=NULL;
	this->vy=NULL;
	this->vz=NULL;
	this->pressure=NULL;
	this->temperature=NULL;
	this->waterfraction=NULL;
	this->gl_melting_rate=0;
	this->basal_melting_rate=NULL;
	this->watercolumn=NULL;
	this->basal_melting_rate_correction=NULL;
	this->basal_melting_rate_correction_apply=0;
	this->geothermalflux=NULL;
	this->elementonbed=NULL;
	this->elementonsurface=NULL;
	this->nodeonbed=NULL;
	this->nodeonsurface=NULL;
	this->nodeonstokes=NULL;
	this->borderstokes=NULL;
	this->thickness_obs=NULL;
	this->thickness_coeff=NULL;
	this->thickness=NULL;
	this->surface=NULL;
	this->bed=NULL;
	this->bathymetry=NULL;
	this->elementoniceshelf=NULL;
	this->elementonwater=NULL;
	this->nodeonicesheet=NULL;
	this->nodeoniceshelf=NULL;
	this->nodeonwater=NULL;

	this->drag_type=0;
	this->drag_coefficient=NULL;
	this->drag_p=NULL;
	this->drag_q=NULL;
	
	
	this->numberofpressureloads=0;
	this->pressureload=NULL;
	this-> spcvx=NULL;
	this-> spcvy=NULL;
	this-> spcvz=NULL;
	this-> spctemperature=NULL;
	this-> spcthickness=NULL;
	this-> spcwatercolumn=NULL;
	this-> diagnostic_ref=NULL;
	this->numberofedges=0;
	this->edges=NULL;
	
	/*!materials: */
	this->rho_water=0;
	this->rho_ice=0;
	this->g=0;
	this->rheology_n=NULL;
	this->rheology_B=NULL;
	this->rheology_law=0;

	/*!solution parameters: */
	this->control_type=NULL;
	this->cm_responses=NULL;
	this->weights=NULL;
	this->cm_jump=NULL;
	this->meanvel=0;
	this->epsvel=0;
	this->nsteps=0;
	this->eps_cm=0;
	this->tolx=0;
	this->maxiter=NULL;
	this->cm_min=NULL;
	this->cm_max=NULL;
	this->cm_gradient=0;
	this->verbose=0;
	this->output_frequency=0;
	this->eps_res=0;
	this->eps_rel=0;
	this->eps_abs=0;
	this->max_nonlinear_iterations=0;
	this->dt=0;
	this->ndt=0;
	this->time_adapt=0;
	this->cfl_coefficient=0;
	this->hydrostatic_adjustment=0;
	this->gl_migration=0;
	this->penalty_offset=0;
	this->penalty_melting=0;
	this->penalty_lock=0;
	this->sparsity=0;
	this->connectivity=0;
	this->lowmem=0;
	this->optscal=NULL;
	this->yts=0;
	this->viscosity_overshoot=0;
	this->artdiff=0;
	this->prognostic_DG=0;
	this->stokesreconditioning=0;
	this->shelf_dampening=0;
	this->waitonlock=0;
	this->isdiagnostic=0;
	this->isprognostic=0;
	this->isthermal=0;

	/*!thermal parameters: */
	this->beta=0;
	this->meltingpoint=0;
	this->referencetemperature=0;
	this->latentheat=0;
	this->heatcapacity=0;
	this->thermalconductivity=0;
	this->min_thermal_constraints=0;
	this->min_mechanical_constraints=0;
	this->stabilize_constraints=0;
	this->mixed_layer_capacity=0;
	this->thermal_exchange_velocity=0;

	
	this->numrifts=0;
	this->riftinfo=NULL;

	/*!penalties: */
	this->numpenalties=0;
	this->penalties=NULL;

	/*!surface: */
	this->surface_mass_balance=NULL;
   this->surface_accumulation_rate=NULL;
	this->surface_ablation_rate=NULL;
	this->dhdt=NULL;

	/*elements type: */
	this->ishutter=0;
	this->ismacayealpattyn=0;
	this->isstokes=0;

	/*exterior data: */
	this->my_elements=NULL;
	this->my_vertices=NULL;
	this->my_nodes=NULL;
	this->singlenodetoelementconnectivity=NULL;
	this->numbernodetoelementconnectivity=NULL;
	this->nodecounter=0;
	this->loadcounter=0;
	this->constraintcounter=0;
}
/*}}}*/
/*FUNCTION IoModel::Echo{{{1*/
void IoModel::Echo(int which_part,int rank) {

	//which_part  determines what gets echoed, otherwise, we'll get too much output.
	//1-> penalties

	int i,j;

	if(which_part==1 && my_rank==rank && this->dim==3){
		printf("IoModel penalties: \n");
		printf("   number of penalties: %i\n",this->numpenalties);
		printf("   nodes: \n");

		for(i=0;i<this->numpenalties;i++){
			for(j=0;j<this->numlayers;j++){
				printf("%i ",(int)*(this->penalties+this->numlayers*i+j));
			}
			printf("\n");
		}
	}
}
/*}}}*/
