/*! \file IoModelFetchData.c
 *  \brief: wrapper to the I/O routines, for special processing of the IoModel structure.
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../../shared/shared.h"
#include "../../include/include.h"
#include "./diskio.h"

/*FUNCTION IoModelFetchData(double** pmatrix,int* pM,int* pN,FILE* model_handle,char* data_name){{{1*/
void  IoModelFetchData(double** pmatrix,int* pM,int* pN,FILE* model_handle,char* data_name){

	extern int my_rank;
	extern int num_procs;

	/*output: */
	int M,N;
	double* matrix=NULL;
	
	FILE* fid=NULL;
	
	/*Set file pointer to beginning of the data: */
	fid=SetFilePointerToData(model_handle,data_name);
	
	/*Now fetch: */

	/*We have to read a matrix from disk. First read the dimensions of the matrix, then the whole matrix: */
	/*numberofelements: */
	if(my_rank==0){  
		if(fread(&M,sizeof(int),1,fid)!=1) _error_("could not read number of rows for matrix ");
	}

	MPI_Bcast(&M,1,MPI_INT,0,MPI_COMM_WORLD); 

	if(my_rank==0){  
		if(fread(&N,sizeof(int),1,fid)!=1) _error_("could not read number of columns for matrix ");
	}
	MPI_Bcast(&N,1,MPI_INT,0,MPI_COMM_WORLD); 

	/*Now allocate matrix: */
	if(M*N){
		matrix=(double*)xmalloc(M*N*sizeof(double));

		/*Read matrix on node 0, then broadcast: */
		if(my_rank==0){  
			if(fread(matrix,M*N*sizeof(double),1,fid)!=1) _error_("could not read matrix ");
		}
		
		MPI_Bcast(matrix,M*N,MPI_DOUBLE,0,MPI_COMM_WORLD); 
	}

	/*Assign output pointers: */
	*pmatrix=matrix;
	if (pM)*pM=M;
	if (pN)*pN=N;

}
/*}}}*/
/*FUNCTION IoModelFetchData(char** pstring,FILE* model_handle,char* data_name){{{1*/
void  IoModelFetchData(char** pstring,FILE* model_handle,char* data_name){

	extern int my_rank;
	extern int num_procs;
	FILE* fid=NULL;

	/*output: */
	char* string=NULL;
	int   string_size;
	
	/*Set file pointer to beginning of the data: */
	fid=SetFilePointerToData(model_handle,data_name);
	
	/*Now fetch: */
	
	/*We have to read a string from disk. First read the dimensions of the string, then the string: */
	if(my_rank==0){  
		if(fread(&string_size,sizeof(int),1,fid)!=1) _error_(" could not read length of string ");
	}

	MPI_Bcast(&string_size,1,MPI_INT,0,MPI_COMM_WORLD); 

	/*Now allocate string: */
	if(string_size){
		string=(char*)xmalloc((string_size+1)*sizeof(char));
		string[string_size]='\0';

		/*Read string on node 0, then broadcast: */
		if(my_rank==0){  
			if(fread(string,string_size*sizeof(char),1,fid)!=1)_error_("  could not read string ");
		}
		MPI_Bcast(string,string_size,MPI_CHAR,0,MPI_COMM_WORLD); 
	}
	else{
		string=(char*)xmalloc(sizeof(char));
		string[0]='\0';
	}


	/*Assign output pointers: */
	*pstring=string;
}
/*}}}*/
/*FUNCTION IoModelFetchData(double* pscalar,FILE* model_handle,char* data_name){{{1*/
void  IoModelFetchData(double* pscalar,FILE* model_handle,char* data_name){


	extern int my_rank;
	extern int num_procs;
	FILE* fid=NULL;

	/*output: */
	double   scalar;
	
	/*Set file pointer to beginning of the data: */
	fid=SetFilePointerToData(model_handle,data_name);
	
	/*We have to read a scalar from disk. First read the dimensions of the scalar, then the scalar: */
	if(my_rank==0){
		if(fread(&scalar,sizeof(double),1,fid)!=1)_error_(" could not read scalar ");
	}
	MPI_Bcast(&scalar,1,MPI_DOUBLE,0,MPI_COMM_WORLD); 

	/*Assign output pointers: */
	*pscalar=scalar;
		 
}
/*}}}*/
/*FUNCTION IoModelFetchData(int* pinteger,FILE* model_handle,char* data_name){{{1*/
void  IoModelFetchData(int* pinteger,FILE* model_handle,char* data_name){

	extern int my_rank;
	extern int num_procs;
	FILE* fid=NULL;

	/*output: */
	int   integer;
	
	/*Set file pointer to beginning of the data: */
	fid=SetFilePointerToData(model_handle,data_name);
	
	/*We have to read a integer from disk. First read the dimensions of the integer, then the integer: */
	if(my_rank==0){  
		if(fread(&integer,sizeof(int),1,fid)!=1) _error_(" could not read integer ");
	}

	MPI_Bcast(&integer,1,MPI_INT,0,MPI_COMM_WORLD); 

	/*Assign output pointers: */
	*pinteger=integer;

}
/*}}}*/
/*FUNCTION SetFilePointerToData(FILE* model_handle,char* data_name){{{1*/
FILE* SetFilePointerToData(FILE* model_handle,char* data_name){

	extern int my_rank;
	extern int num_procs;
	
	int string_size;
	int record_length;
	char* string=NULL;
	char* repository=NULL;
	FILE* fid=NULL;
	int found=0;

	/*Go find in the binary file, the position of the data we want to fetch: */
	if(my_rank==0){
	
		/*First set FILE* position to the beginning of the file: */
		fid=(FILE*)model_handle;
		fseek(fid,0,SEEK_SET);

		/*Now march through file looking for the correct name: */
		for(;;){
			/*Read size of first string name: */
			if(fread(&string_size,sizeof(int),1,fid)==0){
				/*Ok, we have reached the end of the file. break: */
				found=0;
				break;
			}
			/*Allocate string of correct size: */
			string=(char*)xmalloc((string_size+1)*sizeof(char));
			string[string_size]='\0';

			/*Read string: */
			if(fread(string,string_size*sizeof(char),1,fid)==0){
				/*Ok, we have reached the end of the file. break: */
				found=0;
				break;
			}
			/*Is this the correct string? : */
			if (strcmp(string,data_name)==0){
				/*Ok, we have found the correct string. Pass the record length, and break: */
				fseek(fid,sizeof(int),SEEK_CUR);
				found=1;
				break;
			}
			else{
				/*This is not the correct string, read the record length, and skip it: */
				fread(&record_length,sizeof(int),1,fid);
				/*skip: */
				fseek(fid,record_length,SEEK_CUR);
			}
			/*Erase string: */
			xfree((void**)&string);
		}
		/*erase string, if we broke from the for() loop:*/
		xfree((void**)&string);
	}
	MPI_Bcast(&found,1,MPI_INT,0,MPI_COMM_WORLD); 

	if(!found)_error_("%s %s ","could not find data with name",data_name);

	return fid;
}
/*}}}*/
