/*!\file:  control.cpp
 * \brief: control solution
 */ 

#include "../issm.h"
#include "./parallel.h"

#undef __FUNCT__ 
#define __FUNCT__ "control"

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

int main(int argc,char* *argv){

	int n;
	int i;

	/*I/O: */
	FILE* fid=NULL;
	char* inputfilename=NULL;
	char* outputfilename=NULL;
	char* lockname=NULL;
	int   analysis_type;
	int   sub_analysis_type;

	/*Intermediary: */
	FemModel femmodel;
	Vec u_g=NULL;
	ParameterInputs* inputs=NULL;
	int waitonlock=0;
	double search_scalar;
	char* control_type=NULL;
	double* fit=NULL;
	double* optscal=NULL;
	double* u_g_obs=NULL;
	double* u_g_initial=NULL;
	int  gsize;
	int  numberofnodes;
	double*    maxiter=NULL;
	double  tolx;
	double*   param_g=NULL;
	Vec       grad_g=NULL;
	Vec       new_grad_g=NULL;
	Vec       grad_g_old=NULL;
	double*   grad_g_double=NULL;
	double    mincontrolconstraint;
	double    maxcontrolconstraint;
	int       nsteps;
	double*   J=NULL;
	OptArgs   optargs;
	OptPars   optpars;
	Param*    param=NULL;


	MODULEBOOT();

	#if !defined(_PARALLEL_) || (defined(_PARALLEL_) && !defined(_HAVE_PETSC_))
	throw ErrorException(__FUNCT__," parallel executable was compiled without support of parallel libraries!");
	#endif

	PetscInitialize(&argc,&argv,(char *)0,"");  

	/*Size and rank: */
	MPI_Comm_rank(MPI_COMM_WORLD,&my_rank);  
	MPI_Comm_size(MPI_COMM_WORLD,&num_procs); 

	_printf_("recover , input file name and output file name:\n");
	inputfilename=argv[2];
	outputfilename=argv[3];
	lockname=argv[4];

	/*Open handle to data on disk: */
	fid=pfopen(inputfilename,"rb");
	
	_printf_("read and create finite element model:\n");
	CreateFemModel(&femmodel,fid,"control","");

	/*Recover parameters used throughout the solution:*/
	femmodel.parameters->FindParam((void*)&nsteps,"nsteps");
	femmodel.parameters->FindParam((void*)&control_type,"control_type");
	femmodel.parameters->FindParam((void*)&fit,"fit");
	femmodel.parameters->FindParam((void*)&optscal,"optscal");
	femmodel.parameters->FindParam((void*)&maxiter,"maxiter");
	femmodel.parameters->FindParam((void*)&tolx,"tolx");
	femmodel.parameters->FindParam((void*)&waitonlock,"waitonlock");
	femmodel.parameters->FindParam((void*)&mincontrolconstraint,"mincontrolconstraint");
	femmodel.parameters->FindParam((void*)&maxcontrolconstraint,"maxcontrolconstraint");
	femmodel.parameters->FindParam((void*)&param_g,"param_g");
	femmodel.parameters->FindParam((void*)&u_g_obs,"u_g_obs");
	femmodel.parameters->FindParam((void*)&analysis_type,"analysis_type");
	femmodel.parameters->FindParam((void*)&sub_analysis_type,"sub_analysis_type");
	gsize=femmodel.nodes->NumberOfDofs();

	/*Initialize misfit: */
	J=(double*)xmalloc(nsteps*sizeof(double));

	/*Initialize inputs:*/
	femmodel.parameters->FindParam((void*)&u_g_initial,"u_g");
	femmodel.parameters->FindParam((void*)&numberofnodes,"numberofnodes");

	inputs=new ParameterInputs;
	inputs->Add("velocity",u_g_initial,3,numberofnodes);
	
	/*erase useless parameters: */
	param=(Param*)femmodel.parameters->FindParamObject("param_g");
	femmodel.parameters->DeleteObject((Object*)param);

	param=(Param*)femmodel.parameters->FindParamObject("u_g_obs");
	femmodel.parameters->DeleteObject((Object*)param);

	param=(Param*)femmodel.parameters->FindParamObject("u_g");
	femmodel.parameters->DeleteObject((Object*)param);


	/*Start looping: */
	for(n=0;n<nsteps;n++){
			
		_printf_("\n%s%i%s%i\n","   control method step ",n+1,"/",nsteps);
		inputs->Add(control_type,param_g,2,numberofnodes);
		inputs->Add("fit",fit[n]);

		/*Update parameters: */
		UpdateFromInputsx(femmodel.elements,femmodel.nodes,femmodel.loads, femmodel.materials,inputs);

		_printf_("%s\n","      computing gradJ...");
		grad_g=GradJCompute(inputs,&femmodel,u_g_obs);
		_printf_("%s\n","      done.");
			
		_printf_("%s\n","      normalizing directions...");
		Orthx(&new_grad_g,grad_g,grad_g_old);
		VecFree(&grad_g); grad_g=new_grad_g;
		VecFree(&grad_g_old); grad_g_old=grad_g;
		VecToMPISerial(&grad_g_double,grad_g);
		_printf_("%s\n","      done.");
	
		_printf_("%s\n","      optimizing along gradient direction...");
		optargs.femmodel=&femmodel; optargs.param_g=param_g; optargs.u_g_obs=u_g_obs; optargs.grad_g=grad_g_double; optargs.inputs=inputs;optargs.n=n;
		optpars.xmin=0; optpars.xmax=1; optpars.tolerance=tolx; optpars.maxiter=(int)maxiter[n];
		BrentSearch(&search_scalar,J+n,&optpars,&objectivefunctionC,&optargs);
		_printf_("%s\n","      done.");
	
		_printf_("%s\n","      updating parameter using optimized search scalar...");
		for(i=0;i<gsize;i++)param_g[i]=param_g[i]+search_scalar*optscal[n]*grad_g_double[i];
		_printf_("%s\n","      done.");

		_printf_("%s\n","      constraining the new distribution...");    
		ControlConstrainx( param_g,gsize,mincontrolconstraint,maxcontrolconstraint,control_type);
		_printf_("%s\n","      done.");

		//some temporary saving 
		if (((n+1)%5)==0){
			_printf_("%s\n","      saving temporary results...");
			inputs->Add(control_type,param_g,2,numberofnodes);
			inputs->Add("fit",fit[n]);
			UpdateFromInputsx(femmodel.elements,femmodel.nodes,femmodel.loads, femmodel.materials,inputs);
			diagnostic_core_nonlinear(&u_g,NULL,NULL,&femmodel,inputs,analysis_type,sub_analysis_type);
			OutputControl(u_g,param_g,J,nsteps,&femmodel,outputfilename);
			_printf_("%s\n","      done.");
		}

		_printf_("%s%i%s%g\n","      value of misfit J after optimization #",n,": ",J[n]);
		
		/*some freeing:*/
		xfree((void**)&grad_g_double);
	}

	/*Write results to disk: */
	_printf_("%s\n","      preparing final velocity solution...");
	inputs->Add(control_type,param_g,2,numberofnodes);
	inputs->Add("fit",fit[n]);

	/*Update parameters: */
	UpdateFromInputsx(femmodel.elements,femmodel.nodes,femmodel.loads, femmodel.materials,inputs);
	
	diagnostic_core_nonlinear(&u_g,NULL,NULL,&femmodel,inputs,analysis_type,sub_analysis_type);
	
	_printf_("%s\n","      saving final results...");
	OutputControl(u_g,param_g,J,nsteps,&femmodel,outputfilename);
	_printf_("%s\n","      done.");

	/*Write lock file if requested: */
	if (waitonlock){
		WriteLockFile(lockname);
	}
			
	_printf_("closing MPI and Petsc\n");
	PetscFinalize(); 

	/*end module: */
	MODULEEND();
	
	return 0; //unix success return;
}
